# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.core.conf import settings
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    FamilyIsNotAMember,
)
from passport.backend.core.models.account import (
    ACCOUNT_DISABLED_ON_DELETION,
    AccountDeletionOperation,
)
from passport.backend.core.runner.context_managers import UPDATE


log = logging.getLogger(__name__)


class BundleChildMixin(object):

    def check_child_consistent(self, account):
        """
        Проверяет, что данный объект Account является ребенком из действующей
        семьи.

        Входные параметры

            account
                Аккаунт с family_info

            self.family_info
                Объект FamilyInfo

        Исключения

            AccountInvalidTypeError
                Аккаунт не ребенок

            FamilyIsNotAMember
                Аккаунт не входит в действующую семью
        """
        if account.uid not in self.family_info.members:
            raise FamilyIsNotAMember()

        if not (
            account.family_info and
            account.family_info.family_id == self.family_info.family_id
        ):
            raise FamilyIsNotAMember()

        if not account.is_child:
            raise AccountInvalidTypeError()

    def start_child_deletion(self):
        """
        Создаёт операцию удаления ребенка, его аккаунт должен быть в self.account подгружен вместе с семейной информацией в self.family_info
        Так же удаляет ребенка из семьи
        """
        with UPDATE(
            self.account,
            self.request.env,
            {
                'action': 'account_delete',
                'consumer': self.consumer,
            },
        ):
            self.account.last_child_family = self.account.family_info.family_id
            timestamp = datetime.now()
            self.account.disabled_status = ACCOUNT_DISABLED_ON_DELETION
            if not self.account.deletion_operation:
                self.account.deletion_operation = AccountDeletionOperation(self, started_at=timestamp)
            self.account.global_logout_datetime = timestamp

        with UPDATE(
            self.family_info,
            self.request.env,
            {
                'action': 'family_member_account_delete',
                'consumer': self.consumer,
            },
        ):
            self.family_info.remove_member_uid(self.account.uid)

        log.debug('Removed uid=%s from family family_id=%s' % (
            self.account.uid,
            self.family_info.family_id,
        ))

    def change_child_avatar(self, url=None, file_obj=None, is_temp=False, avatar_size='normal'):
        avatar_id = self.upload(url=url, file_obj=file_obj, is_temp=is_temp)
        avatar_url = settings.GET_AVATAR_URL % (
                avatar_id,
                avatar_size,
            )
        return avatar_url
