# -*- coding: utf-8 -*-

from collections import namedtuple
import logging

from passport.backend.api.common import account_manager
from passport.backend.api.common.common import (
    check_spammer,
    CleanWebChecker,
)
from passport.backend.api.views.bundle.exceptions import (
    CaptchaRequiredError,
    TvmUserTicketInvalidError,
    TvmUserTicketMissingScopes,
    TvmUserTicketNoUid,
    ValidationFailedError,
)
from passport.backend.core import authtypes
from passport.backend.core.builders.captcha import get_captcha
from passport.backend.core.conf import settings
from passport.backend.core.counters import registration_karma
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.models.persistent_track import PersistentTrack
from passport.backend.core.runner.context_managers import CREATE
from passport.backend.core.tvm.tvm_credentials_manager import get_tvm_credentials_manager
from passport.backend.core.types.answer import normalize_answer
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import remove_none_values
import six
from six.moves.urllib.parse import (
    urlparse,
    urlunparse,
)
from ticket_parser2.exceptions import TicketParsingException

from .. import exceptions


log = logging.getLogger('passport.api.view.bundle.mixins')


class BundleFixPDDRetpathMixin(object):

    def fix_pdd_retpath(self):
        u"""
        Модифицирует переданный в форме retpath:
        удаляет /for/<домен>, если он есть в строке.
        Сделано для очищения наследия моноавторизации.
        """
        retpath = (self.track and self.track.retpath) or self.form_values.get('retpath')
        if not retpath:
            return

        scheme, host, path, params, query, fragment = urlparse(retpath)

        if path.startswith('/for/'):
            # Удаляем из начала строки '/for/'...
            splitted_path = path.replace('/for/', '', 1).split('/')
            if len(splitted_path) >= 1:
                # ...и удаляем домен, если он есть
                path = '/'.join(splitted_path[1:])

                # если последний элемент пустая строка, значит был '/' в конце изначального урла: сохраним его
                if not splitted_path[-1]:
                    path += '/'

        retpath = urlunparse([scheme, host, path, params, query, fragment])
        self.form_values['retpath'] = retpath
        if self.track:
            self.track.retpath = retpath


class BundleFrodoMixin(object):
    """
    Логика отправки информации в ФО
    """

    def frodo_check_spammer(self, action, increment_counters=False, env=None,
                            **kwargs):
        """Отправим в ФО информацию о событии"""
        if env is None:
            env = self.request.env
        frodo_args = remove_none_values(self.form_values)
        frodo_args.update(
            login=self.account.login,
            action=action,
            **kwargs
        )
        self.account, frodo_status = check_spammer(
            self.account,
            env,
            frodo_args,
            self.track,
            self.consumer,
        )
        if increment_counters:
            if self.account.karma.suffix == 100:
                registration_karma.incr_bad(env.user_ip)
            elif self.account.karma.value == 0:
                registration_karma.incr_good(env.user_ip)
        return frodo_status


class BundleAssertCaptchaMixin(object):
    """Проверка прохождения капчи, установка и сброс флагов капчи в треке"""
    @property
    def is_captcha_passed(self):
        """Если в треке есть поля про то, что была успешная проверка капчи"""
        return self.track.is_captcha_checked and self.track.is_captcha_recognized

    def check_track_for_captcha(self, log_fail_to_statbox=True):
        """Проверим что капча не нужна или уже пройдена (защита от брутфорса №1)"""
        if self.track.is_captcha_required and not self.is_captcha_passed:
            if log_fail_to_statbox and not settings.DISABLE_FAILED_CAPTCHA_LOGGING:
                to_statbox({
                    'action': 'captcha_failed',
                    'track_id': self.track.track_id,
                    'input_login': self.track.user_entered_login,
                    'ip': self.client_ip,
                    'user_agent': self.user_agent,
                    'yandexuid': self.cookies.get('yandexuid'),
                    'mode': 'any_auth',
                })
            raise exceptions.CaptchaRequiredError('CAPTCHA is required and not passed')

    def invalidate_captcha(self):
        """Снимаются флажки успешного прохождения капчи -- нужно будет отгадать новую"""
        self.track.is_captcha_checked = False
        self.track.is_captcha_recognized = False
        self.track.captcha_image_url = None

    def clear_captcha_requirement(self):
        """Для этого трека больше не нужна проверка капчей"""
        self.track.is_captcha_required = False

    def generate_mobile_captcha(self):
        captcha_result = get_captcha().generate(
            language=self.track.display_language or '',
            scale_factor=int(self.track.captcha_scale_factor or 1),
            checks=settings.CAPTCHA_CHECKS,
            https=True,
            request_id=self.request.env.request_id,
        )
        self.track.captcha_key = captcha_result.key
        self.track.is_captcha_required = True
        self.invalidate_captcha()
        self.response_values.update(
            captcha_image_url=captcha_result.image_captcha.url,
        )

    def show_mobile_captcha(self):
        self.generate_mobile_captcha()
        raise CaptchaRequiredError()


class BundleHintAnswerCheckMixin(object):
    def compare_answers(self):
        """
        Сравниваем КО в соответствии с теми же правилами, что были в Паспорте на перле.
        """
        previous_answer = normalize_answer(self.account.hint.answer[:settings.HINT_ANSWER_MAX_LENGTH])
        user_entered_answer = self.form_values['answer'] or ''
        user_entered_answer = normalize_answer(user_entered_answer[:settings.HINT_ANSWER_MAX_LENGTH])
        return previous_answer == user_entered_answer


class BundleCacheResponseToTrackMixin(object):
    """Сохраняем ответ ручки в трек - для последующего вызова `get_state`"""

    # Не сохранять в кэш эти поля из ответа
    sensitive_response_fields = []

    cache_field_name = 'submit_response_cache'

    def cache_response_dict_to_track(self, response_dict):
        for field_name in self.sensitive_response_fields:
            if field_name in response_dict:
                del response_dict[field_name]

        setattr(self.track, self.cache_field_name, response_dict)

    def make_cached_response(self):
        cache = getattr(self.track, self.cache_field_name)
        if cache is not None:
            self.response_values = cache

    def dispatch_request(self, **kwargs):
        response = super(BundleCacheResponseToTrackMixin, self).dispatch_request(**kwargs)
        try:
            if self.track:
                response_dict = dict(response.dict_)
                with self.track_transaction.commit_on_error():
                    self.cache_response_dict_to_track(response_dict)
        except Exception as e:
            return self.respond_error(e)
        return response


class BundlePersistentTrackMixin(object):
    """
    Работа с треком, хранимым в БД.
    """

    def create_persistent_track(self, uid, type_, expires_after, **options):
        track_template = PersistentTrack.create(uid, type_, expires_after=expires_after, **options)

        with CREATE(track_template, self.request.env, {}) as track:
            pass

        return track

    def read_persistent_track(self, uid, track_id):
        response = self.blackbox.get_track(uid, track_id)
        return PersistentTrack().parse(response)


class BundleLastAuthMixin(object):
    """
    Получение списка последних авторизаций пользователя
    """
    CREDENTIAL_TYPE = namedtuple('CredentialTypes', 'web_password app_password token')._make([0, 1, 2])
    WEB_PASSWORD_PROTOCOLS = [
        authtypes.AUTH_TYPE_WEB,
        authtypes.AUTH_TYPE_OAUTH_CREATE,
        authtypes.AGGREGATED_AUTH_TYPE_TOKEN_BY_PASSWORD,
    ]

    WEB_PASSWORD_LASTAUTH = 'password'
    APPLICATION_PASSWORDS_LASTAUTH = 'application_passwords'
    OAUTH_LASTAUTH = 'tokens'

    _KEY_MAPPING = {
        'ip': 'value',
        'tokenId': 'token_id',
        'clientId': 'client_id',
        'deviceId': 'device_id',
        'deviceName': 'device_name',
    }

    def get_lastauth(self, uid):
        return self.historydb_api.lastauth(uid)

    def get_auths_aggregated(self, uid=None, limit=None, hours_limit=None, password_auths=None):
        auths, _ = self.historydb_api.auths_aggregated(
            uid=uid or self.account.uid,
            limit=limit,
            hours_limit=hours_limit,
            password_auths=password_auths,
        )
        if password_auths:
            return auths
        return self.build_auths_aggregated(auths)

    def build_auths_aggregated(self, auths):
        lastauth_by_composite_key = {}
        for auth in auths:
            auth_info = auth['auth']
            timestamp = max(auth['authentications'], key=lambda x: x['timestamp'])['timestamp']

            auth_type = auth_info['authtype']
            credential_type = self._get_credential_type(auth_info)
            ip = auth_info['ip']['ip']
            AS = auth_info['ip'].get('AS')

            token_info = auth_info.get('token', {})
            token_id = token_info.get('tokenId')
            # Старые записи проверки токена без token_id
            if token_info and not token_id:
                continue

            key = (auth_type, credential_type, AS or ip, token_id)
            lastauth = lastauth_by_composite_key.get(key)
            if not lastauth or lastauth['timestamp'] < timestamp:
                lastauth_by_composite_key[key] = self._build_info_block(auth_info, timestamp)

        lastauth = {
            self.WEB_PASSWORD_LASTAUTH: {
                'web': [],
                'apps': [],
            },
            self.APPLICATION_PASSWORDS_LASTAUTH: {},
            self.OAUTH_LASTAUTH: {},
        }

        for key, info in lastauth_by_composite_key.items():
            auth_type, credential_type, ip, token_id = key
            if credential_type == self.CREDENTIAL_TYPE.web_password:
                key = self.WEB_PASSWORD_LASTAUTH
                if auth_type in self.WEB_PASSWORD_PROTOCOLS:
                    subkey = 'web'
                else:
                    subkey = 'apps'
            elif credential_type == self.CREDENTIAL_TYPE.app_password:
                key = self.APPLICATION_PASSWORDS_LASTAUTH
                subkey = token_id
            else:
                key = self.OAUTH_LASTAUTH
                subkey = token_id
            lastauth[key].setdefault(subkey, []).append(info)

        for dict_ in lastauth.values():
            for sublist_ in dict_.values():
                sublist_.sort(key=lambda x: x['timestamp'], reverse=True)
        return lastauth

    def _get_credential_type(self, auth_info):
        type_ = auth_info['authtype']
        if type_ == authtypes.AUTH_TYPE_OAUTH_CHECK:
            return self.CREDENTIAL_TYPE.token

        token_info = auth_info.get('token', {})
        is_application_password = token_info.get('AP', False)
        if is_application_password:
            return self.CREDENTIAL_TYPE.app_password
        return self.CREDENTIAL_TYPE.web_password

    def _convert_info_values(self, values):
        result = {}
        for key, value in values.items():
            result[self._KEY_MAPPING.get(key, key)] = value
        return result

    def _build_info_block(self, auth_info, timestamp):
        default = {
            'authtype': auth_info['authtype'],
            'timestamp': timestamp,
            'ip': self._convert_info_values(auth_info['ip']),
        }
        token_info = auth_info.get('token')
        if token_info:
            default['oauth'] = self._convert_info_values(token_info)
        for key in ['browser', 'os']:
            block = auth_info.get(key)
            if block:
                default[key] = self._convert_info_values(block)
        return default


class BundleDeviceInfoMixin(object):
    """Работа с информацией о мобильном устройстве пользователя"""
    # Маппинг из полей формы passport.backend.api.forms.base.DeviceInfoForm в поля трека

    def save_device_params_to_track(self, app_params=None):
        """Записать в трек значения, полученные от АМ"""
        app_params = app_params or self.form_values
        track_fields = self.device_params_to_track_fields(app_params)
        with self.track_transaction.commit_on_error() as track:
            for name in track_fields:
                setattr(track, name, track_fields[name])

    @staticmethod
    def device_params_to_track_fields(app_params):
        return account_manager.device_params_to_track_fields(app_params)

    def get_device_params_from_track(self):
        return account_manager.get_device_params_from_track(self.track)

    @staticmethod
    def form_to_oauth_params(form_values):
        return account_manager.form_to_oauth_params(form_values)

    @staticmethod
    def track_to_oauth_params(params):
        return account_manager.track_to_oauth_params(params)


class BundleAdminActionMixin(object):
    """
    Обработка admin_name и comment для записи их в historydb
    """
    @property
    def is_admin_action(self):
        return (
            self.form_values['admin_name'] is not None and
            self.form_values['comment'] is not None
        )

    def mark_admin_action(self, events):
        if self.is_admin_action:
            events.update(
                admin=self.form_values['admin_name'],
                comment=self.form_values['comment'],
            )


class BundleCleanWebMixin(object):
    @cached_property
    def _checker(self):
        return CleanWebChecker()

    def clean_web_check_form_values(self):
        self._checker.check_form_values(
            self.form_values,
            error_class=lambda fields: ValidationFailedError(['{}.invalid'.format(f) for f in fields]),
            statbox=self.statbox,
        )


class BundleTvmUserTicketMixin(object):
    def check_user_ticket(self, required_scope=None):
        if required_scope and isinstance(required_scope, six.string_types):
            required_scope = [required_scope]
        required_scopes = required_scope or self.required_user_ticket_scopes

        credentials_manager = get_tvm_credentials_manager()
        user_context = credentials_manager.get_user_context()
        if self.request.env.user_ticket is None:
            raise TvmUserTicketInvalidError('No ticket')
        try:
            user_ticket = user_context.check(self.request.env.user_ticket)
        except TicketParsingException as e:
            log.debug('Invalid TVM user ticket: %s, %s, %s' % (e.status, e.message, e.debug_info))
            raise TvmUserTicketInvalidError(e.message)

        if required_scopes and not any([user_ticket.has_scope(s) for s in required_scopes]):
            raise TvmUserTicketMissingScopes(user_ticket.scopes, required_scopes)
        return user_ticket

    def get_uid_or_default_from_user_ticket(self, user_ticket, uid=None):
        if uid is None:
            if not user_ticket.default_uid:
                raise TvmUserTicketNoUid(user_ticket.uids)
            uid = user_ticket.default_uid
        if uid not in user_ticket.uids:
            raise TvmUserTicketNoUid(user_ticket.uids)
        return uid
