# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api.common import extract_tld
from passport.backend.api.views.bundle.exceptions import (
    FamilyAlreadyExists,
    FamilyAlreadyIsMemberOther,
    FamilyAlreadyIsMemberThis,
    FamilyDoesNotExist,
    FamilyInvalidInvite,
    FamilyIsAdmin,
    FamilyIsNotAMember,
    FamilyMaxCapacity,
    FamilyNotAllowedToManageKiddish,
    FamilyNotIsAdmin,
)
from passport.backend.api.views.bundle.mixins.kiddish import BundleKiddishMixin
from passport.backend.core.builders.yasms import get_yasms
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    MailInfo,
    make_email_context,
    render_to_sendmail,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.family import (
    FamilyInfo,
    FamilyInvite,
)
from passport.backend.core.serializers.logs.historydb.runner import log_events
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.types.phone_number.phone_number import mask_for_statbox
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.core.ydb.processors.family_invite import (
    delete_family_invite,
    find_family_invite,
    find_invites_for_family,
)


log = logging.getLogger('passport.api.view.bundle.mixins.family')


class BundleFamilyMixin(object):
    """ Функции, связанные с семьёй """
    def assert_has_family(self, load_family_info=False):
        """ Проверить, что у пользователя есть семья """
        if not self.account.has_family:
            message = 'Account is not a member of family'
            log.debug(message)
            raise FamilyDoesNotExist(message)
        if load_family_info:
            self.load_family_info_by_family_id(self.account.family_info.family_id)

    def assert_has_no_family(self, current_family_id=None):
        """ Проверить, что у пользователя нет семьи """
        if self.account.has_family:
            message = (
                'Account is already a member of family %s'
                % self.account.family_info.family_id
            )
            log.debug(message)
            if current_family_id is None:
                raise FamilyAlreadyExists(message)
            else:
                if self.account.family_info.family_id == current_family_id:
                    raise FamilyAlreadyIsMemberThis(message)
                else:
                    raise FamilyAlreadyIsMemberOther(message)

    def assert_has_family_member(self, uid):
        """ Проверить, что данный uid состоит в семье пользователя """
        if not uid in self.family_info.members:
            raise FamilyIsNotAMember(
                'Account %s is not a member of family %s' % (uid, self.account.family_info.family_id)
            )

    def is_family_admin(self):
        return str(self.account.family_info.admin_uid) == str(self.account.uid)

    def assert_is_family_admin(self, load_family_info=False):
        """ Проверить, что пользователь - администратор семьи
            (включает проверку на семейность)"""
        self.assert_has_family(load_family_info=load_family_info)
        if not self.is_family_admin():
            message = (
                'Account is not an admin of family %s'
                % self.account.family_info.family_id
            )
            log.debug(message)
            raise FamilyNotIsAdmin(message)

    def assert_is_not_family_admin(self, load_family_info=True):
        self.assert_has_family(load_family_info=load_family_info)
        if str(self.account.family_info.admin_uid) == str(self.account.uid):
            message = (
                'Account is an admin of family %s'
                % self.account.family_info.family_id
            )
            log.debug(message)
            raise FamilyIsAdmin(message)

    def assert_allowed_to_manage_kiddish(self):
        self.assert_has_family()
        if self.account.is_kiddish:
            raise FamilyNotAllowedToManageKiddish()

    def load_family_info_by_family_id(self, family_id, get_place=True):
        response = self.blackbox.family_info(
            family_id=family_id,
            get_members_info='all',
            get_place=get_place,
        )
        if not response:
            message = 'Family doesn\'t exist'
            log.debug(message)
            raise FamilyDoesNotExist()
        family_info = FamilyInfo().parse(response)
        log.debug('Got family %s info from BB: %s' % (family_id, family_info))

        self.family_info = family_info

    def fill_response_with_family_roles(
        self,
        need_adults=False,
        need_kids=False,
    ):
        self.response_values.update(
            self.family_roles_to_response(
                need_adults=need_adults,
                need_kids=need_kids,
            ),
        )

    def family_roles_to_response(
        self,
        need_adults=False,
        need_kids=False,
    ):
        uids = list()
        if need_adults:
            uids.extend(self.family_info.members.keys())
        if need_kids:
            uids.extend(self.family_info.kids.members.keys())

        accounts, _ = get_many_accounts_by_uids(
            uids,
            self.blackbox,
            userinfo_args=dict(need_display_name=True),
        )

        response_dict = dict()
        if need_adults:
            response_dict.update(family_members=self._family_members_to_response(accounts))
        if need_kids:
            response_dict.update(family_kids=self._family_kids_to_response(accounts))

        return response_dict

    def _family_members_to_response(self, family_accounts):
        adult_uids = set(self.family_info.members.keys())
        return [
            {
                'uid': a.uid,
                'place_id': '%s:%s' % (
                    self.family_info.family_id,
                    self.family_info.members[a.uid].place,
                ),
                'default_avatar': a.person.default_avatar,
                'display_name': a.person.display_name.public_name,
                'has_plus': bool(a.plus.has_plus),
            } for a in family_accounts if a.uid in adult_uids
        ]

    def _family_kids_to_response(self, family_accounts):
        kid_uids = set(self.family_info.kids.members.keys())
        response = list()
        for kiddish in family_accounts:
            if kiddish.uid in kid_uids:
                kiddish_response = BundleKiddishMixin.kiddish_to_response(self, kiddish, self.family_info)
                response.append(kiddish_response)
        return response

    def find_invite(self, invite_id):
        invite = find_family_invite(invite_id)
        if invite is None:
            message = 'Invalid invite %s' % invite_id
            log.debug(message)
            raise FamilyInvalidInvite(message)
        return invite

    def member_slots_occupied(self, family_info, exclude_invite=None):
        existing_invites = find_invites_for_family(
            family_info.family_id,
            exclude_invite,
        )
        return len(existing_invites) + len(self.family_info.members)

    def assert_family_has_empty_slots(self, exclude_invite=None):
        size = self.member_slots_occupied(self.family_info, exclude_invite)
        if size >= settings.FAMILY_MAX_SIZE:
            message = 'Family reached max capacity'
            log.debug(message)
            raise FamilyMaxCapacity(message)

    def get_family_member_free_place(self):
        place = self.family_info.get_first_free_place(settings.FAMILY_MAX_SIZE)
        if place is None:
            message = 'No free places in family'
            log.debug(message)
            raise FamilyMaxCapacity(message)
        return place

    def get_family_kid_free_place(self):
        place = self.family_info.kids.get_first_free_place(settings.FAMILY_MAX_KIDS_NUMBER)
        if place is None:
            message = 'No free places in family'
            log.debug(message)
            raise FamilyMaxCapacity(message)
        return place

    def delete_invite(self, invite_id):
        delete_family_invite(invite_id)

    @staticmethod
    def generate_invite_url(tld, invite_id):
        return settings.FAMILY_INVITE_URL_TEMPLATE % {
            'base_url': settings.ID_BASE_URL_TEMPLATE % {'tld': tld},
            'invite_id': invite_id,
        }

    def send_invite_by_sms(self, invite, phone):
        language = get_preferred_language(account=self.account)
        user_tld = extract_tld(
            self.request.env.host,
            settings.PASSPORT_TLDS,
        ) or settings.PASSPORT_DEFAULT_TLD
        invite_url = self.generate_invite_url(user_tld, invite.invite_id)
        translations = settings.translations.NOTIFICATIONS[language]
        message = (
            translations['family.invite_sms.message']
            .replace('%INVITER_NAME%', self.account.person.display_name.public_name)
            .replace('%INVITE_URL%', invite_url)
        )

        yasms = get_yasms()
        yasms.send_sms(
            phone_number=phone.e164,
            text=message,
            from_uid=self.account.uid,
            caller=self.consumer,
            identity='family_invite_send.notify',
            client_ip=request.env.user_ip,
            user_agent=request.env.user_agent,
        )
        log.debug(
            'Invite %s is sent to phone %s by uid %s' %
            (invite.invite_id, phone.e164, self.account.uid),
        )

    def send_invite_by_email(self, invite, email):
        language = get_preferred_language(account=self.account)
        user_tld = extract_tld(
            self.request.env.host,
            settings.PASSPORT_TLDS,
        ) or settings.PASSPORT_DEFAULT_TLD
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'INVITER_NAME': self.account.person.display_name.public_name,
                'invite_url': self.generate_invite_url(user_tld, invite.invite_id),
                'greeting_key': 'greeting.noname',
                'signature_key': 'signature',
            },
        )
        phrases = settings.translations.NOTIFICATIONS[language]
        info = MailInfo(
            subject=phrases['family.invite_email.subject'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )
        render_to_sendmail(
            'mail/family_invite_email.html',
            info,
            [email],
            context,
        )
        log.debug(
            'Invite %s is sent to email %s by uid %s' %
            (invite.invite_id, email, self.account.uid),
        )

    def write_history_db(self, uid, create_time, **kwargs):
        user_ip = self.request.env.user_ip
        user_agent = self.request.env.user_agent
        yandexuid = self.request.env.cookies.get('yandexuid')
        log_events(
            user_events={uid: kwargs},
            user_ip=user_ip,
            user_agent=user_agent,
            yandexuid=yandexuid,
            datetime_=create_time,
        )

    def load_family_invites_by_family_id(self, family_id):
        self.family_invites = find_invites_for_family(family_id)

    def family_invite_to_response(self, invite):
        return {
            'invite_id': invite.invite_id,
            'family_id': invite.family_id,
            'create_time': invite.create_time,
            'issuer_uid': invite.issuer_uid,
            'send_method': FamilyInvite.send_method_to_text(
                invite.send_method,
            ),
            'contact': invite.contact,
        }

    def fill_response_with_family_invites(self):
        self.response_values['family_invites'] = [
            self.family_invite_to_response(invite)
            for invite in self.family_invites
        ]

    def fill_response_with_family_settings(self):
        self.response_values['family_settings'] = {
            'max_capacity': settings.FAMILY_MAX_SIZE,
            'max_kids_number': settings.FAMILY_MAX_KIDS_NUMBER,
        }

    @staticmethod
    def _mask_contact_for_statbox(invite):
        if invite.send_method == FamilyInvite.SEND_METHOD_SMS:
            return mask_for_statbox(invite.contact)
        elif invite.send_method == FamilyInvite.SEND_METHOD_EMAIL:
            return mask_email_for_statbox(invite.contact)
        else:
            return invite.contact

    def invite_to_statbox(self, invite):
        return dict(
            invite_id=invite.invite_id,
            family_id=invite.family_id,
            send_method=FamilyInvite.send_method_to_text(invite.send_method),
            contact=self._mask_contact_for_statbox(invite),
        )
