# -*- coding: utf-8 -*-
from datetime import datetime
import logging
import re

from onelogin.saml2.constants import OneLogin_Saml2_Constants
from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.views.bundle.exceptions import (
    DomainNotFoundError,
    EmailExistError,
    EmailInvalidError,
    EmailUnsupportableDomainError,
    LoginNotavailableError,
)
from passport.backend.core import validators
from passport.backend.core.builders.federal_configs_api import (
    BaseFederalConfigsApiError,
    FederalConfigsApi,
    FederalConfigsApiNotFoundError,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.alias import PddAlias
from passport.backend.core.runner.context_managers import CREATE
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription


log = logging.getLogger(__name__)
NAME_ID_PROHIBITED_SYMBOLS_RE = re.compile(r'[^@a-zA-Z0-9._-]')


def check_pdd_alias(login):
    try:
        validators.PddLogin().to_python(login)
        return True
    except validators.Invalid:
        return False


class BundleFederalMixin(object):

    def register_federal(self, domain, name_id, email, firstname, lastname, display_name=None, is_enabled=True):
        self.check_requirements(domain, email, name_id)

        args = dict(
            firstname=firstname,
            lastname=lastname,
            email=email,
            is_enabled=is_enabled,
        )
        # смущает что мы говорили о case-sensitive алиасе, а внутри он наоборот в lower вгоняется
        self.account = default_account(
            alias_type='federal',
            args=args,
            default_person_info=build_default_person_registration_info(self.client_ip),
            login=self.make_alias(name_id, domain),
            registration_datetime=datetime.now(),
        )
        self.account.domain = domain
        with CREATE(
            environment=self.request.env,
            events=dict(action='account_register_federal', consumer=self.consumer),
            model=self.account,
        ):
            self.account.pdd_alias = PddAlias(
                parent=self.account,
                email=email,
            )
            add_subscription(
                self.account,
                get_service(slug='mail'),
            )
            if display_name is not None:
                self.account.person.display_name = display_name

    def check_requirements(self, domain, email, name_id):
        # проверить что домен правда федеральный, чтоб не завести в пдд домене cлучайно
        if not domain.is_enabled or self.get_saml_settings(domain_id=domain.id, only_enabled=True) is None:
            raise DomainNotFoundError()
        # проверяем что домен тот же что и у IdP (пока так)
        login_part, domain_name = email.rsplit('@', 1)
        if domain_name.lower() != domain.domain.lower():
            raise EmailUnsupportableDomainError()
        # Валидируем логиннyю часть почты отдельно, так как она станет основой pdd алиаса (7й)
        if not check_pdd_alias(login_part):
            raise EmailInvalidError('Email contains prohibited symbols')
        # и что нет такого pdd-шнинка (потому что почта помещается в pdd алиас)
        availability_info = self.blackbox.loginoccupation(
            [email],
            is_pdd=True,
        )
        if availability_info[email]['status'] != 'free':
            raise EmailExistError()
        # Проверим федеральный алиас на запрещенные символы
        if NAME_ID_PROHIBITED_SYMBOLS_RE.search(name_id):
            raise LoginNotavailableError('NameId contains prohibited symbols')

    @staticmethod
    def make_alias(name_id, domain):
        if name_id.endswith('@' + domain.domain):
            name_id = name_id.rsplit('@', 1)[0]
        return '%s/%s' % (domain.id, name_id)

    @staticmethod
    def reformat_saml_sso_config(config):
        """Из конфига пришедшего от federal_configs_api создает конфиг в формате python-saml библиотеки,
           заполняет отсутствующие ключи значениями по умолчанию"""
        # в каждом конфиге IdP доступны дополнительные параметры, которых нет по умолчанию в onelogin
        # - lowercase_urlencoding (bool, False by default) - должен быть выставлен в True для ADFS, иначе не пройдет проверка подписи, хэш будет отличаться
        #   на примере: "foo?a=b" -> lowercase_urlencoding=False -> "foo%3Fa%3Db"
        #               "foo?a=b" -> lowercase_urlencoding=True  -> "foo%3fa%3db"
        # - disable_jit_provisioning (bool, False by default) - отключение автоматической регистрации пользователя при заходе через IdP. После выключения
        #                                                       новые пользователи в домене могут появляться только через scim
        # - enabled - вкл/выкл sso на домене, запрещает и вход и синхронизацию контактов через scim

        result = dict(
            enabled=config.get('enabled', True),
            entityId=config['entity_id'],
            singleSignOnService=dict(
                url=config['saml_config']['single_sign_on_service']['url'].strip(),
                binding=config['saml_config']['single_sign_on_service'].get('binding', OneLogin_Saml2_Constants.BINDING_HTTP_REDIRECT),
            ),
            singleLogoutService=dict(
                url=config['saml_config']['single_logout_service']['url'].strip(),
                binding=config['saml_config']['single_logout_service'].get('binding', OneLogin_Saml2_Constants.BINDING_HTTP_REDIRECT),
            ),
            x509certMulti=dict(
                signing=[cert for cert in config['saml_config']['x509_cert'].values() if cert],
            ),
            disable_jit_provisioning=config.get('disable_jit_provisioning', False),
            lowercase_urlencoding=config.get('lowercase_urlencoding', False),
            domain_ids=config.get('domain_ids', []),
        )
        return result

    def get_saml_settings(self, domain_id=None, entity_id=None, only_enabled=True, with_enabled_jit_provisioning=False):
        """ Найти конфиг saml sso на домене, можно по одному из ключей"""
        if not domain_id and not entity_id or (domain_id and entity_id):
            raise ValueError('Specify one of arguments: domain_id or entity_id')
        try:
            if domain_id:
                saml_idp_settings = FederalConfigsApi().get_config_by_domain_id(domain_id=domain_id)
            else:
                saml_idp_settings = FederalConfigsApi().get_config_by_entity_id(entity_id=entity_id)
            saml_settings = dict(settings.get_yandex_sp_config(), idp=self.reformat_saml_sso_config(saml_idp_settings))
            log.debug(
                'Federal config was found: id {}, domain_ids [{}], entity_id `{}`'.format(
                    saml_idp_settings['config_id'],
                    ','.join(map(str, saml_idp_settings['domain_ids'])),
                    saml_idp_settings['entity_id'])
            )
        except FederalConfigsApiNotFoundError:
            saml_settings = None
        except BaseFederalConfigsApiError:
            self.statbox.log(error='get_federal_settings_failed')
            raise
        if (
            saml_settings is None  # нет настроек на домене
            or (only_enabled and not saml_settings['idp'].get('enabled', True))  # настройки ссо есть, но выключены
            or (with_enabled_jit_provisioning and saml_settings['idp'].get('disable_jit_provisioning', False))  # jit_provisioning выключен на домене
        ):
            return
        return saml_settings
