# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.login import build_available_kiddish_login
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    FamilyIsNotAMember,
    InternalTemporaryError,
)
from passport.backend.api.views.bundle.mixins.account import BundleAccountResponseRendererMixin
from passport.backend.core.conf import settings
from passport.backend.core.models.account import (
    ACCOUNT_DISABLED_ON_DELETION,
    AccountDeletionOperation,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids


log = logging.getLogger(__name__)


class BundleKiddishMixin(object):
    def create_kiddish(
        self,
        display_name,
        avatar_id=None,
        birthday=None,
        content_rating_class=None,
        firstname=None,
        gender=None,
        lastname=None,
        music_content_rating_class=None,
        video_content_rating_class=None,
    ):
        """
        Создаёт объект Account с атрибутами ребёнкиша

        Входные параметры

            display_name: types.display_name.DisplayName
            avatar_id: идентификатор аватарки
            birthday: types.birthday.Birthday
            content_rating_class: int
            firstname: str
            gender: types.gender.Gender
            lastname: str
            music_content_rating_class: int
            video_content_rating_class: int

        Выходные параметры

            Объект класса models.account.Account

        Исключения

            api.exceptions.SuggestKiddishLoginError
        """
        login = build_available_kiddish_login(settings.KIDDISH_LOGIN_GENERATION_RETRIES, self.request.env)
        account = default_account(
            alias_type='kiddish',
            args=dict(),
            default_person_info=build_default_person_registration_info(self.client_ip),
            login=login,
            registration_datetime=datetime.now(),
        )

        account.content_rating_class = content_rating_class
        account.music_content_rating_class = music_content_rating_class
        account.video_content_rating_class = video_content_rating_class

        p = account.person
        p.birthday = birthday
        p.default_avatar = avatar_id
        p.display_name = display_name
        p.firstname = firstname
        p.gender = gender
        p.lastname = lastname

        return account

    def fill_response_with_kiddish(self):
        self.response_values['account'] = self.kiddish_to_response(self.account, self.family_info)

    def kiddish_to_response(self, kiddish, family_info=None):
        response = BundleAccountResponseRendererMixin.account_to_response(
            self,
            kiddish,
            personal_data_required=True,
            account_info_required=False,
        )

        # Добавляя сюда новый класс, не забудь добавить его и в ALLOW_ZERO_VALUE_ATTRS,
        # иначе невозможно будет записать в базу данных значение равное нолю.
        all_special_rating_classes = [
            'music_content_rating_class',
            'video_content_rating_class',
        ]

        if kiddish.content_rating_class or kiddish.content_rating_class == 0:
            response.update(content_rating_class=kiddish.content_rating_class)
            # Если специальные значения не задано, то считаем их равными
            # общему значению.
            for special_rating_class in all_special_rating_classes:
                response.update({special_rating_class: kiddish.content_rating_class})

        for special_rating_class in all_special_rating_classes:
            rating = getattr(kiddish, special_rating_class)
            if rating or rating == 0:
                response.update({special_rating_class: rating})

        if family_info and family_info.kids:
            response.update(
                place_id='%s:%s' % (
                    family_info.family_id,
                    family_info.kids.members[kiddish.uid].place,
                ),
            )
        return response

    def check_kiddish_consistent(self, account):
        """
        Проверяет, что данный объект Account является ребёнкишем из действующей
        семьи.

        Входные параметры

            account
                Аккаунт с family_info

            self.family_info
                Объект FamilyInfo

        Исключения

            AccountInvalidTypeError
                Аккаунт не ребёнкиш

            FamilyIsNotAMember
                Аккаунт не входит в действующую семью
        """
        if account.uid not in self.family_info.kids:
            raise FamilyIsNotAMember()

        if not (
            account.family_info and
            account.family_info.family_id == self.family_info.family_id
        ):
            raise FamilyIsNotAMember()

        if not account.is_kiddish:
            raise AccountInvalidTypeError()

    def load_kids(self):
        kid_uids = self.family_info.kids.members.keys()
        self.kids, _ = get_many_accounts_by_uids(
            kid_uids,
            self.blackbox,
            userinfo_args=dict(
                get_family_info=True,
                ip=self.client_ip,
            ),
        )
        self.check_family_kids_consistent()

    def check_family_kids_consistent(self):
        """
        Проверяет, что данные о семье из family_info и userinfo совпадают

        Нужно вызывать после load_kids
        """
        all_kid_uids = set(self.family_info.kids.members.keys())
        known_kid_uids = set([k.uid for k in self.kids])
        unknown_kid_uids = all_kid_uids - known_kid_uids
        if unknown_kid_uids:
            sorted_unknown_uids = map(str, sorted(unknown_kid_uids))
            log.debug('Unable to get userinfo for kids: ' + ','.join(sorted_unknown_uids))
            raise InternalTemporaryError()

        try:
            for kiddish in self.kids:
                self.check_kiddish_consistent(kiddish)
        except AccountInvalidTypeError:
            log.debug('Family kid is not kiddish: %s' % kiddish.uid)
            raise InternalTemporaryError()
        except FamilyIsNotAMember:
            log.debug('Kid does not belong to the family: %s' % kiddish.uid)
            raise InternalTemporaryError()

    def start_kids_deletion(self):
        """
        Создаёт операцию удаления аккаунта для всех детей в семье

        Нужно вызывать после load_kids
        """
        timestamp = datetime.now()
        for kiddish in self.kids:
            with UPDATE(
                kiddish,
                self.request.env,
                events={'action': 'start_kiddish_deletion', 'consumer': self.consumer},
            ):
                kiddish.disabled_status = ACCOUNT_DISABLED_ON_DELETION
                if not kiddish.deletion_operation:
                    kiddish.deletion_operation = AccountDeletionOperation(started_at=timestamp)
