# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.exceptions import RateLimitExceedError
from passport.backend.core.builders.kolmogor.kolmogor import get_kolmogor
from passport.backend.core.counters.kolmogor import (
    Counter,
    KolmogorToolkit,
    OverflowCounterError,
)
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger(__name__)


class KolmogorMixin(object):
    def failsafe_check_kolmogor_counters(self, counters):
        try:
            kolmogor_available = self.kolmogor_toolkit.failsafe_check_counters(counters)
        except OverflowCounterError as e:
            bits = list()
            for counter, value in e.overflow_list:
                bits.append('%s: %d of %d' % (counter.fqn, value, counter.limit))
            message = 'Overflowed counters: %s' % ', '.join(bits)
            log.debug(message)
            raise RateLimitExceedError(message)
        if not kolmogor_available:
            counters = ', '.join(sorted(c.fqn for c in counters))
            log.debug('Unable to check counters %s: kolmogor API unavailable' % counters)
        return kolmogor_available

    def failsafe_inc_kolmogor_counters(self, counters):
        kolmogor_available = self.kolmogor_toolkit.failsafe_inc_counters(counters)
        if not kolmogor_available:
            counters = ', '.join(sorted(c.fqn for c in counters))
            log.debug('Unable to inc counters %s: kolmogor API unavailable' % counters)
        return kolmogor_available

    def build_counter(self, keyspace, name, limit):
        return Counter(keyspace, name, limit)

    @cached_property
    def kolmogor(self):
        return get_kolmogor()

    @cached_property
    def kolmogor_toolkit(self):
        return KolmogorToolkit(self.kolmogor)
