# -*- coding: utf-8 -*-
from binascii import hexlify
from datetime import datetime
import logging
import os

from passport.backend.api.common import safe_detect_timezone
from passport.backend.api.common.account import magic_link_allowed
from passport.backend.api.common.common import extract_tld
from passport.backend.api.templatetags import escape_percents
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    MagicLinkExpiredError,
    MagicLinkInvalidatedError,
    MagicLinkNotSentError,
    RateLimitExceedError,
)
from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    magic_link_per_ip_counter,
    magic_link_per_uid_counter,
)
from passport.backend.core.mailer.utils import (
    MailInfo,
    make_email_context,
    render_to_sendmail,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    datetime_to_string,
    unixtime_to_datetime,
)
import pytz
import six
from six.moves.urllib.parse import urlencode


log = logging.getLogger('passport.api.view.bundle.mixins.magic_link')


MAGIC_LINK_SECRET_BYTES_LENGTH = validators.MAGIC_LINK_SECRET_HEX_LENGTH // 2
MOSCOW_TIMEZONE = pytz.timezone('Europe/Moscow')
DATETIME_FORMAT = '%d.%m.%Y %H:%M'


class BundleMagicLinkMixin:
    def check_magic_link_send_counters(self):
        per_ip_counter = magic_link_per_ip_counter.get_counter(self.client_ip)
        if per_ip_counter.hit_limit(self.client_ip):
            raise RateLimitExceedError(self.client_ip)

        if self.account and magic_link_per_uid_counter.hit_limit(self.account.uid):
            raise RateLimitExceedError(self.account.uid)

    def incr_magic_link_send_counters(self):
        per_ip_counter = magic_link_per_ip_counter.get_counter(self.client_ip)
        per_ip_counter.incr(self.client_ip)

        if self.account:
            magic_link_per_uid_counter.incr(self.account.uid)

    @property
    def is_magic_link_for_registration(self):
        # Почему это важно: в регистрационных ссылках не зашит уид
        return self.track.track_type == 'register'

    @property
    def is_magic_link_allowed(self):
        return magic_link_allowed(self.account, check_counters=False, check_policies=True, allow_lite=True)

    @property
    def is_magic_link_expired(self):
        # magic_link_start_time может отсутствовать только в одном случае, когда перегреты счётчики на magic link
        # При проставлении magic_link_code ___всегда___ должен выставляться magic_link_start_time,
        # иначе мы не будем проверять срок жизни кода
        return (
            self.track.magic_link_start_time and
            datetime_to_integer_unixtime(datetime.now()) - int(self.track.magic_link_start_time) > settings.AUTH_MAGIC_LINK_TTL
        )

    def assert_magic_link_sent(self):
        if not self.track.magic_link_sent_time:
            raise MagicLinkNotSentError()

    def assert_magic_link_not_expired(self):
        if self.is_magic_link_expired:
            raise MagicLinkExpiredError()

    def assert_magic_link_not_invalidated(self):
        if (
            self.track.magic_link_invalidate_time and
            datetime_to_integer_unixtime(datetime.now()) >= int(self.track.magic_link_invalidate_time)
        ):
            raise MagicLinkInvalidatedError()

    def assert_magic_link_not_confirmed(self):
        if self.track.magic_link_confirm_time:
            raise ActionNotRequiredError()

    def check_track_confirm_counters(self):
        if self.track.magic_link_confirms_count.get(default=0) >= settings.AUTH_MAGIC_LINK_CONFIRMS_LIMIT:
            raise RateLimitExceedError()

    def make_magic_link_url(self, tld, secret, redirect_after_confirm=False):
        query_args = dict(
            track_id=self.track_id,
            secret=secret,
        )
        if redirect_after_confirm:
            query_args['redirect'] = 'true'

        if self.is_magic_link_for_registration:
            template = settings.REGISTER_MAGIC_LINK_TEMPLATE
        else:
            template = settings.AUTH_MAGIC_LINK_TEMPLATE

        return template.format(
            tld=tld,
            query_string=urlencode(sorted(query_args.items())),
        )

    def make_magic_link_secret(self):
        random_part = hexlify(os.urandom(MAGIC_LINK_SECRET_BYTES_LENGTH))
        if six.PY3:
            random_part = random_part.decode('utf8')
        if self.is_magic_link_for_registration:
            secret = random_part
        else:
            secret = '%s%x' % (random_part, self.account.uid)
        return secret.lower()

    @property
    def should_renew_magic_link_secret(self):
        """
        Если секрет в треке невалидный, протухший или
        аккаунт в треке отличается от того, что зашит в секрет
        """
        if self.is_magic_link_expired:
            return True

        if self.track.magic_link_secret and not self.is_magic_link_for_registration:
            # На авторизации в секрете зашит и uid
            try:
                secret = validators.MagicLinkSecretKey().to_python(self.track.magic_link_secret)
                # Перестраховываемся. Такая ситуация не должна быть достижима IRL, т.к. уже
                # защищаемся от перезаписи уида в треке на уровне самого трека
                if int(secret['uid']) != int(self.track.uid):
                    return True
            except validators.Invalid:
                return True

        return False

    def get_tld(self, language, is_mobile):
        if is_mobile:
            tld = settings.LANGUAGE_TO_TLD_MAPPING.get(language)
        else:
            tld = extract_tld(self.request.env.host, settings.PASSPORT_TLDS)
        return tld or settings.PASSPORT_DEFAULT_TLD

    def send_auth_magic_link_to_email(self, email, secret, device_name, is_new_link_generated, is_mobile=False,
                                      redirect_after_confirm=False):
        language = get_preferred_language(account=self.account)
        phrases = settings.translations.NOTIFICATIONS[language]
        user_tld = self.get_tld(language, is_mobile)
        magic_link = self.make_magic_link_url(tld=user_tld, secret=secret, redirect_after_confirm=redirect_after_confirm)

        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'TLD': user_tld,
                'MAGIC_LINK': magic_link,
                'BROWSER': escape_percents(device_name or ''),
                'login': self.account.login,
            },
        )
        info = MailInfo(
            subject=phrases['emails.magic_link_sent.title'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )
        log.debug(
            'Sending magic link mail to %s.',
            email,
        )

        send_mail_for_account(
            'mail/email_magic_link_for_auth_sent.html',
            info,
            context,
            self.account,
            context_shadower=None,
            send_to_external=False,
            send_to_native=False,
            specific_email=email,
            is_plain_text=False,
        )

        self.statbox.log(
            address=mask_email_for_statbox(email),
            action='send_magic_link_email',
            magic_link_confirms_count=self.track.magic_link_confirms_count.get(default=0),
            is_new_link=is_new_link_generated,
        )

    def send_auth_confirm_message_to_email(self, email, is_mobile=False):
        language = get_preferred_language(account=self.account)
        phrases = settings.translations.NOTIFICATIONS[language]
        user_tld = self.get_tld(language, is_mobile)
        user_tz = safe_detect_timezone(self.client_ip) or MOSCOW_TIMEZONE

        confirmed_date = unixtime_to_datetime(self.track.magic_link_confirm_time)
        confirmed_date_with_tz = user_tz.localize(confirmed_date)

        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'BROWSER': escape_percents(self.track.magic_link_start_browser or ''),
                'LOCATION': self.track.magic_link_start_location,
                'TIME': datetime_to_string(confirmed_date_with_tz, DATETIME_FORMAT),
                'restore_url': 'restore_url',
            },
        )
        info = MailInfo(
            subject=phrases['emails.magic_link_confirmed.title'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )
        log.debug(
            'Sending magic link confirmed mail to %s.',
            email,
        )
        self.statbox.log(
            address=mask_email_for_statbox(email),
            action='send_magic_link_confirm_email',
        )

        send_mail_for_account(
            'mail/email_magic_link_for_auth_confirmed.html',
            info,
            context,
            self.account,
            context_shadower=None,
            send_to_external=False,
            send_to_native=False,
            specific_email=email,
            is_plain_text=False,
        )

    def send_registration_magic_link_to_email(self, email, language, secret, device_name, is_new_link_generated,
                                              is_mobile=False, redirect_after_confirm=False):
        language = get_preferred_language(account=None, selected_language=language)
        phrases = settings.translations.NOTIFICATIONS[language]
        user_tld = self.get_tld(language, is_mobile)
        magic_link = self.make_magic_link_url(tld=user_tld, secret=secret, redirect_after_confirm=redirect_after_confirm)

        context = make_email_context(
            language=language,
            account=None,
            context={
                'TLD': user_tld,
                'MAGIC_LINK': magic_link,
                'EMAIL': email,
                'DEVICE_NAME': escape_percents(device_name or ''),
                'greeting_key': 'greeting.noname',
                'signature_key': 'emails.magic_link_for_registration_sent.signature',
            },
        )
        info = MailInfo(
            subject=phrases['emails.magic_link_for_registration_sent.title'],
            from_=phrases['email_sender_display_name'],
            tld=user_tld,
        )
        log.debug(
            'Sending magic link mail to %s.',
            email,
        )

        render_to_sendmail(
            'mail/email_magic_link_for_registration_sent.html',
            info,
            recipients=[email],
            context=context,
            is_plain_text=False,
        )

        self.statbox.log(
            address=mask_email_for_statbox(email),
            action='send_magic_link_email',
            magic_link_confirms_count=self.track.magic_link_confirms_count.get(default=0),
            is_new_link=is_new_link_generated,
        )
