# -*- coding: utf-8 -*-

import datetime
import logging

from passport.backend.api.views.bundle.exceptions import RateLimitExceedError
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.utils.common import MixinDependencyChecker


log = logging.getLogger(__name__)


class MailMixin(object):
    __metaclass__ = MixinDependencyChecker

    mixin_dependencies = [
        KolmogorMixin,
        UserMetaDataMixin,
    ]

    def send_notification_mail_for_account(
        self,
        subject_key,
        html_template_path,
        context,
        specific_email=None,
    ):
        language = get_preferred_language(self.account)
        translations = settings.translations.NOTIFICATIONS[language]

        send_mail_for_account(
            html_template_path,
            MailInfo(
                from_=translations['email_sender_display_name'],
                subject=translations[subject_key],
                tld=get_tld_by_country(self.account.person.country),
                subject_format_type='T',
            ),
            make_email_context(
                account=self.account,
                context=context,
                language=language,
            ),
            self.account,
            context_shadower=login_shadower,
            specific_email=specific_email,
        )

    def send_account_modification_mail(
        self,
        event_name,
        context=None,
        specific_email=None,
    ):
        if not (
            event_name in settings.ACCOUNT_MODIFICATION_MAIL_ENABLE and
            (
                is_experiment_enabled_by_uid(self.account.uid, settings.ACCOUNT_MODIFICATION_NOTIFY_DENOMINATOR) or
                self.account.uid in settings.ACCOUNT_MODIFICATION_NOTIFY_WHITELIST
            )
        ):
            return

        if not self.check_account_modification_mail_counters(event_name):
            log.debug('No account modification email event={}: counters overflow'.format(event_name))
            return

        warning_key = 'account_modification.{}.email.warning'.format(event_name)

        context = dict() if context is None else context
        default_context = dict(
            LOCATION=self.get_location(),
            TIMESTAMP=datetime.datetime.now(),
            USER_IP=self.client_ip,
            UID=str(self.account.uid),
            WARNING_KEY=warning_key,
        )
        for key in default_context:
            context.setdefault(key, default_context[key])

        self.send_notification_mail_for_account(
            subject_key=warning_key,
            html_template_path='mail/account_modification_base.html',
            context=context,
            specific_email=specific_email,
        )
        log.debug('Sent account modification email: event={}'.format(event_name))

    def check_account_modification_mail_counters(self, event_name):
        key_template = settings.ACCOUNT_MODIFICATION_PER_UID_COUNTER_TEMPLATE
        sub_key = 'email:{}'.format(event_name)
        key = key_template % (sub_key, self.account.uid)
        counter = self.build_counter(
            settings.ACCOUNT_MODIFICATION_KOLMOGOR_KEY_SPACE,
            key,
            settings.COUNTERS[key_template][sub_key],
        )
        try:
            self.failsafe_check_kolmogor_counters([counter])
            self.failsafe_inc_kolmogor_counters([counter])
            return True
        except RateLimitExceedError:
            return False
