# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.bot_api import (
    BaseBotApiError,
    get_bot_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport.api.view.bundle.mixins.messenger')


class BundleMessengerMixin(object):
    @cached_property
    def bot_api(self):
        return get_bot_api()

    def make_div_card_state(self, state_id, language, inner_content):
        phrases = settings.translations.NOTIFICATIONS[language]
        header = [
            {
                'text': phrases['email_sender_display_name'],
                'type': 'div-title-block',
            },
            {
                'type': 'div-separator-block',
                'size': 'xxs',
            },
        ]
        footer = [
            {
                'type': 'div-separator-block',
                'size': 's',
            },
        ]
        return {
            'state_id': state_id,
            'blocks': header + list(inner_content) + footer,
        }

    def make_div_card(self, states):
        return {
            'states': list(states),
            'background': [
                {
                    'color': '#ffffff',
                    'type': 'div-solid-background',
                },
            ],
        }

    def send_to_messenger(self, text, notification_text, card, uid=None, raise_on_error=True):
        uid = uid or self.account.uid
        log.debug('Sending message for %s to messenger...', uid)
        try:
            rv = self.bot_api.send_message(
                uid=uid,
                text=text,
                notification_text=notification_text,
                card=card,
                request_id=self.request.env.request_id,
            )
            message_id = rv.get('message', {}).get('message_id')
        except BaseBotApiError:
            log.warning('Request to Bot Api failed while sending message')
            self.statbox.log(
                status='error',
                error='bot_api.request_failed',
            )
            if raise_on_error:
                raise
            return

        return message_id

    def edit_messenger_message(self, message_id, text, notification_text, card, uid=None, raise_on_error=True):
        try:
            self.bot_api.edit_message(
                uid=uid or self.account.uid,
                message_id=message_id,
                text=text,
                notification_text=notification_text,
                card=card,
                request_id=self.request.env.request_id,
            )
        except BaseBotApiError:
            log.warning('Request to Bot Api failed while editing message')
            self.statbox.log(
                status='error',
                error='bot_api.request_failed',
            )
            if raise_on_error:
                raise
            return
