# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.login import build_available_neophonish_login
from passport.backend.api.views.bundle.mixins.messenger import BundleMessengerMixin
from passport.backend.api.yasms.api import (
    SaveSecurePhone,
    Yasms,
)
from passport.backend.core.builders.phone_squatter import (
    BasePhoneSquatterError,
    get_phone_squatter,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.phones.phone import PhoneNumber
from passport.backend.core.runner.context_managers import CREATE


log = logging.getLogger(__name__)


class BundleNeophonishMixin(BundleMessengerMixin):
    def _start_phone_tracking_in_squatter(self, phone_number, ignore_errors=False):
        try:
            get_phone_squatter().start_tracking(
                phone_number=phone_number.e164,
                request_id=self.request.env.request_id,
            )
        except BasePhoneSquatterError as e:
            if ignore_errors:
                log.warning('Failed to start phone tracking: %s. Ignoring error.', e)
                return
            raise

    def register_neophonish(self, person_args, start_phone_tracking_in_squatter=False, ignore_squatter_errors=False):
        # Неофониша без ФИО зарегистрировать нельзя,
        # если только не включен новый флоу саджеста (там ФИО не требуется)
        if not (
            settings.USE_NEW_SUGGEST_BY_PHONE or
            (person_args.get('firstname') and person_args.get('lastname'))
        ):
            raise ValueError('Not enough data to create neophonish')

        login = build_available_neophonish_login(
            settings.PHONISH_LOGIN_GENERATION_RETRIES,
            self.request.env,
        )
        new_account = default_account(
            login,
            datetime.now(),
            person_args,
            build_default_person_registration_info(self.client_ip),
            alias_type='neophonish',
        )
        events = {
            'action': 'account_register_neophonish',
            'consumer': self.consumer,
        }

        phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)

        # Если пользователь заводит неофониша используя старый номер, для
        # которого уже действует новый, то создаём неофониша с новым номером,
        # для того чтобы прекратить появление новых неофонишей со старыми
        # номерами.
        phone_number = PhoneNumber.from_deprecated(phone_number)

        save_phone = SaveSecurePhone(
            account=new_account,
            phone_number=phone_number,
            consumer=self.consumer,
            env=self.request.env,
            statbox=self.statbox,
            blackbox=self.blackbox,
            yasms=Yasms(self.blackbox, self.yasms, self.request.env, self),
            should_ignore_binding_limit=True,
            is_new_account=True,
            aliasify=True,
            enable_search_alias=False,
            allow_to_take_busy_alias_from_any_account=False,
            allow_to_take_busy_alias_from_neophonish=True,
            allow_to_take_busy_alias_if_not_search=True,
            language=person_args.get('language') or settings.DEFAULT_LANGUAGE,
        )
        save_phone.submit()

        with CREATE(new_account, self.request.env, events) as self.account:
            if start_phone_tracking_in_squatter:
                self._start_phone_tracking_in_squatter(phone_number, ignore_errors=ignore_squatter_errors)

            save_phone.commit()
            self.unsubscribe_from_maillists_if_nessesary()

        save_phone.after_commit()

    def _make_div_card_registration_complete(self, language, completion_url):
        phrases = settings.translations.NOTIFICATIONS[language]
        return self.make_div_card(
            states=[
                self.make_div_card_state(
                    state_id=1,
                    language=language,
                    inner_content=[
                        {
                            'title': phrases['messenger.neophonish_register.title'],
                            'title_style': 'title_s',
                            'text': phrases['messenger.neophonish_register.notice'],
                            'type': 'div-universal-block',
                        },
                        {
                            'text': phrases['messenger.neophonish_register.completion'] % {'COMPLETION_URL': completion_url},
                            'type': 'div-universal-block',
                        },
                    ],
                ),
            ],
        )

    def _make_div_card_auth_successful(self, language, device_name, location):
        phrases = settings.translations.NOTIFICATIONS[language]
        return self.make_div_card(
            states=[
                self.make_div_card_state(
                    state_id=1,
                    language=language,
                    inner_content=[
                        {
                            'title': phrases['messenger.neophonish_auth.title'],
                            'title_style': 'title_s',
                            'text': phrases['messenger.neophonish_auth.device_name_and_location'] % {
                                'DEVICE_NAME': device_name,
                                'LOCATION': location,
                            },
                            'type': 'div-universal-block',
                        },
                    ],
                ),
            ],
        )

    def send_neophonish_registration_notification_to_messenger(self):
        language = get_preferred_language(self.account)
        phrases = settings.translations.NOTIFICATIONS[language]
        completion_url = phrases['completion_url']
        text = phrases['messenger.neophonish_register.text'] % {
            'COMPLETION_URL': completion_url,
        }
        notification_text = phrases['messenger.neophonish_register.title']
        card = self._make_div_card_registration_complete(
            language=language,
            completion_url=completion_url,
        )
        self.send_to_messenger(
            text=text,
            notification_text=notification_text,
            card=card,
            raise_on_error=False,
        )

    def send_neophonish_auth_notification_to_messenger(self, device_name, location):
        language = get_preferred_language(self.account)
        phrases = settings.translations.NOTIFICATIONS[language]
        text = phrases['messenger.neophonish_auth.text'] % {
            'DEVICE_NAME': device_name,
            'LOCATION': location,
        }
        notification_text = phrases['messenger.neophonish_auth.title']
        card = self._make_div_card_auth_successful(
            language=language,
            device_name=device_name,
            location=location,
        )
        self.send_to_messenger(
            text=text,
            notification_text=notification_text,
            card=card,
            raise_on_error=False,
        )
