# -*- coding: utf-8 -*-

from contextlib import contextmanager

from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    OAuthInvalidClientIdError,
    OAuthInvalidClientSecretError,
    OAuthUnavailableError,
)
import passport.backend.core.builders.oauth.oauth as builder_exceptions
from passport.backend.utils.common import ClassMapping


_EXCEPTIONS_MAPPING = ClassMapping(
    [
        (builder_exceptions.OAuthInvalidClientIdError, OAuthInvalidClientIdError),
        (builder_exceptions.OAuthInvalidClientSecretError, OAuthInvalidClientSecretError),
        (builder_exceptions.OAuthTemporaryError, OAuthUnavailableError),
        (builder_exceptions.OAuthAccountNotFoundError, AccountNotFoundError),
    ],
)


@contextmanager
def builder_to_bundle_exception():
    try:
        yield
    except Exception as builder_exception:
        bundle_exception = _EXCEPTIONS_MAPPING.get(builder_exception.__class__)
        if bundle_exception:
            raise bundle_exception()
        else:
            raise


class BundleOAuthMixin(object):
    def oauth_authorization_code_by_uid(
        self,
        client_id,
        client_secret,
        uid,
        code_challenge=None,
        code_challenge_method=None,
        code_strength=None,
        require_activation=None,
        ttl=None,
    ):
        kwargs = dict()
        if self.client_ip:
            kwargs['user_ip'] = str(self.client_ip)
        kwargs['code_challenge'] = code_challenge
        kwargs['code_challenge_method'] = code_challenge_method
        kwargs['code_strength'] = code_strength
        kwargs['require_activation'] = require_activation
        kwargs['ttl'] = ttl

        with builder_to_bundle_exception():
            oauth_response = self.oauth.issue_authorization_code(
                client_id=client_id,
                client_secret=client_secret,
                uid=uid,
                by_uid=True,
                **kwargs
            )
        return oauth_response['code']

    def oauth_revoke_device(self, uid, device_id):
        user_ip = None if not self.client_ip else str(self.client_ip)
        with builder_to_bundle_exception():
            self.oauth.revoke_device(uid, device_id, user_ip)
