# -*- coding: utf-8 -*-
from datetime import datetime
import logging
import string

import ldap
from passport.backend.api.common.account import set_password_with_experiment
from passport.backend.api.common.authorization import (
    login_password_ok,
    SessionScope,
)
from passport.backend.api.validators import Password as PasswordValidator
from passport.backend.api.views.bundle.constants import (
    CHANGE_PASSWORD_REASON_HACKED,
    CHANGE_PASSWORD_REASON_PWNED,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountDisabledOnDeletionError,
    AccountNotFoundError,
    BaseBundleError,
    BlackboxPermanentError,
    BlackboxUnavailableError,
    CaptchaRequiredAndPasswordNotMatchedError,
    CaptchaRequiredError,
    LdapUnavailableError,
    PasswordChangeForbiddenError,
    PasswordNotMatchedError,
    SecondStepRequired,
)
from passport.backend.api.views.bundle.mixins import BundleAssertCaptchaMixin
from passport.backend.api.views.bundle.utils import is_user_disabled_on_deletion
from passport.backend.core import validators
from passport.backend.core.authtypes import (
    AUTH_TYPE_VERIFY,
    AUTH_TYPE_WEB,
)
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS,
    BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
    BLACKBOX_LOGIN_DISABLED_STATUS,
    BLACKBOX_LOGIN_NOT_FOUND_STATUS,
    BLACKBOX_LOGIN_UNKNOWN_STATUS,
    BLACKBOX_LOGIN_VALID_STATUS,
    BLACKBOX_PASSWORD_BAD_STATUS,
    BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS,
    BLACKBOX_PASSWORD_UNKNOWN_STATUS,
)
from passport.backend.core.builders.blackbox.parsers import can_parse_login_v2_response
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.counters.change_password_counter import get_per_phone_number_buckets
from passport.backend.core.geobase import get_country_code_by_ip
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import (
    ACCOUNT_DISABLED_ON_DELETION,
    get_preferred_language,
)
from passport.backend.core.services import Service
from passport.backend.core.types.birthday import Birthday
from passport.backend.core.types.phone_number.phone_number import (
    parse_phone_number,
    PhoneNumber,
)
from passport.backend.core.utils.ldap import (
    FOREIGN_USERS_DN_PREFIX,
    ldap_search,
    USERS_DN_PREFIX,
)
from passport.backend.core.validators import (
    PasswordForAuth,
    TotpPinValidator,
)
from passport.backend.utils.time import get_unixtime

from ..exceptions import (
    AccountCompromisedError,
    PhoneCompromisedError,
    PhoneVerificationRequiredError,
    ValidationFailedError,
)
from ..phone import exceptions as phone_exceptions


log = logging.getLogger('passport.api.view.bundle.mixins')

PASSWORD_CHANGE_NOTIFICATION_TEMPLATE = 'mail/password_change_notification.html'

NO_VALIDATION = None
CAPTCHA_AND_PHONE_VALIDATION_METHOD = 'captcha_and_phone'
CAPTCHA_VALIDATION_METHOD = 'captcha'

OTP_LENGTH = 8


def looks_like_otp(password):
    """
    Отвечает на вопрос: может ли указанный пароль быть одноразовым паролем?
    Одноразовый пароль всегда состоит из восьми букв латинского алфавита
    в нижнем регистре, может содержать пробелы между 2мя группами по 4 буквы.
    """
    password = (password or '').replace(' ', '')

    return (
        len(password) == OTP_LENGTH and
        all(
            map(
                lambda char: char in string.ascii_lowercase,
                password,
            ),
        )
    )


def validate_password_before_using_for_auth(password, env):
    try:
        PasswordForAuth().to_python(password, validators.State(env))
    except validators.Invalid as ex:
        exception = ValidationFailedError.from_invalid(ex)
        log.info(
            'Password validation: status=error errors=%s',
            ','.join(exception.errors),
        )
        raise exception


class BundleAuthenticateMixin(BundleAssertCaptchaMixin):

    def save_password_length(self, password):
        # Так как сам пароль мы не храним, его длину мы можем узнать только после успешной проверки пароля
        if password and self.account.password and self.account.password.is_set:
            self.account.password.length = len(password)

    def blackbox_login(self, uid=None, login=None, password=None, retpath=None, service=None,
                       ignore_bruteforce_status=False, need_phones=False):
        """Вызываем метод login в ЧЯ, парсим его ответ, заполняем self.account"""
        if password is None:
            password = self.form_values['password']
        if retpath is None:
            retpath = self.form_values['retpath']
        if service is None:
            service = self.form_values['service']

        bb_kwargs = dict(
            uid=uid,
            login=login,
            from_service=(service or Service.by_sid(8)).slug,
            useragent=self.user_agent,
            referer=self.referer,
            retpath=retpath,
            yandexuid=self.cookies.get('yandexuid'),
            authtype=AUTH_TYPE_WEB,
            emails=True,
            find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
            country=get_country_code_by_ip(self.client_ip),
        )
        if need_phones:
            bb_kwargs = add_phone_arguments(**bb_kwargs)

        validate_password_before_using_for_auth(password, self.request.env)
        bb_response = self.blackbox.login(
            password,
            self.client_ip,
            **bb_kwargs
        )

        login_status = bb_response['login_status']
        password_status = bb_response['password_status']
        bruteforce = bb_response['bruteforce_status'] if not ignore_bruteforce_status else None
        allowed_second_steps = bb_response.get('allowed_second_steps')

        self.track.bruteforce_status = bruteforce
        self.track.badauth_counts = bb_response.get('badauth_counts')

        # Парсим инфо о пользователе если ответ ЧЯ содержательный
        if can_parse_login_v2_response(bb_response):
            self.parse_account_with_domains(bb_response)
            uid = self.account.uid

        if bb_response.get('restricted_session', False):
            self.track.is_session_restricted = True

        self.track.session_scope = str(SessionScope.xsession)

        # Сначала проверим ответ ЧЯ - что нужно или НЕ нужно показывать капчу (защита от брутфорса №2)
        self.track.is_captcha_required = (bruteforce == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS)
        # Если капчу уже вводили правильно, позволим войти даже если ЧЯ снова требует капчу
        is_login_allowed_by_captcha = self.is_captcha_passed if self.track.is_captcha_required else True

        password_like_otp = looks_like_otp(password)
        if self.account and self.account.totp_secret.is_set:
            self.statbox.bind(
                is_2fa_enabled=True,
                password_like_otp=password_like_otp,
            )

        # И только потом (sic!) проверим логин, пароль и капчу
        if login_password_ok(login_status, password_status) and is_login_allowed_by_captcha:
            self.totp_check_time = bb_response.get('totp_check_time')
            self.track.password_verification_passed_at = get_unixtime()
            # PASSP-9728 Сбрасываем необходимость показа капчи при успешном входе
            self.clear_captcha_requirement()
            self.save_password_length(password)
            return

        # Далее идут только ошибки: специфицируем ошибку авторизиции
        self.report_login_error(
            uid,
            bruteforce,
            login_status,
            password_status,
            password_like_otp=password_like_otp,
            allowed_second_steps=allowed_second_steps,
            retpath=retpath,
        )

    def report_login_error_antifraud(self, comment_fields=None, **kwargs):
        view_types = (
            getattr(self, attr, None) for attr in ('type', 'statbox_type')
            if hasattr(self, attr)
        )
        try:
            view_type = next(view_types)
        except StopIteration:
            view_type = self.__class__.__name__
        comment_fields = [view_type] + (comment_fields or [])
        kwargs['comment'] = '/'.join(str(f or '-') for f in comment_fields)
        kwargs['status'] = 'FAILED'
        self.antifraud_log.log(**kwargs)

    def determine_login_error(
        self, bruteforce, login_status, password_status, password_like_otp,
        allowed_second_steps=None,
    ):
        # Если капча была пройдена успешно, покажем откровенную ошибку
        is_captcha_passed = self.is_captcha_passed
        # Если ЧЯ считает, что идет брутфорс - старая капча протухла
        if self.track.is_captcha_required:
            self.invalidate_captcha()

        # Сообщаем о том что пароль неправильный при срабатывании
        # BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS только в том случае, если
        # человек прошел успешно капчу
        if (
            bruteforce == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS and
            login_status == BLACKBOX_LOGIN_VALID_STATUS and
            password_status == BLACKBOX_PASSWORD_BAD_STATUS and
            is_captcha_passed
        ):
            raise CaptchaRequiredAndPasswordNotMatchedError()

        elif bruteforce == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS:
            raise CaptchaRequiredError()

        elif login_status == BLACKBOX_LOGIN_DISABLED_STATUS:
            if self.account.disabled_status == ACCOUNT_DISABLED_ON_DELETION:
                raise AccountDisabledOnDeletionError('Account has been subscribed to blocking SIDs')
            else:
                raise AccountDisabledError('Account is disabled')

        elif login_status == BLACKBOX_LOGIN_NOT_FOUND_STATUS:
            raise AccountNotFoundError('Account is not found')

        elif (
            password_status == BLACKBOX_PASSWORD_UNKNOWN_STATUS or
            login_status == BLACKBOX_LOGIN_UNKNOWN_STATUS
        ):
            log.error('Unknown login status')
            raise BlackboxUnavailableError('Unknown account status')

        elif (
            login_status == BLACKBOX_LOGIN_VALID_STATUS and
            password_status == BLACKBOX_PASSWORD_BAD_STATUS
        ):
            exception = PasswordNotMatchedError('Bad password')
            # Если есть 2ФА, будет та же ошибка, но с инфой о пользователе в ответе
            self.raise_if_otp_enabled(
                exception,
                password_like_otp=password_like_otp,
            )
            raise exception

        elif (
            login_status == BLACKBOX_LOGIN_VALID_STATUS and
            password_status == BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS
        ):
            # Логин и пароль верны, но требуется второй шаг. Дальнейшие действия зависят от вызывающего.
            raise SecondStepRequired(allowed_second_steps=allowed_second_steps)

        # Вообще-то сюда мы не должны никак попасть, а если попали,
        # то ЧЯ возвращает что-то невменяемое. Запишем в лог и упадем.
        log.info(
            'Blackbox returned login_status="%s" password_status="%s" bruteforce_status="%s"',
            login_status,
            password_status,
            bruteforce,
        )
        raise BlackboxPermanentError()

    def report_login_error(
        self, uid, bruteforce, login_status, password_status, password_like_otp,
        allowed_second_steps=None, retpath=None
    ):
        """Обрабатываем ВСЕ ВОЗМОЖНЫЕ ошибки авторизации"""
        self.statbox.log(
            action='failed_auth',
            bruteforce=bruteforce,
            login_status=login_status,
            password_status=password_status,
            uid=uid,
        )
        try:
            self.determine_login_error(
                bruteforce, login_status, password_status, password_like_otp,
                allowed_second_steps,
            )
        except SecondStepRequired:
            raise
        except BaseBundleError as err:
            self.report_login_error_antifraud(
                uid=uid,
                comment_fields=[login_status, password_status, bruteforce, err.error],
                retpath=retpath,
            )
            raise

    def raise_if_otp_enabled(self, exception, password_like_otp):
        """
        Если пользователь использует двухфакторную авторизацию,
        он должен ввести одноразовый пароль (otp).
        Здесь в ответ добавляется информация о пользователе из запроса и пользователях из куки
        После бросается переданное исключение
        """
        if self.account.totp_secret.is_set:
            self.statbox.bind(is_2fa_enabled=True)

            # Ответим всей известной информацией об аккаунте(ах)
            self.fill_response_with_account(
                personal_data_required=True,
                account_info_required=True,  # Добавим в ответ инфо о 2ФА
            )

            # FIXME: При авторизации по паролю куки нужно проверять всегда - перенести выше
            session_info = self.check_session_cookie(dbfields=[])
            self.response_values['accounts'] = self.get_multisession_accounts(session_info)
            self.response_values['password_like_otp'] = password_like_otp

            raise exception


class BundleAuthenticateMixinV2(BundleAuthenticateMixin):
    """В отличие от миксина-родителя, не требует повторно вводить верный пароль после ввода верной капчи"""
    def blackbox_login(self, uid=None, login=None, password=None, retpath=None, service=None,
                       need_phones=False, force_use_cache=True, authtype=AUTH_TYPE_WEB, require_app_password=False,
                       **bb_kwargs):
        """Вызываем метод login в ЧЯ, парсим его ответ, заполняем self.account"""
        bb_response = None
        credentials_passed = bool((login or uid) and password)
        if not self.track.blackbox_login_status or (not force_use_cache and credentials_passed):
            # Идём в ЧЯ, если нет закэшированного ответа, или же если нам явно передали логин и пароль.
            # Если передан force_use_cache=True - игнорируем переданный пароль, пытаемся взять ответ из кэша.
            if not credentials_passed:
                # Если не передали необходимые параметры - отваливаемся сразу
                self.report_login_error_antifraud(uid=uid, comment_fields=['login_or_password_not_passed'])
                raise PasswordNotMatchedError('Login or password not passed')

            bb_kwargs = dict(
                uid=uid,
                login=login,
                from_service=(service or Service.by_sid(8)).slug,
                useragent=self.user_agent,
                referer=self.referer,
                retpath=retpath,
                yandexuid=self.cookies.get('yandexuid'),
                authtype=authtype,
                emails=True,
                find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
                country=get_country_code_by_ip(self.client_ip),
                **bb_kwargs
            )
            if need_phones:
                bb_kwargs = add_phone_arguments(**bb_kwargs)

            try:
                validate_password_before_using_for_auth(password, self.request.env)
            except ValidationFailedError:
                self.report_login_error_antifraud(comment_fields=['invalid_password_format'], uid=uid)
                raise
            bb_response = self.blackbox.login(
                password,
                self.client_ip,
                **bb_kwargs
            )

            if bb_response.get('restricted_session', False):
                self.track.is_session_restricted = True

            if bb_response.get('is_scholar_session'):
                self.track.session_scope = str(SessionScope.scholar)
            else:
                self.track.session_scope = str(SessionScope.xsession)

            blackbox_login_status = bb_response['login_status']
            blackbox_password_status = bb_response['password_status']
            bruteforce_status = bb_response['bruteforce_status']
            allowed_second_steps = bb_response.get('allowed_second_steps')
            if not (
                blackbox_login_status == BLACKBOX_LOGIN_UNKNOWN_STATUS or blackbox_password_status == BLACKBOX_PASSWORD_UNKNOWN_STATUS
            ):
                self.track.blackbox_login_status = blackbox_login_status
                self.track.blackbox_password_status = blackbox_password_status
                self.track.bruteforce_status = bruteforce_status
                self.track.allowed_second_steps = allowed_second_steps

            totp_check_time = bb_response.get('totp_check_time')
            if totp_check_time:
                self.track.blackbox_totp_check_time = totp_check_time

            self.track.badauth_counts = bb_response.get('badauth_counts')
        else:
            blackbox_login_status = self.track.blackbox_login_status
            blackbox_password_status = self.track.blackbox_password_status
            bruteforce_status = self.track.bruteforce_status
            allowed_second_steps = self.track.allowed_second_steps

        if bb_response is not None:
            # Парсим инфо о пользователе, если это возможно
            if can_parse_login_v2_response(bb_response):
                self.parse_account_with_domains(bb_response)
                self.track.uid = self.account.uid
                # Сбросим флаги авторизации, чтобы избежать уязвимостей при гонках.
                # TODO: избавиться от этого костыля, поменяв архитектуру уязвимых ручек
                self.track.allow_authorization = False
                self.track.allow_oauth_authorization = False

        # При повторном запросе, получаем информацию через userinfo()
        # Если в треке нет uid, значит его не вернул ЧЯ при ответе на прошлый запрос
        elif self.track.uid:
            self.get_account_by_uid(
                self.track.uid,
                check_disabled_on_deletion=True,
                need_phones=need_phones,
            )

        self.totp_check_time = self.track.blackbox_totp_check_time

        if require_app_password and self.account and not self.account.enable_app_password:
            blackbox_password_status = BLACKBOX_PASSWORD_BAD_STATUS

        is_password_ok = login_password_ok(blackbox_login_status, blackbox_password_status)

        # Сначала проверим ответ ЧЯ - что не нужно показывать капчу (защита от брутфорса №2)
        self.track.is_captcha_required = (bruteforce_status == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS)
        is_captcha_ok = self.is_captcha_passed if self.track.is_captcha_required else True

        password_like_otp = looks_like_otp(password)
        if self.account and self.account.totp_secret.is_set:
            self.statbox.bind(
                is_2fa_enabled=True,
                password_like_otp=password_like_otp,
            )

        # И только потом (sic!) проверим логин, пароль и капчу
        if is_password_ok and is_captcha_ok:
            self.track.password_verification_passed_at = get_unixtime()
            self.clear_captcha_requirement()
            self.save_password_length(password)
            return
        elif is_captcha_ok:
            self.track.blackbox_login_status = None
            self.track.blackbox_password_status = None
            self.track.bruteforce_status = bruteforce_status = None
            self.clear_captcha_requirement()
            self.invalidate_captcha()

        # Далее идут только ошибки: специфицируем ошибку авторизации
        self.report_login_error(
            self.track.uid,
            bruteforce_status,
            blackbox_login_status,
            blackbox_password_status,
            password_like_otp=password_like_otp,
            allowed_second_steps=allowed_second_steps,
            retpath=retpath,
        )


class BundleVerifyPasswordMixin(object):

    def verify_password(self, uid, password, ignore_bruteforce_status=False,
                        check_disabled_on_deletion=False,
                        ignore_statbox_log=False, need_phones=False, **blackbox_kwargs):
        """
        Проверяем пароль на аккаунте.

        Кроме непосредственной проверки указанного пароля в ЧЯ с помощью method=login,
        также убеждается, что в треке не взведён флаг о необходимости проверки капчи.
        В этом случае, потребитель должен отработать с ручками генерации и проверки
        капчи, которые в случае успешной проверки этот флаг опустят.

        Факт проверки капчи сбрасываем после каждой проверки пароля. Если в ответе от ЧЯ не было
        сообщения о необходимости показать капчу, сбрасываем требование обязательного показа капчи.

        @raise: CaptchaRequiredError, AccountNotFoundError, AccountDisabledError,
        AccountDisabledOnDeletionError, PasswordNotMatchedError,
        CaptchaRequiredAndPasswordNotMatchedError
        """
        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)

        validate_password_before_using_for_auth(password, self.request.env)
        response = self.blackbox.login(
            uid=uid,
            password=password,
            ip=self.client_ip,
            authtype=AUTH_TYPE_VERIFY,
            **blackbox_kwargs
        )

        bruteforce = response.get('bruteforce_status')
        login_status = response['login_status']
        password_status = response['password_status']
        allowed_second_steps = response.get('allowed_second_steps')

        is_bruteforce = bruteforce == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS

        # Сработал bruteforce, но капча не пройдена. В этом случае
        # нам не важно, правильный пароль или нет: требуем капчу
        if not ignore_bruteforce_status and is_bruteforce and not self.track.is_captcha_recognized:
            self.track.is_captcha_required = True
            raise CaptchaRequiredError()

        if not ignore_bruteforce_status and not is_bruteforce:
            self.track.is_captcha_required = False

        if login_password_ok(login_status, password_status):
            self.track.password_verification_passed_at = get_unixtime()
            return response
        else:
            self.clear_captcha_statuses()

            if (
                login_status == BLACKBOX_LOGIN_VALID_STATUS and
                password_status == BLACKBOX_PASSWORD_BAD_STATUS
            ):

                if not ignore_statbox_log:
                    self.statbox.log(operation='check_password', error='password.not_matched')

                if self.track.is_captcha_required:
                    raise CaptchaRequiredAndPasswordNotMatchedError()
                else:
                    raise PasswordNotMatchedError()
            elif login_status == BLACKBOX_LOGIN_NOT_FOUND_STATUS:
                raise AccountNotFoundError()
            elif login_status == BLACKBOX_LOGIN_DISABLED_STATUS:
                if check_disabled_on_deletion and is_user_disabled_on_deletion(response):
                    raise AccountDisabledOnDeletionError()
                else:
                    raise AccountDisabledError()
            elif (
                login_status == BLACKBOX_LOGIN_VALID_STATUS and
                password_status == BLACKBOX_PASSWORD_SECOND_STEP_REQUIRED_STATUS
            ):
                raise SecondStepRequired(allowed_second_steps=allowed_second_steps)
            elif (password_status == BLACKBOX_PASSWORD_UNKNOWN_STATUS or
                  login_status == BLACKBOX_LOGIN_UNKNOWN_STATUS):
                raise BlackboxUnavailableError('Unknown account status')

            # Вообще-то сюда мы не должны никак попасть, а если попали,
            # то ЧЯ возвращает что-то невменяемое. Запишем в лог и упадем.
            log.info(
                'Blackbox returned login_status="%s" password_status="%s" bruteforce_status="%s"',
                login_status,
                password_status,
                bruteforce,
            )
            raise BlackboxPermanentError()

    def clear_captcha_statuses(self):
        self.track.is_captcha_checked = False
        self.track.is_captcha_recognized = False


class BundlePasswordValidationMixin(object):
    """
    Логика проверки пароля на сооответствие определенной политике
    """
    def validate_password(self, uid=None, login=None, phone_number=None, emails=None, is_strong_policy=None,
                          old_password_hash=None, required_check_password_history=False, country=None,
                          env=None):
        """Провалидируем пароль с учетом логина пользователя и политики пароля"""
        if uid is None and self.account:
            uid = self.account.uid

        is_strong_policy = is_strong_policy or (self.track and self.track.is_strong_password_policy_required)
        p = PasswordValidator(required_check_password_history=required_check_password_history)
        args = {
            'password': self.form_values['password'],
            'uid': uid,
            'login': login if login is not None else self.account.login,
            'policy': 'strong' if is_strong_policy else 'basic',
            'old_password_hash': old_password_hash or (self.track and self.track.password_hash),
            'track_id': self.track_id,
            'country': country or (self.track and self.track.country),
            'emails': emails or set(),
        }
        # Если у пользователя есть защищённый номер и ему не требуется его подтверждать,
        # тогда учитываем именно этот номер при валидации пароля.
        if self.track and self.track.can_use_secure_number_for_password_validation and self.track.secure_phone_number:
            args['phone_number'] = self.track.secure_phone_number

        # Если у пользователя есть защищённый номер и ему требуется его подтверждать
        # или защищённого номера нет и пользователь его привязывает,
        # тогда учитываем именно этот номер при валидации пароля.
        # Эта ситуация всегда главнее предыдущей, поскольку номер привязан
        # в ходе процесса задания пароля.
        if self.track and self.track.phone_confirmation_is_confirmed and self.track.phone_confirmation_phone_number:
            args['phone_number'] = self.track.phone_confirmation_phone_number

        # Если номер телефона был передан явно
        if phone_number is not None:
            args['phone_number'] = phone_number

        try:
            result = p.to_python(args, validators.State(env or self.request.env))
            # Этот параметр сохраняется для передачи в ФО
            self.form_values['quality'] = result['quality']
            return result['password'], result['quality']
        except validators.Invalid as ex:
            exception = ValidationFailedError.from_invalid(ex)
            log.info(
                'Password validation: status=error errors=%s',
                ','.join(exception.errors),
            )
            raise exception


class ActiveDirectoryMixin(object):
    @staticmethod
    def _get_ldap_connection():
        ldap_url = '%s://%s:%d' % (settings.LDAP_SCHEME, settings.LDAP_HOST, settings.LDAP_PORT)
        ld = ldap.initialize(ldap_url)
        ld.set_option(ldap.OPT_X_TLS, ldap.OPT_X_TLS_DEMAND)
        ld.set_option(ldap.OPT_X_TLS_CACERTFILE, settings.SSL_CA_CERT)
        ld.set_option(ldap.OPT_X_TLS_REQUIRE_CERT, ldap.OPT_X_TLS_DEMAND)
        ld.set_option(ldap.OPT_NETWORK_TIMEOUT, settings.LDAP_NETWORK_TIMEOUT)
        ld.set_option(ldap.OPT_TIMEOUT, settings.LDAP_TIMEOUT)
        ld.set_option(ldap.OPT_X_TLS_NEWCTX, ldap.OPT_OFF)
        ld.protocol_version = ldap.VERSION3
        ld.start_tls_s()
        return ld

    @staticmethod
    def _encode_password(password):
        return u'"{0}"'.format(password).encode('utf-16-le')

    def change_password_in_active_directory(self, old_password, new_password, is_forced):
        try:
            ld = self._get_ldap_connection()

            if is_forced:
                login_with_domain = '%s@%s' % (settings.PASSPORT_ROBOT_LOGIN, settings.LDAP_USER_DOMAIN)
                log.debug('Binding to AD with login=%s...', login_with_domain)
                ld.simple_bind_s(login_with_domain, settings.PASSPORT_ROBOT_PASSWORD)
            else:
                login_with_domain = '%s@%s' % (self.account.login, settings.LDAP_USER_DOMAIN)
                log.debug('Binding to AD with login=%s and old password...', login_with_domain)
                ld.simple_bind_s(login_with_domain, old_password)

            search_result = ldap_search(
                login=self.account.login,
                ldap_client=ld,
                attrlist=['cn'],  # нужно указать что-то, иначе вернутся все атрибуты
                dn_prefixes=(USERS_DN_PREFIX, FOREIGN_USERS_DN_PREFIX),  # роботам нельзя
            )

            if not search_result:
                raise AccountNotFoundError('User not found in AD')

            dn, _ = search_result
            operations = [
                (ldap.MOD_DELETE, 'unicodePwd', self._encode_password(old_password)),
                (ldap.MOD_ADD, 'unicodePwd', self._encode_password(new_password)),
            ]
            log.debug('Trying to change password for %s', dn)
            ld.modify_s(dn, operations)
            log.debug('Password successfully changed for %s', dn)
            ld.unbind()
        except ldap.INVALID_CREDENTIALS:
            raise PasswordNotMatchedError('Current password not matched')
        except ldap.CONSTRAINT_VIOLATION:
            raise PasswordChangeForbiddenError('Bad new password or too frequent changes')
        except ldap.LDAPError as e:
            log.warning('LDAP error: %s', e)
            raise LdapUnavailableError(e)


class BundlePinValidationMixin(object):
    """
    Логика проверки пина 2фа
    """
    def validate_pin(self, pin):
        country = None
        phone_number = None
        birthday = None
        if self.track:
            country = self.track.country
            phone_number = parse_phone_number(self.track.secure_phone_number, country)
            if self.track.birthday:
                birthday = Birthday.parse(self.track.birthday)

        validator = TotpPinValidator(
            phone_number=phone_number,
            country=country,
            birthday=birthday,
        )
        try:
            return validator.to_python(pin, validators.State(self.request.env))
        except validators.Invalid as ex:
            raise ValidationFailedError.from_invalid(ex, field='pin')


class BundlePasswordChangeMixin(object):
    """
    Логика смены пароля
    """

    def clear_password_options(self, force_password_change=False, changing_reason=None):
        self.account.password.setup_password_changing_requirement(
            is_required=force_password_change,
            changing_reason=changing_reason,
        )
        self.account.password.is_creating_required = False
        self.account.password.pwn_forced_changing_suspended_at = None

    def update_revokers(self, global_logout=True,
                        revoke_web_sessions=False, revoke_tokens=False, revoke_app_passwords=False):
        # Хотим минимизировать число записываемых ревокеров
        if revoke_web_sessions and revoke_tokens and revoke_app_passwords:
            global_logout = True
        if global_logout:
            revoke_web_sessions = False
            revoke_tokens = False
            revoke_app_passwords = False

        if global_logout:
            self.account.global_logout_datetime = datetime.now()
            if self.track:
                self.track.is_web_sessions_logout = True
        if revoke_web_sessions:
            self.account.web_sessions_revoked_at = datetime.now()
            if self.track:
                self.track.is_web_sessions_logout = True
        if revoke_tokens:
            self.account.tokens_revoked_at = datetime.now()
        if revoke_app_passwords:
            self.account.app_passwords_revoked_at = datetime.now()

    def change_password(self, new_password, quality=0,
                        global_logout=True,
                        revoke_web_sessions=False, revoke_tokens=False,
                        revoke_app_passwords=False, force_password_change=False, changing_reason=None):
        """
        Аккаунт получает новый пароль, поэтому
         * Снимаем признак необходимости смены пароля
         * Снимаем признак необходимости смены пароля и КВ/КО при следующем входе, если надо
         * Снимаем признак индульгенции смены pwned-пароля
         * Выполняем глобальный logout или частично отзываем веб-сессии / токены / ПП
         * Очищаем историю "плохих сессий"
        """
        self.clear_password_options(
            force_password_change=force_password_change,
            changing_reason=changing_reason,
        )

        if new_password is not None:
            set_password_with_experiment(
                account=self.account,
                password=new_password,
                quality=quality,
                uid=self.account.uid,
            )
        else:
            self.account.password = None

        self.update_revokers(
            global_logout=global_logout,
            revoke_web_sessions=revoke_web_sessions,
            revoke_tokens=revoke_tokens,
            revoke_app_passwords=revoke_app_passwords,
        )

    def get_sms_notification_text(self, language):
        return settings.translations.NOTIFICATIONS[language]['password_changed.sms']

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = PASSWORD_CHANGE_NOTIFICATION_TEMPLATE
        info = MailInfo(
            subject=translations['password_changed.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
        )
        return template_name, info, context

    def send_notifications(self, frodo_status, secure_number):
        language = get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=False)

        if (
            not self.is_sms_validation_required and
            frodo_status and
            self.is_account_compromised(frodo_status) and
            secure_number
        ):
            self.send_sms(
                secure_number,
                self.get_sms_notification_text(language),
                identity='change_password_notification',
            )

    @property
    def is_sms_validation_required(self):
        """
        Нужно ли затребовать проверку пользователя с помощью sms.
        """
        return self.track.is_change_password_sms_validation_required

    @property
    def saved_number(self):
        if self.track.phone_confirmation_phone_number:
            return PhoneNumber.parse(self.track.phone_confirmation_phone_number)

    def assert_secure_phone(self, secure_number):
        """
        Проверим успешно ли провалидирован защищённый телефон.
        Проверим не достиг ли лимит на привязку данного номера.
        Запишем всю информацию в statbox лог.
        """
        if not (self.track.phone_confirmation_is_confirmed and self.saved_number):
            raise PhoneVerificationRequiredError('Account is not verified by sms to secure number')

        # На данный момент этот блок кода необходим для принудительной смены пароля,
        # потому что сейчас именно там происходит привязка номера
        if self.track.has_secure_phone_number and secure_number == self.saved_number:
            return

        if not self.track.has_secure_phone_number and secure_number:
            raise phone_exceptions.SecurePhoneBoundAndConfirmedError()

        counter = get_per_phone_number_buckets()
        is_exceeded = counter.hit_limit(self.saved_number.e164)

        if not is_exceeded:
            return

        self.statbox.log(
            action='checked_counter',
            error='phone.compromised',
            track_id=self.track.track_id,
            counter_current_value=counter.get(self.saved_number.e164),
            counter_limit_value=counter.limit,
        )
        raise PhoneCompromisedError()

    @property
    def forbidden_change_password_with_bad_frodo_karma(self):
        return False

    def is_account_compromised(self, frodo_status):
        return frodo_status.is_karma_prefix_returned and self.account.karma.prefix in (7, 8)

    def assert_account_not_compromised(self, frodo_status):
        """Определяет по ответу ФО и выставленной карме, был ли скомпрометирован аккаунт"""
        self.statbox.bind(
            action='analyzed_frodo',
            karma_prefix=self.account.karma.prefix,
            is_karma_prefix_returned=frodo_status.is_karma_prefix_returned,
            uid=self.account.uid,
        )

        if (
            self.forbidden_change_password_with_bad_frodo_karma and
            self.is_account_compromised(frodo_status)
        ):
            self.statbox.log(error='account.compromised')
            raise AccountCompromisedError()

        self.statbox.log()

    def is_enable_sms_2fa_required(self):
        if not settings.ENABLE_SMS_2FA_AFTER_CHANGE_PASSWORD:
            return False

        if self.account.sms_2fa_on or self.account.totp_secret.is_set or not self.account.password.is_set:
            return False

        secure_number = self.secure_number_or_none
        if secure_number is None:
            return False
        secure_logical_operation = self.account.phones.secure.get_logical_operation(self.statbox)
        is_in_quarantine = secure_logical_operation and secure_logical_operation.in_quarantine
        if is_in_quarantine:
            return False

        change_password_reasons = [
            CHANGE_PASSWORD_REASON_HACKED,
            CHANGE_PASSWORD_REASON_PWNED,
        ]
        if self.track.change_password_reason not in change_password_reasons:
            return False

        return True


class BundlePasswordVerificationMethodMixin(object):
    """Определение способа верификации пользователя при смене пароля"""

    def get_validation_method(self, default_method=NO_VALIDATION):
        """Тут определим, как валидируем пользователя:
        - будем ли вообще как-то валидировать
        - будем ли валидировать по капче
        - будем ли валидировать по sms и капче
        """
        # Без дополнительных проверок по умолчанию
        validation_method = default_method

        # Только капча для обычного пользователя при принудительной смене пароля
        if self.track.is_password_change:
            is_captcha_passed = self.track.is_captcha_checked and self.track.is_captcha_recognized

            is_captcha_required_and_not_passed = self.track.is_captcha_required and not is_captcha_passed
            validation_method = CAPTCHA_VALIDATION_METHOD if is_captcha_required_and_not_passed else validation_method

            # Если подозреваем робота и настройки позволяют, потребуем еще проверку по sms
            if self.is_sms_validation_required:
                validation_method = CAPTCHA_AND_PHONE_VALIDATION_METHOD

        return validation_method

    @property
    def is_sms_validation_required(self):
        """Нужно ли затребовать проверку пользователя с помощью sms.

        Переопределяется в наследниках.
        """
        raise NotImplementedError()  # pragma: no cover
