# -*- coding: utf-8 -*-

import base64
from collections import namedtuple
from datetime import (
    datetime,
    timedelta,
)
import hashlib
import logging

from passport.backend.api.common.common import get_email_domain
from passport.backend.api.common.octopus import (
    inc_calls_counter,
    inc_sessions_created,
    KOLMOGOR_COUNTER_CALLS_FAILED,
    KOLMOGOR_COUNTER_CALLS_SHUT_DOWN_FLAG,
    KOLMOGOR_COUNTER_SESSIONS_CREATED,
    octopus_log_status,
)
from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.api.common.yasms import (
    generate_fake_global_sms_id,
    log_sms_not_delivered,
)
from passport.backend.api.views.bundle.exceptions import (
    CaptchaRequiredError,
    CodeInvalidError,
    YaSmsUnavailableError,
)
from passport.backend.api.views.bundle.phone import (
    exceptions,
    helpers as phone_helpers,
)
from passport.backend.api.views.bundle.phone.exceptions import (
    ConfirmationsLimitExceededError,
    PhoneNotConfirmedError,
)
from passport.backend.api.yasms import exceptions as yasms_exceptions
from passport.backend.api.yasms.api import (
    BindSecurestPossiblePhone,
    ReplaceSecurePhone,
    SaveSecurePhone,
    SaveSimplePhone,
)
from passport.backend.api.yasms.utils import (
    build_send_confirmation_code,
    Restricter,
)
from passport.backend.core.builders.kolmogor import (
    BaseKolmogorError,
    get_kolmogor,
)
from passport.backend.core.builders.octopus import get_octopus
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    calls_per_ip,
    calls_per_phone,
)
from passport.backend.core.logging_utils.loggers import GraphiteLogger
from passport.backend.core.models.phones.phones import TrackedConfirmationInfo
from passport.backend.core.types.email.email import build_emails
from passport.backend.core.types.phone_number.phone_number import (
    get_alt_phone_numbers_of_phone_number,
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_phone
from passport.backend.core.yasms.notifications import notify_user_by_sms
from passport.backend.core.yasms.phonenumber_alias import (
    Aliasification,
    PhoneAliasManager,
)
from passport.backend.utils.common import (
    ClassMapping,
    normalize_code,
)
from passport.backend.utils.time import datetime_to_integer_unixtime


PhoneValidForCallStatus = namedtuple('PhoneValidForCallStatus', ['valid_for_call', 'valid_for_flash_call'])


PHONE_ALIAS_REGISTER_TEMPLATE = 'mail/phone_alias_register.html'
YASMS_EXCEPTIONS_MAPPING = ClassMapping([
    (yasms_exceptions.YaSmsLimitExceeded, exceptions.SendingLimitExceededError),
    (yasms_exceptions.YaSmsUidLimitExceeded, exceptions.SendingLimitExceededError),
    (yasms_exceptions.YaSmsCodeLimitError, exceptions.SendingLimitExceededError),
    (yasms_exceptions.YaSmsPermanentBlock, exceptions.PhoneBlockedError),
    (yasms_exceptions.YaSmsTemporaryBlock, exceptions.SendingLimitExceededError),
    (yasms_exceptions.YaSmsIpLimitExceeded, exceptions.SendingLimitExceededError),
    (yasms_exceptions.YaSmsPhoneNumberValueError, exceptions.PhoneInvalidError),
    (yasms_exceptions.YaSmsDeliveryError, YaSmsUnavailableError),
    (yasms_exceptions.YaSmsAlreadyVerified, exceptions.PhoneConfirmedError),
    (yasms_exceptions.YaSmsCaptchaRequiredError, CaptchaRequiredError),
])

TEST_OCTOPUS_CONSUMERS = {'kopusha', 'kopalka', 'passport', 'mobileproxy'}


log = logging.getLogger('passport.api.view.mixins.phone')


SecureNumberStatus = namedtuple(
    'SecureNumberStatus',
    ['Invalid', 'Valid', 'Missing'],
)._make([0, 1, 2])


def hash_android_package(package_name, package_public_key):
    """
    Приводит пару имя пакета + публичный ключ пакета в хеш, необходимый для работы Sms Retriever в Андроиде.
    https://developers.google.com/identity/sms-retriever/verify#computing_your_apps_hash_string
    """
    sha_hash = hashlib.sha256('{} {}'.format(
        package_name.strip(),
        package_public_key.strip().lower(),
    )).digest()
    # 9 символов достаточно, потому что длина base64-кодировки предсказывается по формуле:
    # 4 * (n / 3) + округление вверх, т.е. 12 символов. А отдать нам надо 11.
    return base64.b64encode(sha_hash[:9])[:11]


def format_for_android_sms_retriever(sms_text, android_package_hash):
    """
    Форматирует СМС в необходимый вид для работы Sms Retriever в Андроиде.
    https://developers.google.com/identity/sms-retriever/verify#1_construct_a_verification_message
    """
    return sms_text.format(
        PREFIX=u'<#>',
        # Замена code на %s из-за следующих нюансов:
        # 1. format не умеет форматировать только _часть_ шаблона, без переопределения Formatter, а кода здесь ещё нет
        # 2. другой шаблон для смс (SMS.APPROVE_CODE) использует %s
        # 3. даже если там заменить %s на {code}, здесь всё равно надо будет писать code="{code}" из-за пункта 1
        # Т.е. без более серьёзного рефакторинга избавиться от этой подстановки не получится.
        code='{{code}}',
        hash=android_package_hash,
    )


class YasmsPrivateMixin(object):
    def log_sms_not_delivered(self, reason, number, global_sms_id, caller):
        log_sms_not_delivered(
            reason=reason,
            number=number,
            global_sms_id=global_sms_id,
            caller=caller,
            client_ip=str(self.client_ip),
            user_agent=self.user_agent,
        )


class SmsRetrieverMixin(YasmsPrivateMixin):
    @staticmethod
    def _can_format_for_sms_retriever(gps_package_name):
        if gps_package_name is None:
            return False

        for prefix in settings.ANDROID_PACKAGE_PREFIXES_WHITELIST:
            if gps_package_name == prefix or gps_package_name.startswith(prefix + '.'):
                return True
        return False

    def _build_sms_template(self, language, gps_package_name=None):
        gps_package_name = gps_package_name or self.form_values.get('gps_package_name')
        if gps_package_name is None:
            return settings.translations.SMS[language]['APPROVE_CODE']
        if not self._can_format_for_sms_retriever(gps_package_name):
            log.warning('Cannot send SMS for SmsRetriever for package {}'.format(repr(gps_package_name)))
            return settings.translations.SMS[language]['APPROVE_CODE']

        package_public_key = settings.ANDROID_PACKAGE_PUBLIC_KEY_DEFAULT
        for prefix, key in settings.ANDROID_PACKAGE_PREFIX_TO_KEY.items():
            if gps_package_name == prefix or gps_package_name.startswith(prefix + '.'):
                package_public_key = key
                break

        gps_hash = hash_android_package(
            gps_package_name,
            package_public_key,
        )
        sms_template = format_for_android_sms_retriever(
            settings.translations.SMS[language]['APPROVE_CODE.GPS'],
            gps_hash,
        )
        return sms_template

    def _bind_sms_retriever_data_to_phone_confirmator(self, confirmator, gps_package_name):
        if gps_package_name is None:
            return

        confirmator.gps_package_name = gps_package_name
        confirmator.sms_retriever = self._can_format_for_sms_retriever(gps_package_name)


class BundleTrackedPhoneConfirmationMixin(SmsRetrieverMixin):
    """
    Общий код, используемый для отправки кода и подтверждения номера с хранением состояния в треке

    Методы send_code, confirm_code вызывать в транзакции трека, обязательно commit_on_error.

    После выполнения confirm_code следует сериализовать изменения в аккаунте,
    если подтверждается один из номеров привязанных к аккаунту.

    Для записи в Статбокс сохраненных в stash данных нужно вызвать statbox.dump_stashes().

    При использовании рестриктера не по умолчанию, следует наследовать класс от TrackRestricter.

    При возможности изменения номера, на который отправляется СМС, важно сбрасывать состояние (в транзакции трека):
    >>> if is_phone_changed:
    >>>     self.confirmation_info.reset_phone()
    >>>     self.confirmation_info.save()
    >>>     ...
    >>> self.send_code(phone)
    """

    @cached_property
    def confirmation_info(self):
        return TrackedConfirmationInfo(self.track)

    def send_code(self, phone, restricter=None, language=None, account=None, gps_package_name=None,
                  force_new_code=False, code_format=None):
        if not restricter:
            restricter = TrackRestricter(
                confirmation_info=self.confirmation_info,
                env=self.request.env,
                statbox=self.statbox,
                consumer=self.consumer,
                phone_number=phone,
                track=self.track,
            )
            restricter.restrict_ip()
            restricter.restrict_reconfirmation()
            restricter.restrict_rate()
        elif not isinstance(restricter, TrackRestricter):  # pragma: no cover
            raise TypeError('Incorrect restricter type')
        self._restricter = restricter

        send_code = build_send_confirmation_code(
            account=account,
            can_format_for_sms_retriever=self._can_format_for_sms_retriever(gps_package_name),
            confirmation_info=self.confirmation_info,
            consumer=self.consumer,
            env=self.request.env,
            force_new_code=force_new_code,
            gps_package_name=gps_package_name,
            language=language,
            phone_number=phone,
            restricter=self._restricter,
            sms_template_builder=lambda language: self._build_sms_template(language, gps_package_name),
            statbox=self.statbox,
            yasms_builder=self.yasms,
            code_format=code_format,
        )
        try:
            send_code()
            self.confirmation_info.save()
        except Exception as e:
            if e.__class__ not in YASMS_EXCEPTIONS_MAPPING:
                raise
            exc_cls = YASMS_EXCEPTIONS_MAPPING[e.__class__]
            raise exc_cls()

    @property
    def confirmation_code_resend_timeout(self):
        return self._restricter.resend_timeout

    def update_phone_confirmation_time(self, account=None):
        account = account or getattr(self, 'account', None)
        if account:
            phone = account.phones.by_number(self._get_number())
            if phone:
                phone.confirm()
        self._confirmation_to_statbox(account)

    def confirm_code(self, code, account=None):
        checks_count = self.track.phone_confirmation_confirms_count.get(default=0)
        if checks_count >= settings.SMS_VALIDATION_MAX_CHECKS_COUNT:
            global_sms_id = generate_fake_global_sms_id()
            self.log_sms_not_delivered(
                reason='limits',
                number=self._get_number(),
                global_sms_id=global_sms_id,
                caller=self.consumer,
            )
            raise ConfirmationsLimitExceededError()

        self.track.phone_confirmation_confirms_count.incr()
        if normalize_code(self.track.phone_confirmation_code) != normalize_code(code):
            raise CodeInvalidError()

        self.track.phone_confirmation_is_confirmed = True

        self.update_phone_confirmation_time()

    def _confirmation_to_statbox(self, account):
        uid = account and account.uid or None
        login = account and account.login or None

        phone_id = None
        if account:
            phone = account.phones.by_number(self._get_number())
            if phone:
                phone_id = phone.id

        self.statbox.log(
            action='phone_confirmed',
            number=self._get_number().masked_format_for_statbox,
            confirmation_time=datetime.now(),
            code_checks_count=self.track.phone_confirmation_confirms_count.get(default=0),
            uid=uid,
            login=login,
            phone_id=phone_id,
        )

    def _get_number(self):
        original_number = self.track.phone_confirmation_phone_number or self.track.phone_confirmation_phone_number_original
        try:
            number = PhoneNumber.parse(original_number)
        except InvalidPhoneNumber:
            if not self.track.country:
                raise
            number = PhoneNumber.parse(original_number, self.track.country)
        return number


class TrackRestricter(Restricter):
    def update(self):
        super(TrackRestricter, self).update()
        if not self._confirmation_info.code_first_sent:
            self._confirmation_info.code_first_sent = datetime.now()


class RegisterPhoneAliasManager(PhoneAliasManager):
    def send_mail_about_alias_as_login_and_email_enabled(self, account, language, phone_number):
        # При регистрации отправляем письмо с особой регистрационной версткой
        mail_domain = ''
        if account.emails.default and account.emails.default.is_native:
            mail_domain = account.emails.default.domain

        translations = settings.translations.NOTIFICATIONS[language]

        context = {
            'phone_alias_notice_key': '',
            'phone_alias_explanation_key': 'phone_alias.on.explanation.no_mailbox',
            'HELP_URL': translations['phone_alias.help_url'],
            'YASMS_VALIDATOR_URL': translations['validator_url'],
            'logo_url_key': 'logo_url.mail',
            'login_alias': phone_number.digital,
            'domain': mail_domain,
            'LOGIN_ALIAS_VALUE': phone_number.digital,
            'FORMATTED_PHONE_NUMBER': phone_number.international,
        }

        self._send_mail_notifications(
            account=account,
            language=language,
            sender='email_sender_display_name.mail',
            subject_key='digitreg.subject',
            template=PHONE_ALIAS_REGISTER_TEMPLATE,
            context=context,
        )


class SecurePhoneBindAndAliasifyMixin(object):
    def build_emails(self, number, now):
        # выставляем единственный алиасный (7916...@domain) email в список emails на только что созданный аккаунт
        address = '%s@%s' % (number.digital, get_email_domain(self.host))
        return build_emails([{
            'native': True,
            'validated': True,
            'rpop': False,
            'unsafe': False,
            'default': True,
            'silent': False,
            'born-date': now.strftime('%Y-%m-%d %H:%M:%S'),
            'address': address,
        }])


class OctopusAvailabilityMixin(object):
    @cached_property
    def kolmogor(self):
        return get_kolmogor()

    @cached_property
    def graphite(self):
        return GraphiteLogger(service='octopus-calls')

    def are_calls_available(self):
        try:
            calls_shut_down = self.are_calls_shut_down()
            if calls_shut_down:
                octopus_log_status('calls_shut_down')
                return False
            counters = self.get_all_counters()
        except BaseKolmogorError as e:
            log.warning('Request to Kolmogor failed while checking calls counters; %s', e)
            return True

        sessions_count = counters[KOLMOGOR_COUNTER_SESSIONS_CREATED]
        if sessions_count < settings.OCTOPUS_COUNTERS_MIN_COUNT:
            return True

        # Как хорошо работают шлюзы
        calls_failed = counters[KOLMOGOR_COUNTER_CALLS_FAILED]
        if float(calls_failed) / sessions_count >= settings.OCTOPUS_GATES_WORKING_THRESHOLD:
            self.shut_down_calls()
            log.warning('Shutting down calls on fails limit: failed=%s/all=%s', calls_failed, sessions_count)
            return False

        return True

    def shut_down_calls(self):
        octopus_log_status('calls_shut_down')
        try:
            self.kolmogor.inc(
                space=settings.KOLMOGOR_KEYSPACE_OCTOPUS_CALLS_FLAG,
                keys=[KOLMOGOR_COUNTER_CALLS_SHUT_DOWN_FLAG],
            )
        except BaseKolmogorError:
            log.warning('Request to Kolmogor failed while shuting down calls')

    def inc_sessions_created(self):
        inc_sessions_created(self.kolmogor)

    def inc_calls_counter(self, call_counter_key):
        inc_calls_counter(self.kolmogor, call_counter_key)

    def are_calls_shut_down(self):
        counter = self.kolmogor.get(
            space=settings.KOLMOGOR_KEYSPACE_OCTOPUS_CALLS_FLAG,
            keys=[KOLMOGOR_COUNTER_CALLS_SHUT_DOWN_FLAG],
        )
        return bool(counter[KOLMOGOR_COUNTER_CALLS_SHUT_DOWN_FLAG])

    def get_all_counters(self):
        return self.kolmogor.get(
            space=settings.KOLMOGOR_KEYSPACE_OCTOPUS_CALLS_COUNTERS,
            keys=[
                KOLMOGOR_COUNTER_SESSIONS_CREATED,
                KOLMOGOR_COUNTER_CALLS_FAILED,
            ],
        )

    def check_valid_for_call(self, consumer, phone):
        not_valid_for_calls = PhoneValidForCallStatus(valid_for_call=False, valid_for_flash_call=False)

        if settings.PHONE_CONFIRMATION_CALL_ENABLED:

            if bool(phone.country) and phone.number_type == 'mobile':
                phone_country = phone.country.lower()

                valid_for_call = (
                    phone_country in settings.PHONE_CONFIRMATION_CALL_COUNTRIES or
                    is_experiment_enabled_by_phone(
                        phone,
                        settings.PHONE_CONFIRMATION_CALL_COUNTRIES_WITH_DENOMINATOR.get(phone_country, 0),
                    )
                )
                valid_for_flash_call = (
                    phone_country in settings.PHONE_CONFIRMATION_FLASH_CALL_COUNTRIES or
                    is_experiment_enabled_by_phone(
                        phone,
                        settings.PHONE_CONFIRMATION_FLASH_CALL_COUNTRIES_WITH_DENOMINATOR.get(phone_country, 0),
                    )
                )
            elif phone.e164.startswith(settings.TEST_VALID_PHONE_NUMBER_PREFIX) and consumer in TEST_OCTOPUS_CONSUMERS:
                valid_for_call = True
                valid_for_flash_call = True
            else:
                return not_valid_for_calls

            blacklisted = any([phone.digital.startswith(prefix) for prefix in settings.PHONE_DIGITAL_PREFIXES_BLACKLIST])

            if not valid_for_call and not valid_for_flash_call:
                # можно сэкономить на проверке счётчиков
                return not_valid_for_calls

            calls_available = self.are_calls_available()

            if calls_available and not blacklisted and self.check_counters_ok(phone):
                valid_for_calls = PhoneValidForCallStatus(
                    valid_for_call=valid_for_call,
                    valid_for_flash_call=valid_for_flash_call,
                )
                if self.track and self.track.antifraud_tags:
                    return PhoneValidForCallStatus(
                        valid_for_call=valid_for_calls.valid_for_call and bool(set(self.track.antifraud_tags) & {'call', 'dictation'}),
                        valid_for_flash_call=valid_for_calls.valid_for_flash_call and ('flash_call' in self.track.antifraud_tags),
                    )
                return valid_for_calls
        return not_valid_for_calls

    def check_counters_ok(self, phone):
        if self.track:
            if (
                self.track.phone_confirmation_phone_number and
                self.track.phone_confirmation_phone_number == phone.e164 and
                (
                    self.track.phone_confirmation_calls_ip_limit_reached or
                    self.track.phone_confirmation_calls_count_limit_reached or
                    self.track.phone_confirmation_calls_count.get(default=0) >= settings.PHONE_VALIDATION_MAX_CALLS_COUNT
                )
            ):
                return False

        phone_counter = calls_per_phone.get_counter()
        if phone_counter.hit_limit(phone.digital):
            self.statbox.bind(error='calls_limit.exceeded')
            log.debug('Unable to call because of phone number limit')
            return False

        ip_counter = calls_per_ip.get_counter(self.client_ip)
        if ip_counter.hit_limit(self.client_ip):
            self.statbox.bind(error='calls_limit.exceeded')
            log.debug('Unable to call because of ip limit')
            return False

        return True


class BundleReplaceSecurePhone(ReplaceSecurePhone):
    def __init__(self, track=None, *args, **kwargs):
        super(BundleReplaceSecurePhone, self).__init__(*args, **kwargs)
        self._track = track

    def _build_send_confirmation_code(self, confirmation_info, phone_number):
        return build_send_confirmation_code(
            phone_number,
            confirmation_info,
            self._account,
            self._env,
            self._yasms_builder,
            self._consumer,
            self._statbox,
            self._language,
            restricter=self._build_restricter(confirmation_info, phone_number),
        )

    def _build_restricter(self, confirmation_info, phone_number):
        restricter = Restricter(
            confirmation_info=confirmation_info,
            consumer=self._consumer,
            env=self._env,
            phone_number=phone_number,
            statbox=self._statbox,
            track=self._track,
        )
        restricter.restrict_ip()
        restricter.restrict_rate()
        restricter.restrict_reconfirmation()
        return restricter


class BundleBindSecurestPossiblePhone(BindSecurestPossiblePhone):
    def __init__(self, track=None, *args, **kwargs):
        super(BundleBindSecurestPossiblePhone, self).__init__(*args, **kwargs)
        self._track = track

    def _build_replace_secure_phone(self):
        return BundleReplaceSecurePhone(
            account=self._account,
            blackbox=self._blackbox,
            consumer=self._consumer,
            does_user_admit_secure_number=False,
            env=self._env,
            is_password_verified=True,
            is_simple_phone_confirmed=True,
            language=self._language,
            phone_number=self._phone_number,
            statbox=self._statbox,
            track=self._track,
            yasms=self._yasms_api,
            yasms_builder=self._yasms_builder,
        )


class BundlePhoneMixin(object):
    """
    Операции с телефонами пользователя
    """
    @property
    def secure_number(self):
        """Получаем защищённый номер телефона пользователя."""
        if self.secure_phone:
            return self.secure_phone.number

    @property
    def secure_phone(self):
        """
        Получаем объект types.phone.Phone защищённого номера телефона
        пользователя.
        """
        if self.account_phones.secure and self.account_phones.secure.is_confirmed:
            return self.account_phones.secure

    @cached_property
    def _get_secure_number_status(self):
        """
        Выясняем есть ли у пользователя защищённый номер телефона и
        является ли он валидным с точки зрения библиотеки python-phonenumbers.
        """
        try:
            if self.secure_number:
                return SecureNumberStatus.Valid
            return SecureNumberStatus.Missing
        except InvalidPhoneNumber:
            self.statbox.log(
                uid=self.account.uid,
                action='loaded_secure_number',
                error='invalid_phone_number',
            )
            return SecureNumberStatus.Invalid

    @property
    def is_secure_number_valid_or_missing(self):
        """
        Выясняем, является ли защищённый номер (при его наличии) валидным
        с точки зрения библиотеки python-phonenumbers.
        """
        return self._get_secure_number_status != SecureNumberStatus.Invalid

    @property
    def has_secure_number(self):
        """Выясняем есть ли у пользователя защищённый номер."""
        return self._get_secure_number_status == SecureNumberStatus.Valid

    @property
    def secure_number_or_none(self):
        """
        Получаем защищённый номер.
        Не учитываем ошибки валидации телефонных номеров библиотекой python-phonenumbers -
        считаем что у пользователя нет защищённого номера в случае возникновения ошибок.
        """
        if self.has_secure_number:
            return self.secure_number

    @cached_property
    def account_phones(self):
        """
        Возвращает список подтверждённых телефонов аккаунта объектом Phones.
        """
        return phone_helpers.load_phones(self.yasms_api, self.account)

    def save_secure_number_in_track(self):
        """Сохраняет в трэк защищённый номер для валидации пароля при смене пароля"""
        if self.has_secure_number:
            self.track.can_use_secure_number_for_password_validation = True
            self.track.secure_phone_number = self.secure_number.e164
        else:
            self.track.can_use_secure_number_for_password_validation = False

    def is_phone_confirmed_in_track(
        self,
        allow_by_sms=True,
        allow_by_call=True,
        allow_by_flash_call=False,
        track=None,
    ):
        track = track or self.track
        phone_confirmation_method = track.phone_confirmation_method or CONFIRM_METHOD_BY_SMS
        return bool(
            track.phone_confirmation_is_confirmed and (
                (allow_by_sms and phone_confirmation_method == CONFIRM_METHOD_BY_SMS) or
                (allow_by_call and phone_confirmation_method == CONFIRM_METHOD_BY_CALL) or
                (allow_by_flash_call and phone_confirmation_method == CONFIRM_METHOD_BY_FLASH_CALL)
            )
        )

    def is_secure_phone_confirmed_in_track(self, allow_by_sms=True, allow_by_call=True, allow_by_flash_call=False):
        if not self.account.phones.secure:
            return False

        all_secure_phone_numbers = [self.account.phones.secure.number]
        all_secure_phone_numbers.extend(get_alt_phone_numbers_of_phone_number(all_secure_phone_numbers[0]))

        return (
            self.is_phone_confirmed_in_track(
                allow_by_sms=allow_by_sms,
                allow_by_call=allow_by_call,
                allow_by_flash_call=allow_by_flash_call,
            ) and
            self.track.phone_confirmation_phone_number in {p.e164 for p in all_secure_phone_numbers}
        )

    def is_bank_phone_confirmed_in_track(self, allow_by_sms=True, allow_by_call=True, allow_by_flash_call=False):
        if not self.account.bank_phonenumber_alias:
            return False

        bank_phone = self.account.phones.by_number(self.account.bank_phonenumber_alias.alias)
        return (
            self.is_phone_confirmed_in_track(
                allow_by_sms=allow_by_sms,
                allow_by_call=allow_by_call,
                allow_by_flash_call=allow_by_flash_call,
            ) and
            self.track.phone_confirmation_phone_number == bank_phone.number.e164
        )

    def send_sms(self, number, sms_text, identity, ignore_errors=True):
        try:
            notify_user_by_sms(
                phone_number=number,
                message=sms_text,
                uid=self.account.uid,
                yasms_builder=self.yasms,
                statbox=self.statbox,
                consumer=self.consumer,
                identity=identity,
                ignore_errors=ignore_errors,
                client_ip=self.client_ip,
                user_agent=self.user_agent,
            )
        except yasms_exceptions.YaSmsError as exc:
            # Обработаем ошибку, если нас попросили.
            if not ignore_errors and exc.__class__ in YASMS_EXCEPTIONS_MAPPING:
                other_exc = YASMS_EXCEPTIONS_MAPPING[exc.__class__]
                raise other_exc()
            raise

    def assert_secure_phone_confirmed_with_track(self):
        if not self.secure_phone or not self.is_secure_phone_confirmed_with_track():
            raise PhoneNotConfirmedError()

    def is_secure_phone_confirmed_with_track(self):
        """
        Проверяем что телефон(secure_phone) аккаунта либо телефон в треке был подтвержден
        Так же что телефон в треке совпадает с телефоном аккаунта
        Кейс - любое место где нужно решить нужен ли 'челендж'
        """
        recently_confirmed = self.is_secure_phone_confirmed_recently()

        if recently_confirmed:
            return True

        if not (
            self.track.phone_confirmation_phone_number and
            self.is_phone_confirmed_in_track(allow_by_flash_call=True)
        ):
            return False

        if self.track.phone_confirmation_phone_number != self.secure_phone.number.e164:
            return False

        return True

    def is_secure_phone_confirmed_recently(self):
        """
        Проверяем что телефон(secure_phone) аккаунта был подтвержден не более 3 часов назад
        """
        phone = self.account.phones.secure
        if phone is None:
            return False
        return datetime.now() - timedelta(hours=3) <= phone.confirmed


    @cached_property
    def octopus_api(self):
        return get_octopus()

    def build_save_secure_phone(self, **kwargs):
        kwargs.setdefault('account', self.account)
        return SaveSecurePhone(
            consumer=self.consumer,
            env=self.request.env,
            statbox=self.statbox,
            blackbox=self.blackbox,
            yasms=self.yasms_api,
            **kwargs
        )

    def build_save_simple_phone(self, **kwargs):
        kwargs.setdefault('account', self.account)
        return SaveSimplePhone(
            consumer=self.consumer,
            env=self.request.env,
            statbox=self.statbox,
            blackbox=self.blackbox,
            yasms=self.yasms_api,
            **kwargs
        )

    def build_phonenumber_aliasification(self, **kwargs):
        return Aliasification(
            account=self.account,
            consumer=self.consumer,
            blackbox=self.blackbox,
            statbox=self.statbox,
            **kwargs
        )

    def securest_possible_phone_binding_response(self, binder):
        """
        Формирует ответ с результатом работы BindSecurestPossiblePhone
        """
        if not isinstance(binder, BindSecurestPossiblePhone):
            raise NotImplementedError('Binder should be instance of BindSecurestPossiblePhone')

        response = dict(
            is_bound_new_secure_phone=False,
            is_conflicted_operation_exists=False,
            is_replaced_phone_with_quarantine=False,
            is_updated_current_secure_phone=False,
        )

        if binder.is_binding_secure():
            response.update(is_bound_new_secure_phone=True)

        elif binder.is_replacing() or binder.is_starting_replacement():
            response.update(is_replaced_phone_with_quarantine=True)

            quarantine_finished_at = binder.quarantine_finished_at()
            if quarantine_finished_at:
                response.update(secure_phone_pending_until=datetime_to_integer_unixtime(quarantine_finished_at))

        elif binder.is_updating_old_secure():
            response.update(is_updated_current_secure_phone=True)

        elif binder.is_operation_prevent_to_secure_phone():
            response.update(is_conflicted_operation_exists=True)

        return response

    def build_calling_phone_confirmator(self, **kwargs):
        defaults = dict(
            caller=self.consumer,
            calls_limit=settings.PHONE_VALIDATION_MAX_CALLS_COUNT,
            client=self.octopus_api,
            code_length=settings.PHONE_VALIDATION_CODE_LENGTH,
            env=self.request.env,
            statbox=self.statbox,
            track=self.track,
            account=self.account,
        )
        for key in defaults:
            kwargs.setdefault(key, defaults[key])
        return phone_helpers.CallingPhoneConfirmator(**kwargs)

    def build_sending_phone_confirmator(self, **kwargs):
        defaults = dict(
            caller=self.consumer,
            client=self.yasms,
            env=self.request.env,
            sent_sms_limit=settings.SMS_VALIDATION_MAX_SMS_COUNT,
            sms_code_length=settings.SMS_VALIDATION_CODE_LENGTH,
            sms_resend_timeout=settings.SMS_VALIDATION_RESEND_TIMEOUT,
            statbox=self.statbox,
            track=self.track,
            user_agent=self.user_agent,
            account=self.account,
        )
        for key in defaults:
            kwargs.setdefault(key, defaults[key])
        return phone_helpers.SendingPhoneConfirmator(**kwargs)

    def build_flash_calling_phone_confirmator(self, **kwargs):
        defaults = dict(
            caller=self.consumer,
            calls_limit=settings.PHONE_VALIDATION_MAX_CALLS_COUNT,
            client=self.octopus_api,
            env=self.request.env,
            statbox=self.statbox,
            track=self.track,
            account=self.account,
        )
        for key in defaults:
            kwargs.setdefault(key, defaults[key])
        return phone_helpers.FlashCallingPhoneConfirmator(**kwargs)

    def build_bind_securest_possible_phone(self, **kwargs):
        defaults = dict(
            consumer=self.consumer,
            env=self.request.env,
            statbox=self.statbox,
            blackbox=self.blackbox,
            yasms_api=self.yasms_api,
            yasms_builder=self.yasms,
            view=self,
        )
        for key in defaults:
            kwargs.setdefault(key, defaults[key])
        return BundleBindSecurestPossiblePhone(**kwargs)

    def build_replace_secure_phone(self, **kwargs):
        defaults = dict(
            blackbox=self.blackbox,
            consumer=self.consumer,
            env=self.request.env,
            statbox=self.statbox,
            track=self.track,
            view=self,
        )
        for key in defaults:
            kwargs.setdefault(key, defaults[key])
        return BundleReplaceSecurePhone(**kwargs)

    def build_phone_confirmation_restricter(self, confirmation_info, phone_number, login_id=None):
        restricter = Restricter(
            confirmation_info=confirmation_info,
            consumer=self.consumer,
            env=self.request.env,
            phone_number=phone_number,
            statbox=self.statbox,
            track=self.track,
            login_id=login_id,
        )
        restricter.restrict_ip()
        restricter.restrict_rate()
        restricter.restrict_reconfirmation()
        return restricter
