# -*- coding: utf-8 -*-
import json
import logging
import time

from passport.backend.api.common import extract_tld
from passport.backend.api.common.logbroker import get_api_logbroker_writer
from passport.backend.api.views.bundle.exceptions import (
    OAuthTokenDeviceNotTrustedError,
    RateLimitExceedError,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.core.am_pushes.push_request import (
    AppTargetingTypes,
    PushRequest,
    PushRequestRecipient,
    SubscriptionSources,
)
from passport.backend.core.conf import settings
from passport.backend.core.geobase import Region
from passport.backend.core.logbroker.exceptions import BaseLogbrokerError
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.protobuf.challenge_pushes.challenge_pushes_pb2 import ChallengePushRequest
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid
from passport.backend.utils.common import MixinDependencyChecker
from passport.backend.utils.string import smart_bytes
from six.moves.urllib.parse import quote_plus


log = logging.getLogger('passport.api.view.bundle.mixins')
TOKEN_ATTRIBUTE_DEVICE_TRUSTED = 'device_trusted'
MOCK_TRACK_ID = '11111111111111111111111111111111'


class BundlePushMixin(object):
    """ Методы, связанные с пуш-сообщениями """
    __metaclass__ = MixinDependencyChecker

    mixin_dependencies = [
        KolmogorMixin,
    ]

    def _assert_counters(self, key_template, key_space):
        key = key_template % self.account.uid
        counter = self.build_counter(key_space, key, settings.COUNTERS[key_template])
        if not self.failsafe_check_kolmogor_counters([counter]):
            return False
        return self.failsafe_inc_kolmogor_counters([counter])

    def assert_push_2fa_counters(self):
        return self._assert_counters(
            settings.PUSH_2FA_COUNTER,
            settings.PUSH_2FA_KOLMOGOR_KEY_SPACE,
        )

    def assert_auth_push_counters(self):
        return self._assert_counters(
            settings.AUTH_PUSH_COUNTER,
            settings.AUTH_PUSH_KOLMOGOR_KEY_SPACE,
        )

    def is_device_trusted_by_oauth_token(self):
        return bool(self.oauth_info.get(TOKEN_ATTRIBUTE_DEVICE_TRUSTED))

    def assert_device_trusted_by_oauth_token(self):
        if not self.is_device_trusted_by_oauth_token():
            log.debug('Device bound to token is not trusted')
            raise OAuthTokenDeviceNotTrustedError()

    def user_has_push_2fa_subscriptions(self):
        """ Есть ли у пользователя подписки приложений, совместимых с 2fa """
        # Здесь пока заглушка, в дальнейшем будет определение через наличие
        # доверенных x-token
        return True

    def _send_push_message(self, request):
        get_api_logbroker_writer('challenge_pushes').send(
            ChallengePushRequest(push_message_request=request.to_proto()),
        )

    def send_push_2fa(self, context=None):
        """ Отправить 2fa пуш пользователю """
        language = get_preferred_language(self.account)
        translations = settings.translations.NOTIFICATIONS[language]
        tld = extract_tld(self.host, settings.PASSPORT_TLDS)
        request = PushRequest(
            push_service='2fa',
            event_name='2fa_code',
            recipients=PushRequestRecipient(
                uid=self.account.uid,
                app_targeting_type=AppTargetingTypes.ONE_APP_PER_DEVICE,
                required_am_capabilities=['push:passport_protocol'],
                require_trusted_device=True,
                context=context,
            ),
            title=smart_bytes(translations['2fa.push.title']),
            body=smart_bytes(translations['2fa.push.text']),
            webview_url=settings.PUSH_2FA_CODE_URL_TEMPLATE % dict(
                tld=tld,
                track_id=self.track.track_id,
            ),
        )
        self._send_push_message(request)
        log.debug('Sent 2fa push for uid {}'.format(self.account.uid))

    def send_push_with_pictures(self, pictures, expires_at):
        request = PushRequest(
            push_service='2fa',
            event_name='2fa_pictures',
            recipients=PushRequestRecipient(
                uid=self.account.uid,
                app_targeting_type=AppTargetingTypes.CLIENT_DECIDES,
                subscription_source=SubscriptionSources.YAKEY,
                device_ids=self.account.totp_secret.yakey_device_ids,
            ),
            extra_data={
                'track_id': self.track.track_id,
                'uid': self.track.uid,
                '2fa_pictures': pictures,
                '2fa_pictures_expire_at': expires_at,
                'user_ip': str(self.request.env.user_ip),
                'user_agent': self.request.env.user_agent,
            },
        )
        self._send_push_message(request)
        log.debug('Sent 2fa pictures push for uid {}'.format(self.account.uid))

    def _check_account_modification_push_counters(self, event_name):
        key_template = settings.ACCOUNT_MODIFICATION_PER_UID_COUNTER_TEMPLATE
        sub_key = 'push:{}'.format(event_name)
        key = key_template % (sub_key, self.account.uid)
        counter = self.build_counter(
            settings.ACCOUNT_MODIFICATION_KOLMOGOR_KEY_SPACE,
            key,
            settings.COUNTERS[key_template][sub_key],
        )
        try:
            self.failsafe_check_kolmogor_counters([counter])
            self.failsafe_inc_kolmogor_counters([counter])
            return True
        except RateLimitExceedError:
            return False

    def _resolve_user_login(self):
        if (
            self.account.is_mailish or
            self.account.is_phonish or
            self.account.is_neophonish or
            self.account.is_social
        ):
            return self.account.person.display_name.public_name
        else:
            return self.account.login

    def send_account_modification_push(
        self,
        event_name,
        context=None,
    ):
        """ Отправить пуш об изменении настроек аккаунта """
        if not (
            event_name in settings.ACCOUNT_MODIFICATION_PUSH_ENABLE and
            (
                is_experiment_enabled_by_uid(self.account.uid, settings.ACCOUNT_MODIFICATION_NOTIFY_DENOMINATOR) or
                int(self.account.uid) in settings.ACCOUNT_MODIFICATION_NOTIFY_WHITELIST
            )
        ):
            return
        login = self._resolve_user_login()
        if context is not None:
            context = json.dumps(context)
        if not self._check_account_modification_push_counters(event_name):
            log.debug('No account modification push event={}: counters overflow'.format(event_name))
            return
        language = get_preferred_language(self.account)
        translations = settings.translations.NOTIFICATIONS[language]
        tld = extract_tld(self.host, settings.PASSPORT_TLDS)

        region = Region(self.client_ip)
        lon = region.lon or 0.0
        lat = region.lat or 0.0
        linguistics = region.linguistics(language)
        if linguistics is not None:
            region_name = linguistics.nominative
        else:
            region_name = ''

        map_url = settings.PUSH_ACCOUNT_MODIFICATION_MAP_URL_TEMPLATE % dict(
            tld=extract_tld(self.host, settings.PASSPORT_TLDS),
            lon=lon,
            lat=lat,
            lang=language,
        )
        webview_url = settings.PUSH_ACCOUNT_MODIFICATION_URL_TEMPLATE % dict(
            tld=tld,
            uid=self.account.uid,
            timestamp=int(time.time()),
            track_id=quote_plus(self.track_id or MOCK_TRACK_ID),
            ip=quote_plus(str(self.client_ip)),
            location=quote_plus(region_name),
            map_url=quote_plus(map_url),
            title_key=event_name,
        )
        translation_key = 'account_modification.{}.push'.format(event_name)
        title = translations['{}.title'.format(translation_key)].format(login=login)
        body = translations['{}.body'.format(translation_key)].format(login=login)

        request = PushRequest(
            push_service='account_modification',
            event_name=event_name,
            recipients=PushRequestRecipient(
                uid=int(self.account.uid),
                app_targeting_type=AppTargetingTypes.ONE_APP_PER_DEVICE,
                required_am_capabilities=['push:passport_protocol'],
                context=context,
            ),
            title=smart_bytes(title),
            body=smart_bytes(body),
            webview_url=webview_url,
        )
        try:
            self._send_push_message(request)
            log.debug('Sent account modification push: event={}'.format(event_name))
        except BaseLogbrokerError:
            log.exception('Error sending account modification push: event={}'.format(event_name))
