# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import (
    bb_response_to_session,
    build_auth_cookies,
    build_cookie_mda2_beacon,
    build_non_auth_cookies,
    get_session_policy_by_ttl,
    is_user_session_valid,
    is_user_valid_in_multisession,
    log_auth_event,
    log_new_session,
    log_web_auth_updates,
    user_from_multisession,
    users_from_multisession,
)
from passport.backend.api.views.bundle.exceptions import UidNotInSessionError
from passport.backend.core import authtypes
from passport.backend.core.builders.blackbox import (
    BLACKBOX_EDITSESSION_OP_DELETE,
    BLACKBOX_EDITSESSION_OP_SELECT,
    BlackboxInvalidParamsError,
)
from passport.backend.core.conf import settings
from passport.backend.core.historydb.statuses import SESSION_KILL
from passport.backend.core.models.account import Account
from passport.backend.core.utils.domains import (
    build_passport_domain,
    get_cookie_domain_by_host,
)
from passport.backend.utils.time import get_unixtime


log = logging.getLogger('passport.api.view.bundle.mixins.session')


class BundleEditSessionMixin(object):

    def log_logout(self, cookie_session_info, uid):
        if is_user_valid_in_multisession(cookie_session_info, uid):
            log_auth_event(
                env=self.request.env,
                ttl=cookie_session_info.ttl,
                status=SESSION_KILL,
                uid=uid,
                authid=cookie_session_info.authid,
                # при удалении кук это всегда происходит на вебе и важно только это,
                # только запись 'ses_create' может иметь значение отличное от 'web'
                auth_type=authtypes.AUTH_TYPE_WEB,
            )

    def log_auth_events(self, cookie_session_info, editsession_response, deleted_uid=None):
        if deleted_uid is not None:
            self.log_logout(cookie_session_info, uid=deleted_uid)
        multi_session_users = users_from_multisession(cookie_session_info)
        log_web_auth_updates(
            self.request.env,
            multi_session_users,
            cookie_session_info.ttl,
            editsession_response.get('authid', {}).get('id'),
            exclude_uid=deleted_uid,
        )

    def build_empty_session(self, with_ssl_session=False, with_sessguard=False):
        common_session_args = {
            'domain': get_cookie_domain_by_host(self.host),
            'expires': 1,
            'value': '',
        }
        session = {
            'session': dict(common_session_args),
        }
        if with_ssl_session:
            session['sslsession'] = dict(common_session_args)
        if with_sessguard:
            session['sessguard'] = dict(common_session_args, domain='.%s' % build_passport_domain(self.request.env.host))
        return session

    def edit_session(self, operation, uid, cookie_session_info, statbox_params,
                     new_default_uid=None, password_check_time=None):
        session_cookie_body, ssl_session_cookie_body, host = self.get_effective_session_and_host()
        sessguard_body = self.get_sessguard()
        passport_domain = build_passport_domain(host)
        try:
            bb_response = self.blackbox.editsession(
                operation,
                session_cookie_body,
                uid,
                self.request.env.user_ip,
                host,
                new_default=new_default_uid,
                sslsessionid=ssl_session_cookie_body,
                password_check_time=password_check_time,
                sessguard=sessguard_body,
                guard_hosts=[passport_domain],
                request_id=self.request.env.request_id,
                get_login_id=True,
            )
        except BlackboxInvalidParamsError:
            raise UidNotInSessionError()

        session, _ = bb_response_to_session(bb_response, self.request.env)
        log_new_session(session)

        if self.track:
            self.track.session = session['session']['value']
            if 'sslsession' in session:
                self.track.sslsession = session['sslsession']['value']
            if 'sessguard' in session:
                self.track.sessguard = session['sessguard']['value']
            self.track.session_created_at = self.track.session_created_at or get_unixtime()

        new_default_account = None
        default_uid = bb_response['default_uid']
        if default_uid:
            default_uid = int(bb_response['default_uid'])
            new_default_session = user_from_multisession(cookie_session_info, default_uid)
            if is_user_session_valid(new_default_session):
                new_default_account = Account().parse(new_default_session)

        cookies = build_auth_cookies(self.request.env, session)

        cookie_path = '/'
        session_expires = session['session']['expires']

        is_2fa_enabled_yp = None
        if new_default_account:
            is_2fa_enabled_yp = True if bool(new_default_account.totp_secret.is_set) else None

        ignore_lah_for = settings.IGNORE_LAH_AFTER_LOGOUT_FOR if operation == BLACKBOX_EDITSESSION_OP_DELETE else None

        cookies.extend(
            build_non_auth_cookies(
                self.request.env,
                new_default_account.uid if new_default_account else None,
                new_default_account.human_readable_login if new_default_account else None,
                new_default_account.machine_readable_login if new_default_account else None,
                cookie_path,
                authorization_session_policy=get_session_policy_by_ttl(cookie_session_info.ttl),
                # Если новая дефолтная сессия пользователя невалидна(или ее нет)
                ignore_cookie_l=new_default_account is None,
                display_name=new_default_account.person.display_name if new_default_account else None,
                is_2fa_enabled_yp=is_2fa_enabled_yp,
                ignore_lah_for=ignore_lah_for,
                is_child=new_default_account.is_child if new_default_account else False,
            ),
        )

        # Если сессия пустая, то мы разлогинили последнего пользователя и
        # нужно заменить стандартный action на cookie_remove для совместимости
        # с кодом на Перле.
        if not session['session']['value']:
            self.statbox.log(
                action='cookie_remove',
                **statbox_params
            )
        else:
            self.statbox.log(**statbox_params)
        self.log_auth_events(
            cookie_session_info,
            bb_response,
            deleted_uid=uid if operation == BLACKBOX_EDITSESSION_OP_DELETE else None,
        )

        # Если куку с сессией удаляем, то ставим куку mda2_beacon на максимальный срок,
        # иначе на срок жизни сессии
        mda2_beacon_cookie_params = dict()
        if session['session']['value']:
            mda2_beacon_cookie_params['expires'] = session_expires
        cookies.append(
            build_cookie_mda2_beacon(
                self.request.env,
                **mda2_beacon_cookie_params
            ),
        )

        return cookies, default_uid

    def change_default(self, cookie_session_info, new_default_uid,
                       operation=BLACKBOX_EDITSESSION_OP_SELECT):
        """Сменим default-пользователя в валидной мульти-куке"""
        cookies, default_uid = self.edit_session(
            operation,
            new_default_uid,
            cookie_session_info,
            dict(action='cookie_edit'),
        )
        self.response_values.update(cookies=cookies)
        if default_uid:
            self.response_values.update(default_uid=default_uid)
