# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common import is_second_step_allowed
from passport.backend.api.common.account_manager import (
    AmPlatform,
    get_am_platform,
    get_am_version,
)
from passport.backend.api.common.authorization import (
    is_oauth_token_created,
    SessionScope,
)
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.login import allow_special_test_yandex_login_registration
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.auth.exceptions import AuthAlreadyPassedError
from passport.backend.api.views.bundle.auth.password.exceptions import RfcOtpInvalidError
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    CaptchaRequiredError,
    CompareNotMatchedError,
    InvalidTrackStateError,
    PasswordNotMatchedError,
    RateLimitExceedError,
    SecondStepRequired,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAuthenticateMixinV2,
    BundleAuthNotificationsMixin,
    BundleCacheResponseToTrackMixin,
    BundleNeophonishMixin,
    BundlePasswordVerificationMethodMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.challenge import (
    BundleChallengeMixin,
    is_yandex_force_new_challenge_test_login,
    MobileDeviceStatus,
    MobileProfile,
)
from passport.backend.api.views.bundle.mixins.password import CAPTCHA_AND_PHONE_VALIDATION_METHOD
from passport.backend.api.views.bundle.mobile.controllers.base import BaseMobileView
from passport.backend.api.views.bundle.mobile.exceptions import (
    ExternalOrNativeActionRequiredError,
    MagicLinkNotConfirmedError,
    NativeActionRequiredError,
)
from passport.backend.api.views.bundle.mobile.forms import (
    AuthAfterLoginRestoreForm,
    PasswordForm,
    RfcOtpForm,
)
from passport.backend.api.views.bundle.phone import helpers as phone_helpers
from passport.backend.api.views.bundle.phone.exceptions import PhoneNotConfirmedError
from passport.backend.api.views.bundle.states import (
    AuthChallenge,
    EmailCode,
    PasswordChangeForbidden,
    RedirectToCompletion,
    RedirectToForcedLiteCompletion,
    RedirectToLiteCompletion,
    RedirectToPasswordChange,
    RedirectToPDDCompletion,
    RfcTotp,
)
from passport.backend.core.authtypes import (
    AUTH_TYPE_MAGNITOLA,
    AUTH_TYPE_OAUTH_CREATE,
)
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS,
    BLACKBOX_SECOND_STEP_EMAIL_CODE,
    BLACKBOX_SECOND_STEP_RFC_TOTP,
)
from passport.backend.core.builders.captcha import get_captcha
from passport.backend.core.builders.ysa_mirror import (
    get_ysa_mirror_api,
    YsaMirrorBaseError,
)
from passport.backend.core.compare.compare import (
    compare_names,
    STRING_FACTOR_INEXACT_MATCH,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    bad_rfc_otp_counter,
    login_restore_counter,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.logging_utils.loggers.statbox import AntifraudLogger
from passport.backend.core.models.password import get_sha256_hash
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.version import is_version_left_gte_right
from passport.backend.utils.string import smart_str
from passport.backend.utils.time import get_unixtime
from six.moves.urllib.parse import (
    parse_qs,
    urlparse,
)


# По этим стейтам юзера можно слать в веб, и там он сам решит свои проблемы
STATES_USER_CAN_FIX_HIMSELF = (
    AuthChallenge,
    PasswordChangeForbidden,  # на самом деле пользователь не может починить это сам, но для него надо открыть вебвью
    RedirectToCompletion,
    RedirectToPDDCompletion,
    RedirectToLiteCompletion,
    RedirectToForcedLiteCompletion,
    RedirectToPasswordChange,
    EmailCode,
)


log = logging.getLogger(__name__)


class BaseAuthView(BaseMobileView,
                   BundleChallengeMixin,
                   BundleBaseAuthorizationMixin,
                   BundleAuthenticateMixinV2,
                   BundleAccountGetterMixin,
                   BundleCacheResponseToTrackMixin,
                   BundlePasswordVerificationMethodMixin,
                   BundleAccountPropertiesMixin,
                   BundleAuthNotificationsMixin,
                   BundlePhoneMixin):
    allow_captcha = True
    ignore_captcha = False
    is_password_auth = False
    require_track = True
    required_grants = ['mobile.auth']
    supports_auth_challenge = True
    type = None

    def __init__(self):
        super(BaseAuthView, self).__init__()
        self.mobile_profile = MobileProfile()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            type=self.type,
            ip=self.client_ip,
            user_agent=self.user_agent,
            track_id=self.track_id,
            consumer=self.consumer,
        )

    @cached_property
    def antifraud_log(self):
        return AntifraudLogger(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            ip=self.client_ip,
            AS=get_ip_autonomous_system(self.client_ip),
            user_agent=self.user_agent,
            external_id='track-{}'.format(self.track_id),
            uid=self.account.uid if self.account else None,
            service_id='login',
        )

    @cached_property
    def request_am_platform(self):
        return get_am_platform(self.track)

    def _check_captcha(self, captcha_key, captcha_answer):
        if not captcha_key or not captcha_answer:
            return False
        if allow_special_test_yandex_login_registration(self.track.user_entered_login, self.client_ip):
            return True
        return get_captcha().check(
            answer=captcha_answer,
            key=captcha_key,
            request_id=self.request.env.request_id,
        )

    def check_captcha_requirements(self):
        if not self.track.is_captcha_required or self.is_captcha_passed:
            return

        captcha_answer = self.form_values['captcha_answer']
        if self._check_captcha(self.track.captcha_key, captcha_answer):
            self.track.is_captcha_checked = True
            self.track.is_captcha_recognized = True
            return

        self.report_login_error_antifraud(comment_fields=['captcha_wrong_answer'])
        self.show_mobile_captcha()

    @property
    def is_magnitola(self):
        return (
            self.track.device_hardware_model and
            self.track.device_hardware_model.lower() in settings.MAGNITOLA_MODELS and
            self.track.device_application and
            self.track.device_application in settings.MAGNITOLA_APP_IDS
        )

    def can_use_new_challenge(self):
        am_version = get_am_version(self.track)
        am_platform = self.request_am_platform

        if is_yandex_force_new_challenge_test_login(self.account.login):
            log.debug('can_use_new_challenge: special login => new_challenge')
            return True

        # совсем старые АМ не передают версию, и уж тем более не умеют внешнее действие
        if not am_version or am_platform == AmPlatform.UNKNOWN:
            # Не красиво получается, что это условие в базовом классе, а само поле на форме - в дочернем
            if self.form_values.get('password_source'):
                log.debug('can_use_new_challenge: unknown AM + password_source passed => new_challenge')
                return True
            else:
                log.debug('can_use_new_challenge: unknown AM + no password_source => captcha')
                return False

        if am_platform == AmPlatform.IOS:
            am_version_with_external_action = settings.AM_IOS_CHALLENGE_MIN_VERSION
        elif am_platform == AmPlatform.ANDROID:
            if self.account.is_lite:
                am_version_with_external_action = settings.AM_ANDROID_CHALLENGE_MIN_VERSION_FOR_LITE
            else:
                am_version_with_external_action = settings.AM_ANDROID_CHALLENGE_MIN_VERSION
        else:
            log.debug('can_use_new_challenge: unknown platform {}'.format(am_platform))
            return False

        if self.is_magnitola:
            log.debug('can_use_new_challenge: is magnitola => captcha')
            return False

        left_gte_right = is_version_left_gte_right(am_version, am_version_with_external_action)
        log.debug('can_use_new_challenge: (platform {}) {} {} {} => {}'.format(
            smart_str(am_platform),
            smart_str(am_version),
            '>=' if left_gte_right else '<',
            am_version_with_external_action,
            'new_challenge' if left_gte_right else 'captcha',
        ))

        return left_gte_right

    def check_track_state(self):
        if not self.track.user_entered_login or not self.track.x_token_client_id:
            raise InvalidTrackStateError()
        if is_oauth_token_created(self.track):
            raise AuthAlreadyPassedError()

    def try_experiments(self):
        """Никаких экспериментов в этой ручке"""
        return

    @property
    def is_sms_validation_required(self):
        if self.track.is_change_password_sms_validation_required is None:
            self.track.is_change_password_sms_validation_required = (
                not settings.DISABLE_CHANGE_PASSWORD_PHONE_EXPERIMENT and
                self.is_secure_number_valid_or_missing
            )
        return self.track.is_change_password_sms_validation_required

    def save_data_for_next_step_to_track(self, redirect_state):
        # Составляем и сохраняем в трек информацию о пользователе и о стейте.
        # Нужно для того, чтобы с этим треком можно было продолжить работу в вебе (бОльшая часть полей нужна фронту).
        response_dict = {
            'status': 'ok',
            'track_id': self.track_id,
            'account': self.account_to_response(
                account=self.account,
                personal_data_required=True,
                account_info_required=True,
            ),
            'source': 'am',
        }
        redirect_state.update_response(response_dict)
        if isinstance(redirect_state, RedirectToForcedLiteCompletion):
            response_dict['has_recovery_method'] = self.check_has_recovery_method()

        if isinstance(redirect_state, RedirectToPasswordChange):
            self.save_secure_number_in_track()
            if (
                # is_sms_validation_required вызываем, только если validation_method подразумевает валидацию телефона
                # (у этой проперти есть сайд-эффект в виде выставления трековых полей). При этом secure_number может
                # и не быть (и фронт, не увидев полей в ответе, предложит его привязать)
                redirect_state.validation_method == CAPTCHA_AND_PHONE_VALIDATION_METHOD and
                self.is_sms_validation_required and
                self.has_secure_number
            ):
                self.track.has_secure_phone_number = True
                response_dict['number'] = phone_helpers.dump_number(self.secure_number)

        if self.form_values.get('password'):
            self.track.password_hash = get_sha256_hash(self.form_values['password'])

        # В треке мог остаться retpath от magic link. Удалим его, чтобы веб не средиректил туда пользователя ещё раз.
        self.track.retpath = None

        self.cache_response_dict_to_track(response_dict)

    def check_redirect_state(self, redirect_state=None, avatar_secret=None):
        redirect_state = redirect_state or self.check_user_policies()

        if self.supports_auth_challenge:
            can_use_new_challenge = self.can_use_new_challenge()
            if not can_use_new_challenge:
                self.track.do_not_save_fresh_profile = True

            if can_use_new_challenge:
                captcha_reason = None
            elif self.is_magnitola:
                captcha_reason = 'magnitola'
            else:
                captcha_reason = 'old_am_version'

            redirect_state = redirect_state or self.show_challenge_if_necessary(
                allow_new_challenge=can_use_new_challenge,
                mobile_profile=self.mobile_profile,
                avatar_secret=avatar_secret,
                captcha_reason=captcha_reason,
                allow_captcha=self.allow_captcha,
                is_magnitola=self.is_magnitola,
            )

        if (
            not isinstance(redirect_state, AuthChallenge) and
            self.mobile_profile.device_name and
            self.mobile_profile.device_status == MobileDeviceStatus.new
        ):
            self.try_send_auth_notifications(device_name=self.mobile_profile.device_name)

        if redirect_state is not None:
            redirect_state.update_response(self.response_values)
            if isinstance(redirect_state, STATES_USER_CAN_FIX_HIMSELF):
                self.save_data_for_next_step_to_track(redirect_state)
                raise ExternalOrNativeActionRequiredError('Perform natively or send user to web')
            else:
                raise NativeActionRequiredError('Perform natively or reject user')

    def process_request(self):
        if self.basic_form is not None:
            self.process_basic_form()
        self.read_track()
        self.check_track_state()

        with self.track_transaction.commit_on_error():
            if not self.ignore_captcha:
                self.check_captcha_requirements()
            self.check_credentials_and_user()
            use_avatar_with_secret = (self.request_am_platform != AmPlatform.IOS and
                                      (self.track.is_captcha_checked or self.track.is_avatar_secret_checked) and
                                      self.is_password_auth
                                      )
            self.fill_response_with_account_info(avatar_with_secret=use_avatar_with_secret)
            self.issue_oauth_tokens(password_passed=self.is_password_auth)
            self.process_env_profile()
            self.write_phone_to_log()
            self.try_bind_related_phonish_account()

    def check_credentials_and_user(self):
        # Важно тут запросить емейлы в ЧЯ
        raise NotImplementedError()  # pragma: no cover


class AuthByPasswordView(BaseAuthView):
    allow_captcha = True
    basic_form = PasswordForm
    is_password_auth = True
    supports_auth_challenge = True
    type = 'mobile_password'
    antifraud_auth_type = 'mobile_password'

    def check_if_device_is_blacklisted(self):
        if self.track.device_id in settings.DEVICE_ID_BLACKLIST:
            log.debug('Device_id `%s` is blacklisted: auth is forbidden', self.track.device_id)
            # Отдаём ошибку о неверном пароле для обратной совместимости и чтобы не раскрывать информацию
            self.report_login_error_antifraud(comment_fields=['device_is_blacklisted'])
            raise PasswordNotMatchedError()

    @staticmethod
    def extract_secret_from_avatar_url(avatar_url):
        return parse_qs(urlparse(avatar_url).query).get('secret') if avatar_url is not None else None

    def get_mirror_ysa_resolution(self):
        if not settings.YSA_MIRROR_API_ENABLED:
            return

        request_id = self.request.env.request_id
        if not request_id:
            log.warning('Cannot query mirror-ysa, nginx request id is empty')
            return

        if self.consumer != settings.MOBILEPROXY_CONSUMER:
            log.debug('Skipping mirror-ysa check for request_id {}: request is from {}, expected from {}'.format(
                request_id,
                self.consumer,
                settings.MOBILEPROXY_CONSUMER,
            ))
            return

        # проверим фродер или нет
        try:
            resolution = get_ysa_mirror_api().check_client_by_requestid_v2(
                ip_address=self.consumer_ip,
                request_id=request_id,
            )
            log.info('Mirror-ysa resolution: request with request_id {} is {}'.format(
                request_id,
                repr(resolution),
            ))
        except YsaMirrorBaseError as e:
            log.warning('Request to mirror-ysa for request_id {} has failed: {}'.format(
                request_id,
                e.message,
            ))
            self.statbox.log(
                action='ysa_mirror',
                status='error',
                error=e.__class__.__name__,
                request_id=request_id,
            )
        else:
            self.statbox.log(
                action='ysa_mirror',
                status='ok',
                request_id=request_id,
            )
            self.track.ysa_mirror_resolution = resolution.to_base64() if resolution else None

    def check_credentials_and_user(self):
        self.mobile_profile.password_source = self.form_values['password_source']

        authtype = AUTH_TYPE_OAUTH_CREATE
        require_app_password = False
        if self.is_magnitola:
            authtype = AUTH_TYPE_MAGNITOLA
            require_app_password = True
            self.antifraud_log.bind(
                model=self.track.device_hardware_model,
                app_id=self.track.device_application,
            )

        try:
            redirect_state = None

            self.blackbox_login(
                login=self.track.user_entered_login,
                password=self.form_values['password'],
                force_use_cache=False,
                authtype=authtype,
                need_phones=True,
                get_public_id=True,
                require_app_password=require_app_password,
            )
        except CaptchaRequiredError:
            if self.account:
                self.statbox.bind(uid=self.account.uid)
                self.antifraud_log.bind(uid=self.account.uid)

            # Надо не только ошибку кинуть, но и капчу сгенерировать
            self.show_mobile_captcha()
        except SecondStepRequired as e:
            if self.account:
                self.statbox.bind(uid=self.account.uid)
                self.antifraud_log.bind(uid=self.account.uid)

            if BLACKBOX_SECOND_STEP_RFC_TOTP in e.allowed_second_steps:
                redirect_state = RfcTotp()
            elif BLACKBOX_SECOND_STEP_EMAIL_CODE in e.allowed_second_steps:
                redirect_state = EmailCode()
            else:
                raise  # pragma: no cover

            self.prepare_track_for_second_step(e.allowed_second_steps)
        else:
            self.statbox.bind(uid=self.account.uid)
            self.antifraud_log.bind(uid=self.account.uid)

        self.get_mirror_ysa_resolution()
        self.check_if_device_is_blacklisted()
        self.mobile_profile.setup(self.track, self.account.uid)

        self.fill_track_with_account_data(password_passed=True)
        self.check_redirect_state(redirect_state,
                                  self.extract_secret_from_avatar_url(self.form_values['avatar_url']))
        self.track.allow_oauth_authorization = True


class AuthByRfcOtpView(BaseAuthView):
    allow_captcha = False
    basic_form = RfcOtpForm
    is_password_auth = True
    type = 'mobile_rfc_otp'
    antifraud_auth_typetype = 'mobile_rfc_otp'

    @property
    def supports_auth_challenge(self):
        # чтобы AM мог поддержать у себя это и проверить
        return settings.IS_TEST

    def check_track_state(self):
        super(AuthByRfcOtpView, self).check_track_state()
        if not is_second_step_allowed(self.track, BLACKBOX_SECOND_STEP_RFC_TOTP) or not self.track.uid:
            raise InvalidTrackStateError()

    def check_credentials_and_user(self):
        self.get_account_from_track(
            enabled_required=True,
            need_phones=True,
            emails=True,
            get_public_id=True,
        )
        if bad_rfc_otp_counter.get_counter().hit_limit(self.account.uid) and not self.is_captcha_passed:
            self.show_mobile_captcha()

        rv = self.blackbox.check_rfc_totp(
            uid=self.account.uid,
            totp=self.form_values['rfc_otp'],
        )

        if rv['status'] == BLACKBOX_CHECK_RFC_TOTP_VALID_STATUS:
            events = {'action': 'set_rfc_otp_check_time'}
            with UPDATE(self.account, self.request.env, events):
                self.account.rfc_totp_secret.check_time = rv['time']
            self.track.password_verification_passed_at = get_unixtime()
        else:
            if bad_rfc_otp_counter.incr_counter_and_check_limit_afterwards(self.account.uid):
                self.invalidate_captcha()
            raise RfcOtpInvalidError(rv['status'])

        self.check_redirect_state()
        self.track.allow_oauth_authorization = True


class AuthByMagicLinkView(BaseAuthView):
    allow_captcha = False
    ignore_captcha = True  # на этом шаге капча возникать не может, а капча от другого метода нас не интересует
    type = 'mobile_magic_link'
    antifraud_auth_type = 'mobile_magic_link'

    def check_track_state(self):
        super(AuthByMagicLinkView, self).check_track_state()
        if not self.track.uid:
            raise InvalidTrackStateError()

    def check_credentials_and_user(self):
        if not self.track.magic_link_confirm_time:
            self.report_login_error_antifraud(['magic_link_not_confirmed'])
            raise MagicLinkNotConfirmedError()

        self.get_account_from_track(
            enabled_required=True,
            need_phones=True,
            emails=True,
            get_public_id=True,
        )
        self.fill_track_with_account_data(password_passed=False)
        self.track.session_scope = str(SessionScope.xsession)
        self.check_redirect_state()
        self.track.allow_oauth_authorization = True


class AuthBySmsCodeView(BaseAuthView):
    allow_captcha = False
    ignore_captcha = True  # капча тут возникать не может, а капча от другого метода нас не интересует
    type = 'mobile_sms_code'
    antifraud_auth_type = 'mobile_sms_code'

    def check_track_state(self):
        super(AuthBySmsCodeView, self).check_track_state()
        if not (
            self.track.allowed_auth_methods and
            settings.AUTH_METHOD_SMS_CODE in self.track.allowed_auth_methods
        ):
            # TODO: унести подобную проверку в базовый класс через 3 часа после выкладки PASSP-26599 в прод
            raise InvalidTrackStateError()

    def check_credentials_and_user(self):
        self.get_account_by_login(
            login=self.track.user_entered_login,
            enabled_required=True,
            need_phones=True,
            emails=True,
            get_public_id=True,
        )
        if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
            raise PhoneNotConfirmedError()
        self.fill_track_with_account_data(password_passed=False)
        self.track.session_scope = str(SessionScope.xsession)
        self.check_redirect_state()
        self.track.allow_oauth_authorization = True


class AuthAfterLoginRestoreView(
    BundleNeophonishMixin,
    BaseAuthView,
):
    allow_captcha = False
    basic_form = AuthAfterLoginRestoreForm
    ignore_captcha = True  # капча тут возникать не может, а капча от другого метода нас не интересует
    type = 'mobile_after_login_restore'
    antifraud_auth_type = 'mobile_after_login_restore'

    def check_track_state(self):
        # не проверяем тип трека и наличие user_defined_login
        if not self.track.x_token_client_id:
            raise InvalidTrackStateError()
        if is_oauth_token_created(self.track):
            raise AuthAlreadyPassedError()

    def check_global_counters(self):
        # Те же счётчики, что и в восстановлении логина
        if (
            login_restore_counter.get_per_phone_buckets().hit_limit(self.account.phones.secure.number.digital) or
            login_restore_counter.get_per_ip_buckets().hit_limit_by_ip(self.client_ip)
        ):
            raise RateLimitExceedError()

    def increase_global_counters(self):
        # Те же счётчики, что и в восстановлении логина
        login_restore_counter.get_per_phone_buckets().incr(self.account.phones.secure.number.digital)
        login_restore_counter.get_per_ip_buckets().incr(self.client_ip)

    def check_credentials_and_user(self):
        self.get_account_by_uid(
            uid=self.form_values['uid'],
            enabled_required=True,
            need_phones=True,
            emails=True,
            get_public_id=True,
        )

        if not (self.account.is_neophonish or settings.ALLOW_AUTH_AFTER_LOGIN_RESTORE_FOR_ALL):
            raise AccountInvalidTypeError()

        if not self.is_secure_phone_confirmed_in_track(allow_by_call=False):
            raise PhoneNotConfirmedError()

        self.check_global_counters()
        firstname_factor, lastname_factor = compare_names(
            orig_names=[self.account.person.firstname, self.account.person.lastname],
            supplied_names=[self.form_values['firstname'], self.form_values['lastname']]
        )
        if firstname_factor < STRING_FACTOR_INEXACT_MATCH or lastname_factor < STRING_FACTOR_INEXACT_MATCH:
            self.increase_global_counters()
            raise CompareNotMatchedError()

        self.fill_track_with_account_data(password_passed=False)
        self.track.session_scope = str(SessionScope.xsession)
        self.check_redirect_state()
        self.track.allow_oauth_authorization = True

        location = self.get_location(self.track.language)
        device_name = self.track.device_name or self.track.device_hardware_model or self.track.device_os_id
        self.send_neophonish_auth_notification_to_messenger(location=location, device_name=device_name)
