# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import is_oauth_token_created
from passport.backend.api.common.profile import process_env_profile
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    OAuthUnavailableError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BindRelatedPhonishAccountMixin,
    BundleAccountResponseRendererMixin,
    BundleDeviceInfoMixin,
    BundlePhoneMixin,
    MailSubscriptionsMixin,
)
from passport.backend.api.views.bundle.utils import write_phone_to_log
from passport.backend.core import validators
from passport.backend.core.builders.oauth import (
    BaseOAuthError,
    get_fast_oauth,
    get_oauth,
)
from passport.backend.core.builders.oauth.oauth import OAUTH_PAYMENT_AUTH_PENDING_ERROR
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import merge_dicts
from passport.backend.utils.time import get_unixtime


log = logging.getLogger('passport.api.view.bundle.mobile')

DEFAULT_AVATAR_SIZE = 'normal'


class BaseMobileView(
    BaseBundleView,
    BundleDeviceInfoMixin,
    BundlePhoneMixin,
    BundleAccountResponseRendererMixin,
    BindRelatedPhonishAccountMixin,
    MailSubscriptionsMixin,
):
    required_headers = [HEADER_CONSUMER_CLIENT_IP]

    @cached_property
    def oauth(self):
        return get_oauth()

    @cached_property
    def fast_oauth(self):
        return get_fast_oauth()

    def fill_response_with_account_info(self, avatar_with_secret=False):
        self.fill_response_with_short_account_info(
            avatar_size=self.track.avatar_size or DEFAULT_AVATAR_SIZE,
            avatar_with_secret=avatar_with_secret,
        )

    def issue_oauth_tokens(self, password_passed=False):
        if not self.track.allow_oauth_authorization or is_oauth_token_created(self.track):   # pragma: no cover
            raise InvalidTrackStateError()

        device_info = self.track_to_oauth_params(self.get_device_params_from_track())
        x_token_response = self.oauth.token_by_uid(
            client_id=self.track.x_token_client_id,
            client_secret=self.track.x_token_client_secret,
            uid=self.account.uid,
            user_ip=self.request.env.user_ip,
            password_passed=password_passed,
            cloud_token=self.track.cloud_token,
            passport_track_id=self.track_id,
            **device_info
        )
        x_token = x_token_response.get('access_token')
        x_token_expires_in = x_token_response.get('expires_in')
        if not x_token:
            log.error('Failed to get x_token: %s', x_token_response)
            description = x_token_response.get('error_description', 'No `access_token` found in response')
            raise OAuthUnavailableError(description)

        self.response_values.update(
            cloud_token=self.track.cloud_token,
            x_token=x_token,
        )
        if x_token_expires_in:
            self.response_values.update(x_token_expires_in=x_token_expires_in)

        access_token = None
        if self.track.client_id:
            try:
                access_token_response = self.oauth.token_by_x_token(
                    client_id=self.track.client_id,
                    client_secret=self.track.client_secret,
                    x_token=x_token,
                    user_ip=self.request.env.user_ip,
                    payment_auth_retpath=self.track.payment_auth_retpath,
                    passport_track_id=self.track_id,
                    **device_info
                )
                access_token = access_token_response.get('access_token')
                access_token_expires_in = access_token_response.get('expires_in')
                if not access_token:
                    log.error('Failed to get access_token: %s', access_token_response)
                    if access_token_response.get('error') == OAUTH_PAYMENT_AUTH_PENDING_ERROR:
                        self.response_values.update(
                            payment_auth_context_id=access_token_response.get('payment_auth_context_id'),
                            payment_auth_url=access_token_response.get('payment_auth_url'),
                            payment_auth_app_ids=access_token_response.get('payment_auth_app_ids'),
                        )
                else:
                    self.response_values.update(access_token=access_token)
                    if access_token_expires_in:
                        self.response_values.update(access_token_expires_in=access_token_expires_in)
            except BaseOAuthError as e:
                log.error('Failed to get access_token: %s', e)

        self.track.oauth_token_created_at = self.track.oauth_token_created_at or get_unixtime()
        self.response_values.update(x_token_issued_at=self.track.oauth_token_created_at)
        self.statbox.log(
            action='tokens_issued',
            uid=self.account.uid,
            login=self.account.login,
            password_passed=password_passed,
            x_token_client_id=self.track.x_token_client_id,
            x_token_issued=True,
            client_id=self.track.client_id,
            client_token_issued=bool(access_token),
            cloud_token=self.track.cloud_token,
            **device_info
        )

    def process_env_profile(self):
        if self.track.do_not_save_fresh_profile:
            return

        if getattr(self, 'profile_load_result', None) and self.profile_load_result.success:
            ufo_profile = self.profile_load_result.content
        else:
            ufo_profile = None

        process_env_profile(
            account=self.account,
            track=self.track,
            ufo_profile=ufo_profile,
        )

    def write_phone_to_log(self):
        write_phone_to_log(
            account=self.account,
            user_cookies={},
        )


class MobilePhoneNumberValidationMixin(object):
    def try_parse_phone_number(self, raw_number, state, country_list=None):
        """Пытаемся распарсить номер со всеми переданными странами"""
        country_list = country_list or [None]
        error = None
        for country in country_list:
            try:
                result = validators.PhoneNumber().to_python(
                    {
                        'phone_number': raw_number,
                        'country': country,
                    },
                    state,
                )
                return result['phone_number'], country
            except validators.Invalid as e:
                error = e

        raise ValidationFailedError.from_invalid(error)


class MobileLiteRegistrationDataMixin(BundleDeviceInfoMixin):
    @property
    def lite_data_necessity(self):
        device_info = self.track_to_oauth_params(self.get_device_params_from_track())
        app_id = device_info.get('app_id', '')

        default_settings = settings.MOBILE_LITE_DATA_STATUS_DEFAULT
        custom_settings = {}

        # Будем перебирать правила в порядке уменьшения детализации префикса
        # (чем больше точек - тем более детален префикс)
        sorted_rules = sorted(
            settings.MOBILE_LITE_DATA_STATUS_BY_APP_ID_PREFIX.items(),
            key=lambda prefix_and_settings: prefix_and_settings[0].count('.'),
            reverse=True,
        )
        for app_id_prefix, settings_for_prefix in sorted_rules:
            if app_id == app_id_prefix or app_id.startswith(app_id_prefix + '.'):
                custom_settings = settings_for_prefix
                break

        return merge_dicts(default_settings, custom_settings)

    @property
    def lite_name_status(self):
        return self.lite_data_necessity[settings.DATA_TYPE_NAME]

    @property
    def lite_password_status(self):
        return self.lite_data_necessity[settings.DATA_TYPE_PASSWORD]

    @property
    def lite_phone_number_status(self):
        return self.lite_data_necessity[settings.DATA_TYPE_PHONE_NUMBER]

    @property
    def is_name_required_for_lite(self):
        return self.lite_name_status == settings.DATA_STATUS_REQUIRED

    @property
    def is_name_allowed_for_lite(self):
        return self.lite_name_status != settings.DATA_STATUS_NOT_USED

    @property
    def is_password_required_for_lite(self):
        return self.lite_password_status == settings.DATA_STATUS_REQUIRED

    @property
    def is_password_allowed_for_lite(self):
        return self.lite_password_status != settings.DATA_STATUS_NOT_USED

    @property
    def is_phone_number_required_for_lite(self):
        return self.lite_phone_number_status == settings.DATA_STATUS_REQUIRED

    @property
    def is_phone_number_allowed_for_lite(self):
        return self.lite_phone_number_status != settings.DATA_STATUS_NOT_USED
