# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.api.common.authorization import is_oauth_token_created
from passport.backend.api.views.bundle.auth.exceptions import AuthAlreadyPassedError
from passport.backend.api.views.bundle.exceptions import (
    ActionImpossibleError,
    ActionNotRequiredError,
    InvalidTrackStateError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleMagicLinkMixin,
    UserMetaDataMixin,
)
from passport.backend.api.views.bundle.mobile.controllers.base import (
    BaseMobileView,
    MobileLiteRegistrationDataMixin,
)
from passport.backend.api.views.bundle.mobile.forms import MagicLinkSendForm
from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import datetime_to_integer_unixtime


SEND_TO_EMAIL = 'email'


class BaseMagicLinkView(BaseMobileView,
                        BundleAccountGetterMixin,
                        BundleMagicLinkMixin):
    require_track = True
    required_grants = ['mobile.magic_link']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='magic_link',
            track_id=self.track_id,
            track_type=self.track.track_type,
            ip=self.client_ip,
        )

    def check_track_state(self):
        if not self.track.user_entered_login or not self.track.x_token_client_id:
            raise InvalidTrackStateError()
        if is_oauth_token_created(self.track):
            raise AuthAlreadyPassedError()
        self.assert_magic_link_not_invalidated()


class MagicLinkSendView(BaseMagicLinkView, UserMetaDataMixin):
    basic_form = MagicLinkSendForm

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        self.check_track_state()

        if not self.is_magic_link_for_registration:
            self.get_account_from_track(emails=True)
            self.statbox.bind_context(uid=self.account.uid)

        self.check_magic_link_send_counters()
        self.check_track_confirm_counters()

        if not self.is_magic_link_allowed:
            raise ActionImpossibleError()

        self.assert_magic_link_not_confirmed()

        if self.is_magic_link_for_registration:
            # На всякий случай повторно провалидируем логин из трека
            try:
                email = validators.LiteLogin().to_python(
                    self.track.user_entered_login,
                    state=validators.State(self.request.env),
                )
            except validators.Invalid:
                raise InvalidTrackStateError()

            language = self.track.language
        else:
            email = self.account.emails.default.address
            language = get_preferred_language(account=self.account)

        location = self.get_location(language)
        # TODO: после переезда OAuth в аркадию заюзать его эвристики по определению device_name
        device_name = self.track.device_name or self.track.device_hardware_model or self.track.device_os_id

        with self.track_transaction.rollback_on_error() as track:
            new_link_generated = False
            if not self.track.magic_link_secret or self.should_renew_magic_link_secret:
                track.magic_link_secret = self.make_magic_link_secret()
                time_now = datetime_to_integer_unixtime(datetime.now())
                track.magic_link_start_time = time_now
                track.magic_link_start_location = location
                track.magic_link_start_browser = device_name
                track.magic_link_sent_time = time_now
                new_link_generated = True

            track.retpath = self.form_values['retpath']

            track.magic_link_send_to = SEND_TO_EMAIL
            if self.is_magic_link_for_registration:
                self.send_registration_magic_link_to_email(
                    email,
                    language=language,
                    secret=track.magic_link_secret,
                    is_new_link_generated=new_link_generated,
                    device_name=device_name,
                    is_mobile=True,
                    redirect_after_confirm=True,
                )
            else:
                self.send_auth_magic_link_to_email(
                    email,
                    secret=track.magic_link_secret,
                    is_new_link_generated=new_link_generated,
                    device_name=device_name,
                    is_mobile=True,
                    redirect_after_confirm=True,
                )

        self.incr_magic_link_send_counters()
        self.response_values.update(
            poll_interval=settings.AUTH_MAGIC_LINK_POLL_INTERVAL,
            expires_in=settings.AUTH_MAGIC_LINK_TTL,
        )


class MagicLinkStatusView(BaseMagicLinkView, MobileLiteRegistrationDataMixin):
    def process_request(self):
        self.read_track()
        self.check_track_state()

        if not self.is_magic_link_allowed:
            raise ActionImpossibleError()

        self.check_track_state()
        self.assert_magic_link_sent()
        self.assert_magic_link_not_expired()

        magic_link_confirmed = False
        try:
            self.assert_magic_link_not_confirmed()
        except ActionNotRequiredError:
            magic_link_confirmed = True

        self.response_values.update(magic_link_confirmed=magic_link_confirmed)
        if magic_link_confirmed and self.is_magic_link_for_registration:
            self.response_values.update(
                lite_data_necessity=self.lite_data_necessity,
            )
