# -*- coding: utf-8 -*-
from datetime import datetime
import json
import logging

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.authorization import is_oauth_token_created
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    UserNotVerifiedError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleCleanWebMixin,
    BundleFrodoMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.api.views.bundle.register.exceptions import RegistrationSmsSendPerIPLimitExceededError
from passport.backend.api.views.register import (
    incr_registration_sms_per_ip_counter,
    incr_registration_sms_per_phone_counter,
    is_registration_sms_per_ip_counter_exceed,
    is_registration_sms_per_phone_counter_exceed,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.phones.phone import PhoneNumber
from passport.backend.core.runner.context_managers import CREATE
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.utils.decorators import cached_property

from ..forms import MobileRegisterForm
from .base import (
    BaseMobileView,
    DEFAULT_AVATAR_SIZE,
)


log = logging.getLogger('passport.api.views.bundle.mobile.register')


class MobileRegisterView(BaseMobileView,
                         BundleFrodoMixin,
                         BundleAccountGetterMixin,
                         BundleCleanWebMixin,
                         BundlePasswordValidationMixin):

    required_grants = ['mobile.register']
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    basic_form = MobileRegisterForm
    type = 'mobile'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register',
            type=self.type,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            consumer=self.consumer,
        )

    def log_statbox(self):
        self.statbox.log(
            action='account_created',
            karma=self.account.karma.value,
            password_quality=self.form_values['quality'],
            is_suggested_login=self.account.login in self.track.suggested_logins.get(),
            suggest_generation_number=self.track.suggest_login_count.get(default=0),
            country=self.track.country,
            retpath=self.track.retpath,
            uid=self.account.uid,
            login=self.account.normalized_login,
            aliasify=True,
        )

    def register_account(self):
        time_now = datetime.now()

        person_args = self.form_values.copy()
        person_args.update(
            language=self.track.language,
            country=self.track.country,
        )

        new_account = default_account(
            self.form_values['login'],
            time_now,
            person_args,
            build_default_person_registration_info(self.client_ip),
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }

        phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)

        save_secure_phone = self.build_save_secure_phone(
            aliasify=True,
            enable_search_alias=False,
            allow_to_take_busy_alias_from_any_account=False,
            language=self.track.language,
            account=new_account,
            phone_number=phone_number,
            should_ignore_binding_limit=False,
            is_new_account=True,
        )
        save_secure_phone.submit()

        with CREATE(new_account, self.request.env, events, datetime_=time_now) as self.account:
            save_secure_phone.commit()
            add_subscription(self.account, get_service(slug='mail'))
            self.unsubscribe_from_maillists_if_nessesary()

        self.statbox.bind_context(uid=self.account.uid)
        save_secure_phone.after_commit()

    def process_request(self):
        self.read_track()

        if is_oauth_token_created(self.track):
            if self.track.is_successful_registered:
                raise AccountAlreadyRegisteredError()
            else:
                raise InvalidTrackStateError()

        is_new_account = True

        with self.track_transaction.rollback_on_error():
            if not self.track.is_successful_registered:
                self.statbox.log(
                    action='submitted',
                )

                self.process_basic_form()
                self.clean_web_check_form_values()

                if not self.form_values['eula_accepted']:
                    raise EulaIsNotAcceptedError()

                if not self.is_phone_confirmed_in_track(allow_by_flash_call=True):
                    raise UserNotVerifiedError()

                if not self.track.phone_confirmation_phone_number:
                    raise InvalidTrackStateError()

                if (
                    is_registration_sms_per_ip_counter_exceed(self.request.env, self.track, mode=self.type) or
                    is_registration_sms_per_phone_counter_exceed(
                        self.track.phone_confirmation_phone_number,
                        self.track,
                        mode=self.type,
                    )
                ):
                    raise RegistrationSmsSendPerIPLimitExceededError()

                try:
                    self.validate_password(
                        uid='',
                        login=self.form_values['login'],
                    )
                except ValidationFailedError as e:
                    if 'password.likelogin' in e.errors:
                        # Из соображений совместимости отдаём более общую ошибку
                        e.errors.remove('password.likelogin')
                        e.errors.append('password.weak')
                    raise e

                self.register_account()

                self.track.uid = self.account.uid
                self.track.user_entered_login = self.form_values['login']
                self.track.is_successful_registered = True
                self.track.allow_oauth_authorization = True

                self.frodo_check_spammer(
                    'mobilereg',
                    increment_counters=True,
                    country=self.track.country,
                    language=self.track.language,
                )
                self.log_statbox()

                incr_registration_sms_per_ip_counter(self.request.env)
                incr_registration_sms_per_phone_counter(phone_number=self.track.phone_confirmation_phone_number)

            else:
                is_new_account = False
                if not self.track.uid:
                    raise InvalidTrackStateError()

                self.get_account_by_uid(
                    self.track.uid,
                    emails=True,
                    email_attributes='all',
                    get_public_id=True,
                )

            self.issue_oauth_tokens()
            self.process_env_profile()
            self.write_phone_to_log()
            self.fill_response_with_account_info()
            self.try_bind_related_phonish_account()

            if is_new_account:
                # А еще в ответе не будет дефолтового нативного почтового адреса,
                # потому что не хотим ни дублировать логику работы ЧЯ, ни отдавать неправду.
                # АМ обновит данные через ~сутки после регистрации сам.
                self.response_values.update(
                    display_login=self.account.login,
                    display_name=self.account.login,
                    avatar_url=settings.GET_AVATAR_URL % (
                        settings.DEFAULT_AVATAR_KEY,
                        self.track.avatar_size or DEFAULT_AVATAR_SIZE,
                    ),
                    is_avatar_empty=True,
                    public_id=self.blackbox.generate_public_id(uid=self.track.uid),
                )
