# -*- coding: utf-8 -*-
from datetime import datetime
import json
import logging

from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.authorization import (
    is_oauth_token_created,
    set_authorization_track_fields,
)
from passport.backend.api.common.pdd import does_domain_belong_to_pdd
from passport.backend.api.email_validator.mixins import EmailValidatorMixin
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    ActionImpossibleError,
    DomainInvalidTypeError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    PhoneVerificationRequiredError,
    UserNotVerifiedError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleCleanWebMixin,
    BundleFrodoMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.phones.phone import PhoneNumber
from passport.backend.core.runner.context_managers import CREATE
from passport.backend.core.utils.decorators import cached_property

from ..forms import MobileRegisterLiteForm
from .base import (
    BaseMobileView,
    DEFAULT_AVATAR_SIZE,
    MobileLiteRegistrationDataMixin,
)


log = logging.getLogger('passport.api.views.bundle.mobile.register_lite')


class MobileRegisterLiteView(BaseMobileView,
                             BundleFrodoMixin,
                             BundleAccountGetterMixin,
                             EmailValidatorMixin,
                             BundleCleanWebMixin,
                             BundlePasswordValidationMixin,
                             MobileLiteRegistrationDataMixin):

    required_grants = ['mobile.register_lite']
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    basic_form = MobileRegisterLiteForm
    type = 'mobile'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_lite',
            type=self.type,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            consumer=self.consumer,
        )

    def log_statbox(self):
        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.normalized_login,
            karma=self.account.karma.value,
            country=self.track.country,
            language=self.track.language,
            password_quality=self.form_values['quality'] if self.has_password else None,
        )

    def validate_extra_data(self):
        form_errors = []

        if self.is_name_required_for_lite:
            if not self.form_values['firstname']:
                form_errors.append('firstname.empty')
            if not self.form_values['lastname']:
                form_errors.append('lastname.empty')
        elif not self.is_name_allowed_for_lite:
            self.form_values.pop('firstname', None)
            self.form_values.pop('lastname', None)

        if self.is_password_required_for_lite and not self.form_values['password']:
            form_errors.append('password.empty')
        elif not self.is_password_allowed_for_lite:
            self.form_values.pop('password', None)
            self.form_values.pop('quality', None)

        if (
            self.is_phone_number_required_for_lite and
            not (
                self.is_phone_confirmed_in_track(allow_by_flash_call=True) and
                self.track.phone_confirmation_phone_number
            )
        ):
            raise PhoneVerificationRequiredError()

        if form_errors:
            raise ValidationFailedError(form_errors)

    @property
    def has_phone(self):
        return (
            self.is_phone_number_allowed_for_lite and
            bool(self.track.phone_confirmation_phone_number) and
            self.is_phone_confirmed_in_track(allow_by_flash_call=True)
        )

    @property
    def has_password(self):
        return self.is_password_allowed_for_lite and bool(self.form_values['password'])

    def register_account(self, email_address):
        person_args = self.form_values.copy()
        person_args.update(
            language=self.track.language,
            country=self.track.country,
        )

        new_account = default_account(
            email_address,
            datetime.now(),
            person_args,
            build_default_person_registration_info(self.client_ip),
            alias_type='lite',
        )
        events = {
            'action': 'account_register_lite',
            'consumer': self.consumer,
        }

        if self.has_phone:
            phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)

            if self.has_password:
                save_phone = self.build_save_secure_phone(
                    account=new_account,
                    phone_number=phone_number,
                    should_ignore_binding_limit=False,
                    is_new_account=True,
                    aliasify=True,
                    enable_search_alias=False,
                    allow_to_take_busy_alias_from_any_account=False,
                    language=self.track.language,
                )
            else:
                save_phone = self.build_save_simple_phone(
                    account=new_account,
                    phone_number=phone_number,
                    should_ignore_binding_limit=False,
                    is_new_account=True,
                )
            save_phone.submit()

        with CREATE(new_account, self.request.env, events) as self.account:
            if self.has_phone:
                save_phone.commit()

            self.account.emails.add(
                self.create_confirmed_email(email_address, is_unsafe=not self.has_password),
            )
            self.unsubscribe_from_maillists_if_nessesary()

        self.statbox.bind_context(uid=self.account.uid)
        if self.has_phone:
            save_phone.after_commit()

    def validate_email_address(self):
        try:
            state = validators.State(self.request.env)
            email_address = validators.LiteLogin().to_python(
                self.track.user_entered_login,
                state=state,
            )
            availability_validator = validators.Availability(login_type=ALIAS_NAME_TO_TYPE['lite'])
            availability_validator.to_python({'login': email_address}, state)
            if does_domain_belong_to_pdd(email_address):
                # на практике такое может возникнуть только в случае гонки
                raise DomainInvalidTypeError()
            return email_address
        except validators.Invalid as e:
            raise ValidationFailedError.from_invalid(e)

    def _process_registration(self):
        self.statbox.log(
            action='submitted',
        )

        if not self.track.magic_link_confirm_time:
            raise UserNotVerifiedError()

        self.process_basic_form()
        self.clean_web_check_form_values()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        email_address = self.validate_email_address()

        self.validate_extra_data()

        if self.has_password:
            self.validate_password(
                uid='',
                login=email_address,
            )

        self.register_account(email_address=email_address)

        self.track.is_successful_registered = True
        set_authorization_track_fields(
            self.account,
            self.track,
            allow_create_session=False,
            allow_create_token=True,
            password_passed=self.has_password,
        )

        self.frodo_check_spammer(
            'email_reg',
            increment_counters=True,
            country=self.track.country,
            language=self.track.language,
        )
        self.log_statbox()

    def process_request(self):
        if not settings.ALLOW_LITE_REGISTRATION:
            raise ActionImpossibleError()

        self.read_track()
        if not self.track.user_entered_login:
            raise InvalidTrackStateError()

        if is_oauth_token_created(self.track):
            if self.track.is_successful_registered:
                raise AccountAlreadyRegisteredError()
            else:
                raise InvalidTrackStateError()

        with self.track_transaction.rollback_on_error():
            if self.track.is_successful_registered:
                is_new_account = False
                if not self.track.uid:
                    raise InvalidTrackStateError()
                self.get_account_by_uid(
                    self.track.uid,
                    emails=True,
                    email_attributes='all',
                    get_public_id=True,
                )
            else:
                is_new_account = True
                self._process_registration()

            self.issue_oauth_tokens()
            self.process_env_profile()
            self.write_phone_to_log()
            self.fill_response_with_account_info()
            self.try_bind_related_phonish_account()

            if is_new_account:
                # А еще в ответе не будет дефолтового нативного почтового адреса,
                # потому что не хотим ни дублировать логику работы ЧЯ, ни отдавать неправду.
                # (Узнать точные данные можно лишь сходив в ЧЯ, но тут мы это делать не хотим)
                # АМ обновит данные через ~сутки после регистрации сам.
                self.response_values.update(
                    display_login=self.account.login,
                    display_name=self.account.login,
                    avatar_url=settings.GET_AVATAR_URL % (
                        settings.DEFAULT_AVATAR_KEY,
                        self.track.avatar_size or DEFAULT_AVATAR_SIZE,
                    ),
                    is_avatar_empty=True,
                    public_id=self.blackbox.generate_public_id(uid=self.track.uid),
                )
