# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.api.common.authorization import (
    is_oauth_token_created,
    set_authorization_track_fields,
)
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    ActionImpossibleError,
    EulaIsNotAcceptedError,
    InvalidTrackStateError,
    PhoneVerificationRequiredError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleCleanWebMixin,
    BundleNeophonishMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property

from ..forms import MobileRegisterNeophonishForm
from .base import (
    BaseMobileView,
    DEFAULT_AVATAR_SIZE,
)


log = logging.getLogger('passport.api.views.bundle.mobile.register_neophonish')


class MobileRegisterNeophonishView(BundleCleanWebMixin,
                                   BundleNeophonishMixin,
                                   BundleAccountGetterMixin,
                                   BaseMobileView):

    required_grants = ['mobile.register_neophonish']
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    basic_form = MobileRegisterNeophonishForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_neophonish',
            type='mobile',
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            consumer=self.consumer,
        )

    def log_statbox(self):
        self.statbox.log(
            action='account_created',
            uid=self.account.uid,
            login=self.account.normalized_login,
            country=self.track.country,
            language=self.track.language,
        )

    def _process_registration(self):
        self.process_basic_form()
        self.clean_web_check_form_values()

        if not self.form_values['eula_accepted']:
            raise EulaIsNotAcceptedError()

        person_args = self.form_values.copy()
        person_args.update(
            language=self.track.language,
            country=self.track.country,
        )
        self.register_neophonish(person_args)
        self.statbox.bind_context(uid=self.account.uid)

        self.track.is_successful_registered = True
        set_authorization_track_fields(
            self.account,
            self.track,
            allow_create_session=False,
            allow_create_token=True,
            password_passed=False,
        )
        self.log_statbox()
        self.send_neophonish_registration_notification_to_messenger()

    def process_request(self):
        self.statbox.log(
            action='submitted',
        )

        if not settings.ALLOW_NEOPHONISH_REGISTRATION:
            raise ActionImpossibleError()

        self.read_track()

        if not (
            self.is_phone_confirmed_in_track(allow_by_flash_call=True) and
            self.track.phone_confirmation_phone_number
        ):
            raise PhoneVerificationRequiredError()

        # Если в этом треке и зарегали аккаунт, и выдали токен - больше делать ничего не надо.
        # Если токен не выдали - вероятно, случился таймаут при запросе в oauth, так что продолжаем.
        # Если токен выдали, а аккаунт не зарегали - вероятно, это вообще посторонний трек.
        if is_oauth_token_created(self.track):
            if self.track.is_successful_registered:
                raise AccountAlreadyRegisteredError()
            else:
                raise InvalidTrackStateError()

        with self.track_transaction.rollback_on_error():
            if self.track.is_successful_registered:
                is_new_account = False
                if not self.track.uid:
                    raise InvalidTrackStateError()
                self.get_account_by_uid(self.track.uid, get_public_id=True)
            else:
                is_new_account = True
                self._process_registration()

            self.issue_oauth_tokens()
            self.process_env_profile()
            self.write_phone_to_log()
            self.fill_response_with_account_info()
            self.try_bind_related_phonish_account()

            if is_new_account:
                # Сами вычисляем данные, которые обычно вычисляет ЧЯ, чтобы отдать АМу полную инфу об аккаунте
                self.response_values.update(
                    display_name=' '.join([self.account.person.firstname, self.account.person.lastname]),
                    avatar_url=settings.GET_AVATAR_URL % (
                        settings.DEFAULT_AVATAR_KEY,
                        self.track.avatar_size or DEFAULT_AVATAR_SIZE,
                    ),
                    is_avatar_empty=True,
                    public_id=self.blackbox.generate_public_id(uid=self.track.uid),
                )
