# -*- coding: utf-8 -*-
from datetime import datetime
import logging

from passport.backend.api.common import ip
from passport.backend.api.common.account import (
    build_default_person_registration_info,
    default_account,
)
from passport.backend.api.common.authorization import is_oauth_token_created
from passport.backend.api.common.login import build_available_phonish_login
from passport.backend.api.common.phonish import get_phonish_namespace
from passport.backend.api.views.bundle.exceptions import (
    AccountAlreadyRegisteredError,
    AccountDisabledError,
    InvalidTrackStateError,
    RateLimitExceedError,
    UserNotVerifiedError,
)
from passport.backend.api.views.bundle.headers import HEADER_CONSUMER_CLIENT_IP
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleLastAuthMixin,
)
from passport.backend.api.views.bundle.register.exceptions import RegistrationSmsSendPerIPLimitExceededError
from passport.backend.api.views.register import (
    incr_registration_sms_per_ip_counter,
    incr_registration_sms_per_phone_counter,
    is_registration_sms_per_ip_counter_exceed,
    is_registration_sms_per_phone_counter_exceed,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.phones.phone import PhoneNumber
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.utils.decorators import cached_property

from .base import BaseMobileView


log = logging.getLogger('passport.api.views.bundle.mobile.register')


DEFAULT_AVATAR_SIZE = 'normal'


class RegisterPhonishView(BaseMobileView, BundleAccountGetterMixin, BundleLastAuthMixin):

    required_grants = ['mobile.register_phonish']
    require_track = True
    required_headers = (
        HEADER_CONSUMER_CLIENT_IP,
    )
    type = 'mobile'

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='register_phonish',
            type=self.type,
            track_id=self.track_id,
            ip=self.client_ip,
            user_agent=self.user_agent,
            consumer=self.consumer,
        )

    def log_statbox(self, login):
        self.statbox.log(
            action='account_created',
            country=self.track.country,
            retpath=self.track.retpath,
            uid=self.account.uid,
            login=login,
        )

    def create_account(self, phone_number, namespace=None):
        login = build_available_phonish_login(
            settings.PHONISH_LOGIN_GENERATION_RETRIES,
            self.request.env,
        )

        # Если пользователь заводит фониша используя старый номер, для которого
        # уже действует новый, то создаём фониша с новым номером, для того чтобы
        # прекратить появление новых фонишей со старыми номерами.
        phone_number = PhoneNumber.from_deprecated(phone_number)

        time_now = datetime.now()

        new_account = default_account(
            login,
            time_now,
            self.form_values,
            build_default_person_registration_info(self.client_ip),
            alias_type='phonish',
        )
        events = {
            'action': 'account_register',
            'consumer': self.consumer,
        }

        save_simple_phone = self.build_save_simple_phone(
            account=new_account,
            phone_number=phone_number,
            is_new_account=True,
            should_ignore_binding_limit=True,
        )

        with CREATE(new_account, self.request.env, events, datetime_=time_now) as self.account:
            save_simple_phone.submit()
            save_simple_phone.commit()
            add_subscription(
                self.account,
                service=get_service(slug='phonish'),
                login=login,
            )
            self.account.phonish_namespace = namespace

        self.statbox.bind_context(uid=self.account.uid)
        self.log_statbox(login)
        save_simple_phone.after_commit()

    def update_account(self, phone_number):
        if not self.account.is_enabled:
            raise AccountDisabledError()

        # СИБ: если видим, что отдаём тот же Phonish-аккаунт, то
        # делаем global logout, чтобы отозвать токены со старого
        # устройства (токены должны жить только на 1 устройстве).
        #
        # Правильно не отзывать токены полученные на данное устройство.
        with UPDATE(
                self.account,
                self.request.env,
                {
                    'action': 'login_phonish',
                    'consumer': self.consumer,
                },
        ):
            self.account.global_logout_datetime = datetime.now()
            phone = self.account.phones.by_number(phone_number)
            phone.confirm()

        self.statbox.log(
            action='login_phonish',
            phone_number=phone_number.masked_format_for_statbox,
            phone_id=self.account.phones.by_number(phone_number).id,
        )

    def process_request(self):
        if ip.is_ip_blacklisted(self.client_ip):
            log.debug('Phonish creation attempt from blacklisted ip %s', self.client_ip)
            raise RateLimitExceedError()

        self.read_track()

        if is_oauth_token_created(self.track):
            if self.track.is_successful_registered:
                raise AccountAlreadyRegisteredError()
            else:
                raise InvalidTrackStateError()

        if not self.is_phone_confirmed_in_track():
            raise UserNotVerifiedError()

        if not self.track.phone_confirmation_phone_number:
            raise InvalidTrackStateError()

        phone_number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)
        is_new_account = True

        with self.track_transaction.rollback_on_error():
            if not self.track.is_successful_registered:
                self.statbox.log(action='submitted')

                phonish_namespace = get_phonish_namespace(app_id=self.track.device_application)
                self.account, phone_number = self.get_account_by_phone_number(
                    phone_number,
                    phonish_namespace=phonish_namespace,
                )

                if self.account is not None:
                    self.update_account(phone_number)
                else:
                    if (
                        is_registration_sms_per_ip_counter_exceed(
                            self.request.env,
                            self.track,
                            mode='mobile_register_phonish',
                        ) or
                        is_registration_sms_per_phone_counter_exceed(
                            self.track.phone_confirmation_phone_number,
                            self.track,
                            mode='mobile_register_phonish',
                        )
                    ):
                        raise RegistrationSmsSendPerIPLimitExceededError()

                    self.create_account(phone_number, namespace=phonish_namespace)

                self.track.uid = self.account.uid
                self.track.is_successful_registered = True
                self.track.allow_oauth_authorization = True

                incr_registration_sms_per_ip_counter(self.request.env)
                incr_registration_sms_per_phone_counter(phone_number=self.track.phone_confirmation_phone_number)

            else:
                is_new_account = False
                if not self.track.uid:
                    raise InvalidTrackStateError()
                self.get_account_by_uid(self.track.uid, get_public_id=True)

            self.issue_oauth_tokens()
            self.write_phone_to_log()
            self.fill_response_with_account_info()

            if is_new_account:
                # Сами вычисляем данные, которые обычно вычисляет ЧЯ, чтобы отдать АМу полну инфу об аккаунте
                self.response_values.update(
                    display_name=phone_number.e164,
                    avatar_url=settings.GET_AVATAR_URL % (
                        settings.DEFAULT_AVATAR_KEY,
                        self.track.avatar_size or DEFAULT_AVATAR_SIZE,
                    ),
                    is_avatar_empty=True,
                    public_id=self.blackbox.generate_public_id(uid=self.track.uid),
                )
