# -*- coding: utf-8 -*-
from datetime import datetime
import re
import uuid

from passport.backend.api.common import (
    get_allowed_auth_methods,
    get_masked_magic_link_email,
)
from passport.backend.api.common.pdd import does_domain_belong_to_pdd
from passport.backend.api.common.social_api import try_get_social_profiles_for_auth
from passport.backend.api.common.suggest import get_countries_suggest
from passport.backend.api.forms.base import DeviceInfoForm
from passport.backend.api.views.bundle.exceptions import (
    AccountNotFoundError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.views.bundle.mobile.controllers.base import (
    BaseMobileView,
    MobileLiteRegistrationDataMixin,
    MobilePhoneNumberValidationMixin,
)
from passport.backend.api.views.bundle.mobile.forms import (
    StartForm,
    StartV1Form,
)
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.core import validators
from passport.backend.core.builders.blackbox.exceptions import BlackboxInvalidParamsError
from passport.backend.core.conf import settings
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.types.account.account import ACCOUNT_TYPE_NORMAL
from passport.backend.core.types.login.login import strongly_masked_login
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import datetime_to_integer_unixtime


PHONE_NUMBER_RE = re.compile(r'^\+?[\d()\- ]+$')


class StartProcessV1View(BaseMobileView, BundleAccountGetterMixin, MobilePhoneNumberValidationMixin):
    require_track = False
    required_grants = ['mobile.base']
    basic_form = StartV1Form

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='any_auth',
            type='mobile_start',
            ip=self.client_ip,
            user_agent=self.user_agent,
            consumer=self.consumer,
        )

    def create_and_fill_track(self, track_type, login=None, country=None, allowed_auth_methods=None):
        self.create_track(track_type)

        with self.track_transaction.rollback_on_error() as track:
            track.avatar_size = self.form_values['avatar_size']
            track.captcha_scale_factor = self.form_values['captcha_scale_factor']
            track.display_language = self.form_values['display_language']
            track.language = self.form_values['display_language']
            track.cloud_token = self.form_values['cloud_token'] or self.generate_cloud_token()
            track.country = country
            track.payment_auth_retpath = self.form_values['payment_auth_retpath']
            track.x_token_client_id = self.form_values['x_token_client_id']
            track.x_token_client_secret = self.form_values['x_token_client_secret']
            if login:
                track.user_entered_login = login
            if allowed_auth_methods:
                track.allowed_auth_methods = allowed_auth_methods
            if self.form_values['client_id']:
                track.client_id = self.form_values['client_id']
                track.client_secret = self.form_values['client_secret']

            if self.form_values.get('old_track_id'):
                old_track = self.track_manager.read(self.form_values.get('old_track_id'))
                if self.is_phone_confirmed_in_track(track=old_track, allow_by_flash_call=True):
                    track.copy_phone_confirmation_from_other_track(old_track)

        self.save_device_params_to_track(self.device_info)

        self.response_values.update(track_id=self.track_id)
        self.log_statbox_opened()

    @staticmethod
    def generate_cloud_token():
        return 'cl-%s' % uuid.uuid4().get_hex()

    def try_validate_login(self, login, state):
        try:
            validators.Login().to_python(login, state)
            validators.Availability().to_python(self.form_values, state)

            return login, ACCOUNT_TYPE_NORMAL
        except validators.Invalid as validation_error:
            raise ValidationFailedError.from_invalid(validation_error)

    def try_register(self, account_not_found_error):
        # Сейчас саджест для мобильных всегда возвращает список длины 1. Но перестрахуемся и поддержим все случаи.
        countries = get_countries_suggest() or [None]
        country = countries[0]

        login, phone_number = None, None

        state = validators.State(self.request.env)
        if self.form_values['login']:
            login, _ = self.try_validate_login(self.form_values['login'], state)
        else:
            # Если есть номер телефона - возьмём ту страну, для которой его смогли распарсить
            phone_number, country = self.try_parse_phone_number(
                self.form_values['phone_number'],
                state,
                countries,
            )
            self.response_values.update(
                phone_number=dump_number(phone_number),
            )

        # Переданный логин или телефон валидны, можно предложить регистрацию, если она возможна
        if settings.ALLOW_REGISTRATION:
            self.response_values.update(can_register=True)
            # Здесь в трек кладём логин, но не номер телефона, иначе регистрирующая ручка запутается
            self.create_and_fill_track('register', login=login, country=country)

        raise account_not_found_error

    def get_allowed_auth_methods(self):
        auth_methods = []
        if self.account.have_password:
            auth_methods.append(settings.AUTH_METHOD_PASSWORD)
        return auth_methods

    def try_auth(self):
        auth_methods = self.get_allowed_auth_methods()

        self.response_values.update(
            primary_alias_type=self.get_alias_type(can_handle_neophonish=False),
            auth_methods=auth_methods,
        )

        if self.account.is_neophonish:
            # По primary_alias_type не можем отличить неофониша от лайта
            self.response_values.update(is_neophonish=True)

        if auth_methods:
            # Здесь в трек кладём логин либо номер телефона, чтобы на следующем шаге отдать в ЧЯ
            self.create_and_fill_track(
                'authorize',
                login=self.user_entered_value,
                allowed_auth_methods=auth_methods,
            )
            self.response_values.update(can_authorize=True)
            if not self.form_values['login']:
                self.response_values.update(masked_login=strongly_masked_login(self.account.login))

            with self.track_transaction.rollback_on_error() as track:
                if settings.AUTH_METHOD_MAGIC_LINK in auth_methods:
                    track.uid = self.account.uid
                    track.magic_link_start_time = datetime_to_integer_unixtime(datetime.now())
                    magic_link_email = get_masked_magic_link_email(
                        account=self.account,
                        user_entered_login=self.form_values['login'],
                    )
                    self.response_values.update(magic_link_email=magic_link_email)
                if settings.AUTH_METHOD_SMS_CODE in auth_methods:
                    track.uid = self.account.uid
                    track.has_secure_phone_number = bool(self.secure_number)
                    track.secure_phone_number = self.secure_number.e164
                    self.response_values.update(
                        secure_phone_number=dump_number(self.secure_number, only_masked=True),
                    )


    @property
    def user_entered_value(self):
        return self.form_values['login'] or self.form_values['phone_number']

    def log_statbox_opened(self):
        device_info = self.process_form(DeviceInfoForm, self.all_values)
        self.statbox.log(
            action='opened',
            track_id=self.track_id,
            user_entered_login=self.user_entered_value,
            **device_info
        )

    def process_request(self):
        self.process_basic_form()
        self.device_info = self.process_form(DeviceInfoForm, self.all_values)

        try:
            if self.form_values['force_register']:
                # даже не пытаемся искать аккаунт, пригодный для авторизации
                raise AccountNotFoundError('registration forced')
            self.get_account_by_alt_logins(login=self.user_entered_value, emails=True, need_phones=True)
        except AccountNotFoundError as blackbox_error:
            self.try_register(account_not_found_error=blackbox_error)
        except BlackboxInvalidParamsError:
            # возможность регистрации не проверяем, логин заведомо невалидный
            raise ValidationFailedError(['login.prohibitedsymbols'])
        else:
            # Пользователь нашёлся и не заблокирован. Проверим, можно ли им авторизоваться
            self.try_auth()


class StartProcessView(StartProcessV1View, MobileLiteRegistrationDataMixin):
    basic_form = StartForm

    @property
    def user_entered_value(self):
        return self.form_values['login']

    def get_allowed_auth_methods(self):
        profiles = try_get_social_profiles_for_auth(self.account)
        return get_allowed_auth_methods(
            account=self.account,
            is_web=False,
            social_profiles=profiles,
            user_ip=self.client_ip,
            device_id=self.device_info.get('device_id'),
            cloud_token=self.form_values['cloud_token'],
        )

    def _try_validate_login(self, login, login_type, login_validator, state):
        if login_type == 'portal' and '@' in login:
            # если юзер ввёл яндексовый логин с доменной частью - отрежем яндексовый домен
            login_part, domain_part = login.rsplit('@', 1)
            if domain_part.strip() in settings.NATIVE_EMAIL_DOMAINS:
                login = login_part.strip()

        if login_type == 'lite' and not settings.ALLOW_LITE_REGISTRATION:
            raise validators.Invalid('invalid', login, state)

        login_validator.to_python(login, state)
        availability_validator = validators.Availability(login_type=ALIAS_NAME_TO_TYPE[login_type])
        availability_validator.to_python({'login': login}, state)

        if login_type == 'lite':
            if does_domain_belong_to_pdd(login):
                # Лайтов на ПДД-домене мы не разрешаем
                raise AccountNotFoundError()

        return login, login_type

    def try_validate_login(self, login, state):
        for login_type, login_validator in (
            ('lite', validators.LiteLogin()),
            ('portal', validators.Login()),  # предпочтительный тип должен стоять последним
        ):
            try:
                validated_login, account_type = self._try_validate_login(
                    login=login,
                    login_type=login_type,
                    login_validator=login_validator,
                    state=state,
                )
                # всё хорошо, мы нашли, какому типу аккаунта соответствует логин
                return validated_login, account_type
            except validators.Invalid as e:
                validation_error = e

        raise ValidationFailedError.from_invalid(validation_error)

    def try_register(self, account_not_found_error, phone_number=None, country=None):
        login, account_type = None, None

        state = validators.State(self.request.env)
        if phone_number:
            account_type = 'portal'  # допустимы разные регистрации, но портальную считаем приоритетной
        else:
            if self.form_values['is_phone_number'] or PHONE_NUMBER_RE.match(self.user_entered_value):
                # если уверены, что это телефон, можно не пытаться валидировать значение как логин
                raise ValidationFailedError(['phone_number.invalid'])
            else:
                login, account_type = self.try_validate_login(
                    login=self.user_entered_value,
                    state=state,
                )

        # Переданный логин или телефон валидны, можно предложить регистрацию, если она возможна
        if settings.ALLOW_REGISTRATION:
            allowed_account_types = [account_type]
            if phone_number is not None and settings.ALLOW_NEOPHONISH_REGISTRATION:
                allowed_account_types.append('neophonish')

            self.create_and_fill_track('register', login=login, country=country)
            self.response_values.update(
                can_register=True,
                account_type=account_type,
                allowed_account_types=allowed_account_types,
            )
            if login is not None:
                self.response_values.update(
                    login=login,
                )
            if phone_number is not None:
                self.response_values.update(
                    phone_number=dump_number(phone_number),
                    country=country,
                )
            if account_type == 'lite':
                self.response_values.update(
                    lite_data_necessity=self.lite_data_necessity,
                )

        raise account_not_found_error

    def process_request(self):
        self.process_basic_form()
        self.device_info = self.process_form(DeviceInfoForm, self.all_values)

        # Сейчас саджест для мобильных всегда возвращает список длины 1. Но перестрахуемся и поддержим все случаи.
        countries = get_countries_suggest() or [None]
        country = countries[0]
        try:
            state = validators.State(self.request.env)
            phone_number, country = self.try_parse_phone_number(
                self.user_entered_value,
                state,
                countries,
            )
            self.response_values.update(
                phone_number=dump_number(phone_number),
                country=country,
            )
        except ValidationFailedError:
            phone_number = None

        try:
            if self.form_values['force_register']:
                # даже не пытаемся искать аккаунт, пригодный для авторизации
                raise AccountNotFoundError('registration forced')
            self.get_account_by_alt_logins(login=self.user_entered_value, emails=True, need_phones=True)
            # Пользователь нашёлся и не заблокирован. Проверим, можно ли им авторизоваться
            self.try_auth()
        except AccountNotFoundError as blackbox_error:
            self.try_register(
                account_not_found_error=blackbox_error,
                phone_number=phone_number,
                country=country,
            )
        except BlackboxInvalidParamsError:
            # возможность регистрации не проверяем, логин заведомо невалидный
            raise ValidationFailedError(['login.prohibitedsymbols'])
        finally:
            if not self.track_id:
                self.create_and_fill_track('restore')
