# -*- coding: utf-8 -*-
import time

from passport.backend.api.common.phone_number import MIN_STATBOX_PHONE_LENGTH
from passport.backend.api.common.suggest import get_countries_suggest
from passport.backend.api.views.bundle.exceptions import ValidationFailedError
from passport.backend.api.views.bundle.mixins.phone import OctopusAvailabilityMixin
from passport.backend.api.views.bundle.mobile.controllers.base import (
    BaseMobileView,
    MobilePhoneNumberValidationMixin,
)
from passport.backend.api.views.bundle.mobile.forms import PhoneNumberForm
from passport.backend.api.views.bundle.phone import helpers as phone_helpers
from passport.backend.core import validators
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.types.phone_number.phone_number import mask_for_statbox
from passport.backend.core.utils.decorators import cached_property


class ValidatePhoneNumberView(BaseMobileView, MobilePhoneNumberValidationMixin, OctopusAvailabilityMixin):
    require_track = True
    required_grants = ['phone_number.validate']
    basic_form = PhoneNumberForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            action='sanitize_phone_number',
            track_id=self.track_id if self.track_id else '',
        )

    def process_request(self):
        self.process_basic_form()
        self.read_track()

        user_phone_number = self.form_values['phone_number']
        phone_number, country = None, None
        valid_for_call_status = None
        try:
            phone_number, country = self.try_parse_phone_number(
                self.form_values['phone_number'],
                state=validators.State(self.request.env),
                country_list=get_countries_suggest(),
            )
            self.response_values.update(
                phone_number=phone_helpers.dump_number(phone_number),
            )
            self.statbox.bind(sanitize_phone_result=phone_number.masked_format_for_statbox)

            if self.form_values['validate_for_call']:
                valid_for_call_status = self.check_valid_for_call(self.consumer, phone_number)
                self.response_values['valid_for_call'] = valid_for_call_status.valid_for_call
                self.response_values['valid_for_flash_call'] = valid_for_call_status.valid_for_flash_call
        except ValidationFailedError as e:
            self.statbox.bind(
                sanitize_phone_result=mask_for_statbox(user_phone_number),
                sanitize_phone_error=e._invalid_exception.code,
            )
            raise
        finally:
            with self.track_transaction.rollback_on_error():
                if country:
                    self.track.country = country
                self.track.sanitize_phone_error = 'sanitize_phone_error' in self.statbox.all_values
                self.track.sanitize_phone_count.incr()
                if self.track.sanitize_phone_first_call is None:
                    self.track.sanitize_phone_first_call = time.time()
                self.track.sanitize_phone_last_call = time.time()
                if phone_number and self.form_values['validate_for_call'] and valid_for_call_status:
                    self.track.phone_valid_for_call = valid_for_call_status.valid_for_call
                    self.track.phone_valid_for_flash_call = valid_for_call_status.valid_for_flash_call
                    self.track.phone_validated_for_call = phone_number.e164

            if len(user_phone_number) >= MIN_STATBOX_PHONE_LENGTH:
                self.statbox.log()
