# -*- coding: utf-8 -*-
import logging

from passport.backend.api.forms import DeviceInfoForm
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    BaseBundleError,
    HeadersEmptyError,
)
from passport.backend.api.views.bundle.headers import (
    AUTHORIZATION_HEADERS,
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleDeviceInfoMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.oauth.forms import (
    ClientEditedForm,
    DeviceAuthorizeSubmitOrCommitForm,
    TokenBySessionidForm,
)
from passport.backend.core.builders.oauth import (
    BaseOAuthError,
    get_oauth,
    OAuthPermanentError,
)
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language


SEND_MAIL_GRANT = 'oauth_client.send_mail'
DEVICE_AUTHORIZE_GRANT = 'oauth_client.device_authorize'
TOKEN_BY_SESSIONID_GRANT = 'oauth_client.token_by_sessionid'


log = logging.getLogger('passport.api.view.bundle.oauth')


class ClientEditedNotificationView(BaseBundleView, BundleAccountGetterMixin):
    require_track = False
    required_headers = AUTHORIZATION_HEADERS
    required_grants = [SEND_MAIL_GRANT]
    basic_form = ClientEditedForm

    email_template = 'mail/oauth_client_edited_notification.html'
    email_subject_key = 'oauth_client_edited.subject'

    def process_request(self):
        self.process_basic_form()
        if not (
            self.form_values['redirect_uris_changed'] or
            self.form_values['scopes_changed']
        ):
            raise ActionNotRequiredError()

        self.get_account_from_session(
            multisession_uid=self.form_values['uid'],
            emails=True,
        )
        self.send_email_notifications()

    def send_email_notifications(self):
        language = get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = self.email_template
        info = MailInfo(
            subject=translations[self.email_subject_key],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
            context={
                'is_2fa_on': self.account.totp_secret.is_set,
                'client_id': self.form_values['client_id'],
                'client_title': self.form_values['client_title'],
                'redirect_uris_changed': self.form_values['redirect_uris_changed'],
                'scopes_changed': self.form_values['scopes_changed'],
            },
        )
        return template_name, info, context


class BaseDeviceAuthorizeSubmitCommitView(BaseBundleView, BundleDeviceInfoMixin):
    require_track = False
    required_grants = [DEVICE_AUTHORIZE_GRANT]
    basic_form = DeviceAuthorizeSubmitOrCommitForm

    def get_oauth_response(self):
        raise NotImplementedError()

    def process_request(self):
        self.process_basic_form()

        try:
            self.check_header(HEADER_CONSUMER_AUTHORIZATION)
        except HeadersEmptyError:
            try:
                self.check_headers([HEADER_CLIENT_HOST, HEADER_CLIENT_COOKIE])
            except HeadersEmptyError:
                # Ошибка может сбивать с толку, т.к.
                # по факту нужны или authorization, или host + cookie,
                # но ошибка будет про все три разом.
                raise HeadersEmptyError([
                    HEADER_CONSUMER_AUTHORIZATION.code_for_error,
                    HEADER_CLIENT_HOST.code_for_error,
                    HEADER_CLIENT_COOKIE.code_for_error,
                ])

        try:
            response = self.get_oauth_response()
            response.pop('status')
        except OAuthPermanentError as e:
            exception = BaseBundleError()
            exception.error = e.args[0]
            raise exception
        else:
            self.response_values.update(
                **response
            )


class DeviceAuthorizeSubmit(BaseDeviceAuthorizeSubmitCommitView):
    def get_oauth_response(self):
        response = get_oauth().device_authorize_submit(
            language=self.form_values['language'],
            uid=self.form_values['uid'],
            code=self.form_values['code'],
            cookie=self.headers.get(HEADER_CLIENT_COOKIE.name),
            authorization=self.authorization,
            client_id=settings.TV_AUTH_CLIENT_ID,
            client_host=self.host,
            **self.form_to_oauth_params(self.form_values)
        )
        return response


class DeviceAuthorizeInfo(BaseBundleView):
    required_grants = [DEVICE_AUTHORIZE_GRANT]
    require_track = True

    def process_request(self):
        self.read_track()

        response = dict(
            browser=settings.BROWSER_DECODE.get(int(self.track.browser_id)) if self.track.browser_id else None,
            os_family=settings.OS_DECODE.get(int(self.track.os_family_id)) if self.track.os_family_id else None,
            region_id=int(self.track.region_id) if self.track.region_id else None,
            device_name=self.track.device_name,
        )

        self.response_values.update(
            **response
        )


class DeviceAuthorizeCommit(BaseDeviceAuthorizeSubmitCommitView):
    def get_oauth_response(self):
        response = get_oauth().device_authorize_commit(
            language=self.form_values['language'],
            uid=self.form_values['uid'],
            code=self.form_values['code'],
            cookie=self.headers.get(HEADER_CLIENT_COOKIE.name),
            authorization=self.authorization,
            client_id=settings.TV_AUTH_CLIENT_ID,
            client_host=self.host,
            **self.form_to_oauth_params(self.form_values)
        )
        return response


class TokenBySessionid(
    BaseBundleView,
    BundleAccountGetterMixin,
    BundlePhoneMixin,
):
    required_grants = [TOKEN_BY_SESSIONID_GRANT]
    basic_form = TokenBySessionidForm
    required_headers = [HEADER_CONSUMER_CLIENT_IP, HEADER_CLIENT_HOST, HEADER_CLIENT_COOKIE]

    def _should_set_trusted_device(self):
        if not self.track_id:
            log.debug('Token not trusted: no track')
            return False
        self.read_track()
        if self.track.allow_set_xtoken_trusted:
            log.debug('Token is trusted by track, challenge={}'.format(
                self.track.auth_challenge_type,
            ))
            return True
        else:
            log.debug('Token is not trusted by track')
            return False

    def process_request(self):
        self.process_basic_form()
        self.get_account_from_session(multisession_uid=self.form_values['uid'])

        device_info = self.process_form(DeviceInfoForm, self.all_values)

        try:
            response = get_oauth().token_by_uid(
                client_id=self.form_values['client_id'],
                client_secret=self.form_values['client_secret'],
                uid=self.account.uid,
                user_ip=self.client_ip,
                set_is_xtoken_trusted=self._should_set_trusted_device() or None,
                login_id=self.login_id,
                auth_source='sessionid',
                passport_track_id=self.track_id,
                **device_info
            )
        except BaseOAuthError as err:
            exception = BaseBundleError()
            exception.error = err.args[0]
            raise exception

        self.response_values.update(response)
