# -*- coding: utf-8 -*-

from datetime import datetime

from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.utils.time import datetime_to_unixtime

from ..exceptions import (
    AccountDisabledError,
    AccountNotFoundError,
    InternalPermanentError,
    InternalTemporaryError,
    InvalidTrackStateError,
    ResourceUnavailableError,
    TrackNotFoundError,
    ValidationFailedError,
)
from .exceptions import (
    Yakey2FAPictureExpiredError,
    Yakey2FAPictureInvalidError,
)


class BaseBundleAppView(BaseBundleView):

    def process_error(self, exception):
        """
        Этот метод скрывает большинство ошибок внутри,
        отдает наверх только интересные приложению ошибки,
        internal.temporary - если имеет смысл повторить запрос
        internal.permanent - в остальных случаях

        @param exception: Исключение для выдачи правильной диагностики полученной ошибки.
        @return: Объект JsonLoggedResponse.
        """
        processed_error = super(BaseBundleAppView, self).process_error(exception)

        # Эти ошибки мы отдаем приложению
        if isinstance(
            processed_error,
            (
                AccountDisabledError,
                AccountNotFoundError,
                InternalTemporaryError,
                TrackNotFoundError,
                ValidationFailedError,
                Yakey2FAPictureExpiredError,
                Yakey2FAPictureInvalidError,
            ),
        ):
            return processed_error
        elif isinstance(processed_error, ResourceUnavailableError):
            return InternalTemporaryError()
        elif isinstance(processed_error, InvalidTrackStateError):
            # К нам пришли не с тем треком. Отдадим как internal.permanent, но в лог писать не будем
            return InternalPermanentError()
        # Остальные исключения отдаем как internal.permanent. Залогируем перед этим как необработанное
        log_internal_error(exception)
        return InternalPermanentError()

    def process_request(self):
        self.response_values.update(server_time=int(datetime_to_unixtime(datetime.now())))
