# -*- coding: utf-8 -*-
from passport.backend.api.common.authorization import is_session_created
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    InvalidTrackStateError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
    BundleAccountResponseRendererMixin,
    BundlePhoneMixin,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


class OtpDisableBase(BaseBundleView,
                     BundlePhoneMixin,
                     BundleAccountGetterMixin,
                     BundleAccountPropertiesMixin,
                     BundleAccountResponseRendererMixin):

    required_grants = ['otp.edit']

    track_type = 'authorize'

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            uid=self.account.uid,
            consumer=self.consumer,
            track_id=self.track.track_id,
            mode='disable_otp',
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
            origin=self.track.origin,
        )

    def check_track(self):
        if not self.track.is_it_otp_disable:
            raise InvalidTrackStateError('Track is not for otp disable process')
        if is_session_created(self.track):
            raise ActionNotRequiredError('Already disabled otp on this track')
        if not self.track.uid:
            raise InvalidTrackStateError('No uid in track')

    def assert_otp_is_enabled(self):
        if not self.account.totp_secret.is_set:
            raise ActionNotRequiredError()

    def get_account(self, need_phones=False):
        try:
            self.get_account_from_session(
                multisession_uid=int(self.track.uid),
                emails=True,
                check_disabled_on_deletion=True,
                need_phones=need_phones,
            )
        except UidNotInSessionError:
            raise InvalidTrackStateError('Uid %s from track not found in session' % self.track.uid)

    def process(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self, *args, **kwargs):
        self.process()
