# -*- coding: utf-8 -*-
import logging

from passport.backend.api.common.authorization import login_password_ok
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountNotFoundError,
    BlackboxPermanentError,
    BlackboxUnavailableError,
    CaptchaRequiredAndPasswordNotMatchedError,
    CaptchaRequiredError,
    PasswordNotMatchedError,
)
from passport.backend.api.views.bundle.mixins import BundleAssertCaptchaMixin
from passport.backend.api.views.bundle.states import (
    OtpAuthFinished,
    OtpAuthNotReady,
)
from passport.backend.core import (
    authtypes,
    validators,
)
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS,
    BLACKBOX_LOGIN_DISABLED_STATUS,
    BLACKBOX_LOGIN_NOT_FOUND_STATUS,
    BLACKBOX_LOGIN_UNKNOWN_STATUS,
    BLACKBOX_LOGIN_VALID_STATUS,
    BLACKBOX_PASSWORD_BAD_STATUS,
    BLACKBOX_PASSWORD_UNKNOWN_STATUS,
)

from .base import OtpDisableBase


log = logging.getLogger('passport.api.view.bundle.otp.disable')


class OtpDisableCheckOtpForm(validators.Schema):
    otp = validators.String(not_empty=True, if_missing=None)


class OtpDisableCheckOtpView(OtpDisableBase,
                             BundleAssertCaptchaMixin):

    basic_form = OtpDisableCheckOtpForm

    require_track = True

    def check_otp(self, otp):
        # TODO: Возможно стоит объединить в будущем этот код и код похода в ЧЯ
        # на обычной авторизации. Они очень похожи, с небольшими особенностями.
        if not self.track.blackbox_login_status:
            bb_response = self.blackbox.login(
                otp,
                self.client_ip,
                uid=self.account.uid,
                useragent=self.user_agent,
                referer=self.referer,
                yandexuid=self.cookies.get('yandexuid'),
                authtype=authtypes.AUTH_TYPE_VERIFY,
                dbfields=[],  # Не выбирать из ЧЯ доп информацию о пользователе
                attributes=[],  # Аналогично выше
            )

            self.track.blackbox_login_status = bb_response['login_status']
            self.track.blackbox_password_status = bb_response['password_status']
            self.track.bruteforce_status = bb_response['bruteforce_status']

        is_password_ok = login_password_ok(self.track.blackbox_login_status, self.track.blackbox_password_status)

        # Сначала проверим ответ ЧЯ - что не нужно показывать капчу (защита от брутфорса №2)
        self.track.is_captcha_required = (self.track.bruteforce_status == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS)
        is_captcha_ok = self.is_captcha_passed if self.track.is_captcha_required else True

        # И только потом (sic!) проверим логин, пароль и капчу
        if is_password_ok and is_captcha_ok:
            self.track.is_captcha_required = False
            return  # Это успех

        # Запоминаем прошел ли человек капчу, не важно нужна она была или нет
        captcha_passed = self.is_captcha_passed
        if self.track.is_captcha_required:
            # Если нас брутфорсят - нужно показать ещё одну капчу
            self.invalidate_captcha()

        # Далее идут только ошибки: специфицируем ошибку авторизиции
        if (self.track.bruteforce_status == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS and
                self.track.blackbox_password_status == BLACKBOX_PASSWORD_BAD_STATUS and captcha_passed):
            raise CaptchaRequiredAndPasswordNotMatchedError()
        elif self.track.bruteforce_status == BLACKBOX_BRUTEFORCE_CAPTCHA_STATUS:
            raise CaptchaRequiredError()
        elif (self.track.blackbox_login_status == BLACKBOX_LOGIN_VALID_STATUS and
                self.track.blackbox_password_status == BLACKBOX_PASSWORD_BAD_STATUS):
            raise PasswordNotMatchedError('Bad otp')
        elif self.track.blackbox_login_status == BLACKBOX_LOGIN_DISABLED_STATUS:
            # Раньше мы здесь обрабатывали случай с заблокированным
            # пользователем, у которого есть подписка на блокирующие SIDы
            # (e.g. Деньги). Теперь это отлавливается при парсинге аккаунта.
            raise AccountDisabledError('Account is disabled')
        elif self.track.blackbox_login_status == BLACKBOX_LOGIN_NOT_FOUND_STATUS:
            raise AccountNotFoundError('Account is not found')
        elif (self.track.blackbox_password_status == BLACKBOX_PASSWORD_UNKNOWN_STATUS or
              self.track.blackbox_login_status == BLACKBOX_LOGIN_UNKNOWN_STATUS):
            self.track.blackbox_login_status = ''
            self.track.blackbox_password_status = ''
            self.track.bruteforce_status = ''
            raise BlackboxUnavailableError('Unknown account status')

        # Вообще-то сюда мы не должны никак попасть, а если попали,
        # то ЧЯ возвращает что-то невменяемое. Запишем в лог и упадем.
        log.info(
            'Blackbox returned login_status="%s" password_status="%s" bruteforce_status="%s"',
            self.track.blackbox_login_status,
            self.track.blackbox_password_status,
            self.track.bruteforce_status,
        )
        raise BlackboxPermanentError()

    def process(self):

        self.read_track()
        self.process_basic_form()
        self.response_values['track_id'] = self.track_id

        self.check_track()
        self.check_track_for_captcha()
        self.get_account(need_phones=True)

        self.assert_otp_is_enabled()
        self.state = OtpAuthNotReady()
        with self.track_transaction.commit_on_error():
            otp = self.form_values.get('otp')
            if otp:
                # Если otp нам был передан(введен пользователем), то
                # сбрасываем предыдущий результат проверки в треке
                self.track.blackbox_login_status = ''
                self.track.blackbox_password_status = ''
            else:
                # Иначе одноразовый пароль может лежать в треке, если сработала магия.
                if not self.track.otp:
                    # И если его нет говорим фронту, что не готовы ответить
                    return
                otp = self.track.otp
            self.check_otp(otp)

            self.track.is_otp_checked = True
            self.state = OtpAuthFinished()

            # Эти поля нужно сохранить для ручки проверки пароля
            self.track.login = self.account.login
            self.track.is_strong_password_policy_required = self.account.is_strong_password_required

            # Сохраним секьюрный номер в треке, для дальнейшего сравнения с паролем
            self.save_secure_number_in_track()

        self.statbox.log(action='otp_checked')
