# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.common.profile.profile import process_env_profile
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundlePasswordChangeMixin,
    BundlePasswordValidationMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE

from .base import OtpDisableBase


log = logging.getLogger('passport.api.view.bundle.otp.disable')


OTP_DISABLED_NOTIFICATION_TEMPLATE = 'mail/otp_disabled_notification.html'


class OtpDisableCommitForm(validators.Schema):
    password = validators.String(not_empty=True)

    revoke_web_sessions = validators.StringBool(if_missing=True, strip=True)
    revoke_tokens = validators.StringBool(if_missing=True, strip=True)
    revoke_app_passwords = validators.StringBool(if_missing=True, strip=True)


class OtpDisableCommitView(OtpDisableBase,
                           BundlePasswordChangeMixin,
                           BundlePasswordValidationMixin,
                           BundleBaseAuthorizationMixin,
                           BundlePushMixin,
                           KolmogorMixin,
                           MailMixin,
                           UserMetaDataMixin):

    basic_form = OtpDisableCommitForm

    require_track = True

    def check_track(self):
        super(OtpDisableCommitView, self).check_track()
        if not self.track.is_otp_checked:
            raise InvalidTrackStateError('Otp was not checked')

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = OTP_DISABLED_NOTIFICATION_TEMPLATE
        info = MailInfo(
            subject=translations['2fa_disabled.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
        )
        return template_name, info, context

    def send_email_notifications(self):
        language = get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

    def process(self):

        self.process_basic_form()
        self.read_track()
        self.response_values['track_id'] = self.track_id

        self.check_track()

        self.get_account()

        self.assert_otp_is_enabled()

        is_strong_policy = self.account.is_strong_password_required

        with self.track_transaction.commit_on_error():
            self.fill_track_with_account_data(password_passed=True)
            self.set_old_session_track_fields(self.session_info)

            password, quality = self.validate_password(
                is_strong_policy=is_strong_policy,
                required_check_password_history=True,
                emails=set([email.address for email in self.account.emails.native]),
            )

            events = {
                'action': 'disable_otp',
                'consumer': self.consumer,
            }

            with UPDATE(self.account, self.request.env, events):
                self.account.totp_secret = None
                self.change_password(
                    password,
                    quality,
                    # Так как мы здесь создаем новый пароль, то здесь у нас точно
                    # не будет странных атрибутов.
                    global_logout=is_strong_policy,
                    revoke_web_sessions=self.form_values['revoke_web_sessions'],
                    revoke_tokens=self.form_values['revoke_tokens'],
                    revoke_app_passwords=self.form_values['revoke_app_passwords'],
                )
                process_env_profile(self.account, track=self.track)

            # Если человек воспользовался otp-магией, в треке будет заполнено это поле
            otp_magic_passed = (self.track.otp is not None)

            self.fill_response_with_account_and_session(
                cookie_session_info=self.session_info,
                is_2fa_enabled_yp=False,
                otp_magic_passed=otp_magic_passed,
                personal_data_required=True,
                session_scope=SessionScope.xsession,
            )
            self.send_account_modification_push(
                event_name='login_method_change',
                context=dict(track_id=self.track_id),
            )
            self.send_email_notifications()
            self.send_account_modification_mail(
                event_name='login_method_change',
            )

        self.response_values['retpath'] = self.track.retpath
        self.send_account_modification_push(event_name='changed_password')
        self.statbox.log(
            action='disabled',
        )
