# -*- coding: utf-8 -*-
from passport.backend.api.common.authorization import is_session_created
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionNotRequiredError,
    InvalidTrackStateError,
    SecurePhoneNotFoundError,
    UidNotInSessionError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundlePhoneMixin,
)
from passport.backend.core.builders.social_api import (
    BaseSocialApiError,
    get_social_api,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


class OtpEnableBase(BaseBundleView,
                    BundlePhoneMixin,
                    BundleAccountGetterMixin,
                    BundleAccountResponseRendererMixin):

    required_grants = ['otp.edit']

    track_type = 'authorize'

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            uid=self.account.uid,
            consumer=self.consumer,
            track_id=self.track.track_id,
            mode='enable_otp',
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
            origin=self.track.origin,
        )

    def check_track(self):
        if self.track.session_reissue_interval:
            session_reissue_interval = int(self.track.session_reissue_interval)
        else:
            session_reissue_interval = None
        if is_session_created(self.track, session_reissue_interval):
            raise ActionNotRequiredError()
        if not self.track.is_it_otp_enable:
            raise InvalidTrackStateError('Track is not for otp enable process')
        if not self.track.uid:
            raise InvalidTrackStateError('No uid in track')

    def assert_secure_phone_exist_and_confirmed(self):
        if not self.has_secure_number:
            raise SecurePhoneNotFoundError()
        if self.track.phone_confirmation_phone_number != self.secure_number.e164:
            raise InvalidTrackStateError('Confirmed phone is different to account secure phone')

    def assert_otp_is_disabled(self):
        if self.account.totp_secret.is_set:
            raise ActionNotRequiredError()

    def get_account(self):
        try:
            self.get_account_from_session(
                multisession_uid=int(self.track.uid),
                emails=True,
                check_disabled_on_deletion=True,
                need_phones=True,
            )
        except UidNotInSessionError:
            raise InvalidTrackStateError('Uid %s from track not found in session' % self.track.uid)

    def get_social_profiles(self):
        try:
            return get_social_api().get_profiles_by_uid(
                self.account.uid,
                allow_auth=True,
            )
        except BaseSocialApiError:
            return []

    def process(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self, *args, **kwargs):
        self.process()
