# -*- coding: utf-8 -*-
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    OtpNotMatchedError,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_EDIT_TOTP_STATUS_OK
from passport.backend.core.models.totp_secret import TotpSecretAutoId
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.types.totp_secret import TotpSecretType
from passport.backend.utils.time import get_unixtime

from .base import OtpEnableBase
from .forms import OtpEnableCheckOtpForm


class OtpEnableCheckOtpView(OtpEnableBase):

    require_track = True

    basic_form = OtpEnableCheckOtpForm

    def check_track(self):
        super(OtpEnableCheckOtpView, self).check_track()
        if not (self.track.totp_app_secret and self.track.totp_pin):
            raise InvalidTrackStateError('Secret was not generated')

    def process(self):

        self.process_basic_form()
        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_track()

        self.get_account()

        self.assert_otp_is_disabled()
        self.assert_secure_phone_exist_and_confirmed()

        with self.track_transaction.rollback_on_error():
            # Проверим одноразовый пароль, убедимся, что
            # пользователь правильно инициализировал приложение
            secret = TotpSecretType.build(self.track.totp_app_secret, self.track.totp_pin)
            with CREATE(TotpSecretAutoId(), self.request.env, {}) as auto_id:
                pass
            response = self.blackbox.edit_totp(
                operation='create',
                uid=self.account.uid,
                pin=secret.pin,
                secret_id=auto_id.id,
                secret=secret.base64_secret,
                otp=self.form_values['otp'],
            )
            if response.get('error') != BLACKBOX_EDIT_TOTP_STATUS_OK:
                self.track.invalid_otp_count.incr()
                self.response_values['invalid_otp_count'] = self.track.invalid_otp_count.get()

                if response.get('junk_secret_value'):
                    events = {
                        'consumer': self.consumer,
                        'action': 'enable_otp',
                    }
                    with UPDATE(self.account, self.request.env, events):
                        self.account.totp_junk_secret = response['junk_secret_value']

                raise OtpNotMatchedError('Bad otp')

            self.track.is_otp_checked = True
            self.track.totp_secret_ids = {auto_id.id: get_unixtime()}
            totp_check_time = response.get('totp_check_time')
            if totp_check_time:
                self.track.blackbox_totp_check_time = totp_check_time
            self.track.totp_secret_encrypted = response['secret_value']

        self.response_values['is_password_required'] = False
