# -*- coding: utf-8 -*-
from datetime import datetime
import json
import logging

from passport.backend.api.common.authorization import SessionScope
from passport.backend.api.views.bundle.auth.base import BundleBaseAuthorizationMixin
from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundleAssertCaptchaMixin,
    BundleDeviceInfoMixin,
    BundlePasswordChangeMixin,
)
from passport.backend.api.views.bundle.mixins.account import UserMetaDataMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.mail import MailMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.conf import settings
from passport.backend.core.historydb.events import EVENT_APP_KEY_INFO
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.totp_secret import TotpSecretType

from .base import OtpEnableBase


log = logging.getLogger('passport.api.view.bundle.otp.enable')

OTP_ENABLED_NOTIFICATION_TEMPLATE = 'mail/otp_enabled_notification.html'


class OtpEnableCommitView(OtpEnableBase,
                          BundleAssertCaptchaMixin,
                          BundlePasswordChangeMixin,
                          BundleBaseAuthorizationMixin,
                          BundleDeviceInfoMixin,
                          BundlePushMixin,
                          KolmogorMixin,
                          MailMixin,
                          UserMetaDataMixin):

    require_track = True

    def check_track(self):
        super(OtpEnableCommitView, self).check_track()
        if not self.track.is_otp_checked:
            raise InvalidTrackStateError('Otp not checked')
        if not self.track.totp_secret_encrypted:
            raise InvalidTrackStateError('Missing encrypted secret in track')
        self.check_track_for_captcha(log_fail_to_statbox=False)

    def bind_secret(self, secret, extra_events=None):
        events = {
            'consumer': self.consumer,
            'action': 'enable_otp',
        }
        if extra_events:
            events.update(extra_events)

        with UPDATE(self.account, self.request.env, events):
            self.account.totp_secret.set(secret)
            self.change_password(None)
            self.account.sms_2fa_on = False
            self.account.forbid_disabling_sms_2fa = False
            self.account.enable_app_password = True
            self.account.totp_secret.secret_ids = {
                int(id_): datetime.fromtimestamp(timestamp)
                for id_, timestamp in self.track.totp_secret_ids.items()
            }
            if self.track.blackbox_totp_check_time:
                self.account.totp_secret.check_time = self.track.blackbox_totp_check_time
            if self.track.totp_push_device_ids:
                self.account.totp_secret.yakey_device_ids = sorted(list(self.track.totp_push_device_ids.get()))

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = OTP_ENABLED_NOTIFICATION_TEMPLATE
        info = MailInfo(
            subject=translations['2fa_enabled.subject'],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
        )
        return template_name, info, context

    def send_email_notifications(self):
        language = get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

    def process(self):
        self.read_track()
        self.response_values['track_id'] = self.track_id

        self.check_track()

        self.get_account()
        self.assert_otp_is_disabled()
        self.assert_secure_phone_exist_and_confirmed()

        secret = TotpSecretType(
            encrypted_pin_and_secret=self.track.totp_secret_encrypted,
        )
        with self.track_transaction.commit_on_error():
            self.track.is_strong_password_policy_required = self.account.is_strong_password_required
            self.fill_track_with_account_data(password_passed=True)

            self.set_old_session_track_fields(self.session_info)

            # Если приложение вызывало ручку /account/short, то в треке сохранены параметры от приложения
            app_params = self.get_device_params_from_track()
            extra_events = {}
            if any(app_params.values()):
                self.statbox.bind(
                    **app_params
                )
                extra_events = {EVENT_APP_KEY_INFO: json.dumps(app_params, sort_keys=True)}

            self.bind_secret(secret, extra_events)

            self.fill_response_with_account_and_session(
                cookie_session_info=self.session_info,
                is_2fa_enabled_yp=True,
                personal_data_required=True,
                session_scope=SessionScope.xsession,
            )
            self.send_email_notifications()
            self.send_account_modification_push(
                event_name='login_method_change',
                context=dict(track_id=self.track_id),
            )
            self.send_account_modification_mail(
                event_name='login_method_change',
            )

        self.response_values['retpath'] = self.track.retpath
        self.statbox.log(
            action='enabled',
            pin_length=len(self.track.totp_pin),
        )
