# -*- coding: utf-8 -*-
from passport.backend.api.common.common import create_csrf_token
from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.mixins.password import BundlePinValidationMixin
from passport.backend.core.types.totp_secret import TotpSecretType

from .base import OtpEnableBase


MAX_SECRET_GENERATE_COUNT = 5


class OtpEnableGetSecretView(OtpEnableBase, BundlePinValidationMixin):

    require_track = True

    sensitive_fields = [
        'pin',
        'app_secret',
        'app_secret_container',
        'push_setup_secret',
    ]

    def check_track(self):
        super(OtpEnableGetSecretView, self).check_track()
        if not self.is_phone_confirmed_in_track():
            raise InvalidTrackStateError('Phone on this track was not confirmed')

    def generate_secret(self):
        error = None
        # При генерации секрета удостоверимся,
        # что пин проходит нашу валидацию
        for _ in range(MAX_SECRET_GENERATE_COUNT):
            try:
                totp_secret = TotpSecretType.generate()
                self.validate_pin(pin=totp_secret.pin)
                return totp_secret
            except ValidationFailedError as e:
                error = e
        raise error

    def process(self):

        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_track()

        self.get_account()
        self.assert_otp_is_disabled()
        self.assert_secure_phone_exist_and_confirmed()

        if not self.track.totp_app_secret:
            totp_secret = self.generate_secret()
            with self.track_transaction.commit_on_error():
                self.track.totp_pin = totp_secret.pin
                self.track.totp_app_secret = totp_secret.human_readable_app_secret
        else:
            totp_secret = TotpSecretType.build(self.track.totp_app_secret, self.track.totp_pin)

        with self.track_transaction.commit_on_error():
            self.track.push_setup_secret = create_csrf_token()

        self.response_values.update(
            pin=totp_secret.pin,
            app_secret=totp_secret.human_readable_app_secret,
            push_setup_secret=self.track.push_setup_secret,
            pin_length=len(totp_secret.pin),
            app_secret_container=totp_secret.build_container_for_yakey(uid=int(self.track.uid)),
            sensitive_fields=self.sensitive_fields,
        )
