# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.mixins import (
    BundleAccountPropertiesMixin,
    BundleFixPDDRetpathMixin,
)
from passport.backend.api.views.bundle.phone.helpers import dump_number
from passport.backend.api.views.bundle.states import PasswordChangeForbidden
from passport.backend.core.models.phones.phones import RemoveSecureOperation

from .base import OtpEnableBase
from .forms import OtpEnableSubmitForm


log = logging.getLogger('passport.api.view.bundle.otp.enable')


class OtpEnableSubmitView(OtpEnableBase, BundleFixPDDRetpathMixin, BundleAccountPropertiesMixin):
    require_track = False

    basic_form = OtpEnableSubmitForm

    def assert_account_policies(self):
        if self.account.is_pdd and not self.account.domain.can_users_change_password:
            return PasswordChangeForbidden()
        return self.check_have_password()

    def do_process(self):
        redirect_state = self.assert_account_policies()
        if redirect_state is not None:
            return redirect_state

        self.read_or_create_track(self.track_type)
        with self.track_transaction.commit_on_error():
            self.set_uid_to_track(self.account.uid)
            self.track.origin = self.form_values['origin'] or self.track.origin
            self.track.session_reissue_interval = self.form_values.get('session_reissue_interval')
            self.track.secure_phone_number = ''
            self.track.has_secure_phone_number = False
            self.track.is_it_otp_enable = True
            self.response_values['secure_number'] = None

            if self.account.person.birthday:
                self.track.birthday = str(self.account.person.birthday)
            if self.secure_number:
                self.track.has_secure_phone_number = True
                self.track.secure_phone_number = self.secure_number.e164
                secure_number_info = dump_number(self.secure_number)
                # Значит, что включена новая схема и телефоны перелиты
                operation = self.account.phones.secure.get_logical_operation(None)
                secure_number_info['is_deleting'] = type(operation) in [
                    RemoveSecureOperation,
                ]

                self.response_values['secure_number'] = secure_number_info

                # Так как мы пропускаем подтверждение телефона при приходе с восстановления,
                # то нужно вручную проставить подтвержденный номер.
                if self.track.is_otp_restore_passed:
                    self.track.phone_confirmation_phone_number = self.secure_number.e164
                    self.track.phone_confirmation_is_confirmed = True
                    self.statbox.log(action='skip_phone_confirmation')

            if self.form_values['retpath']:
                self.track.retpath = self.form_values['retpath']
                if self.account.is_pdd:
                    self.fix_pdd_retpath()

        self.fill_response_with_account(personal_data_required=True)
        self.response_values['profiles'] = self.get_social_profiles()
        self.statbox.log(action='submitted')

    def process(self):
        self.process_basic_form()
        self.get_account_from_session(
            check_disabled_on_deletion=True,
            need_phones=True,
        )
        self.assert_otp_is_disabled()
        if self.account.is_pdd:
            # Некоторые ПДД-пользователи не имеют права изменять свой пароль
            response = self.blackbox.hosted_domains(domain=self.account.domain.domain)
            self.account.parse(response)

        self.state = self.do_process()
        if self.state is not None:
            return

        self.response_values['track_id'] = self.track_id
        self.response_values['skip_phone_check'] = bool(self.track.is_otp_restore_passed)
