# -*- coding: utf-8 -*-
from passport.backend.api.common.authorization import is_session_created
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    ActionImpossibleError,
    ActionNotRequiredError,
    PasswordRequiredError,
)
from passport.backend.api.views.bundle.headers import (
    AUTHORIZATION_HEADERS,
    HEADER_CLIENT_USER_AGENT,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.utils.decorators import cached_property


class BaseOtpMigrateView(BaseBundleView, BundleAccountGetterMixin, BundleAccountPropertiesMixin):

    require_track = True
    required_headers = tuple(AUTHORIZATION_HEADERS) + (
        HEADER_CLIENT_USER_AGENT,
    )
    required_grants = ['otp.edit']
    track_type = 'authorize'

    def check_track(self):
        if is_session_created(self.track):
            raise ActionNotRequiredError()

    def assert_otp_is_enabled(self):
        if not self.account.totp_secret.is_set:
            raise ActionImpossibleError('2fa is off')
        if len(self.account.totp_secret.secret_ids) != 1:
            raise ActionImpossibleError('Selecting secret for migration is not implemented yet')

    def assert_password_checked(self):
        if self.is_password_verification_required():
            raise PasswordRequiredError()

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            uid=self.account.uid,
            consumer=self.consumer,
            track_id=self.track.track_id,
            mode='migrate_otp',
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
        )

    def process_request(self):
        self.read_track()
        self.response_values['track_id'] = self.track_id
        self.check_track()
        self.get_pinned_account_from_session(emails=True)
        self.assert_otp_is_enabled()
        self.assert_password_checked()

        self.process()

    def process(self):
        raise NotImplementedError()  # pragma: no cover
