# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.api.views.bundle.exceptions import (
    InvalidTrackStateError,
    OtpNotMatchedError,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_EDIT_TOTP_STATUS_OK
from passport.backend.core.models.totp_secret import TotpSecretAutoId
from passport.backend.core.runner.context_managers import (
    CREATE,
    UPDATE,
)
from passport.backend.core.types.totp_secret import TotpSecretType
from passport.backend.utils.time import datetime_to_integer_unixtime

from .base import BaseOtpMigrateView
from .forms import OtpMigrateCheckOtpForm


class OtpMigrateCheckOtpView(BaseOtpMigrateView):

    basic_form = OtpMigrateCheckOtpForm

    def check_track(self):
        super(OtpMigrateCheckOtpView, self).check_track()
        if not self.track.totp_app_secret:
            raise InvalidTrackStateError('Secret was not generated')

    def process(self):
        self.process_basic_form()

        with self.track_transaction.commit_on_error():
            old_secret_id = list(self.account.totp_secret.secret_ids.keys())[0]

            # Проверим одноразовый пароль, убедимся, что
            # пользователь правильно инициализировал приложение
            secret = TotpSecretType.build(self.track.totp_app_secret, pin=0)  # пин не нужен
            with CREATE(TotpSecretAutoId(), self.request.env, {}) as auto_id:
                pass
            response = self.blackbox.edit_totp(
                operation='replace',
                uid=self.account.uid,
                old_secret_id=old_secret_id,
                new_secret_id=auto_id.id,
                secret=secret.base64_secret,
                otp=self.form_values['otp'],
            )
            if response.get('error') != BLACKBOX_EDIT_TOTP_STATUS_OK:
                self.track.invalid_otp_count.incr()
                self.response_values['invalid_otp_count'] = self.track.invalid_otp_count.get()

                if response.get('junk_secret_value'):
                    events = {
                        'consumer': self.consumer,
                        'action': 'migrate_otp',
                    }
                    with UPDATE(self.account, self.request.env, events):
                        self.account.totp_junk_secret = response['junk_secret_value']

                raise OtpNotMatchedError('Bad otp')

            self.track.is_otp_checked = True

            totp_secret_ids = self.account.totp_secret.secret_ids.copy()
            totp_secret_ids.pop(old_secret_id)
            totp_secret_ids[auto_id.id] = datetime.now()
            self.track.totp_secret_ids = {
                id_: datetime_to_integer_unixtime(dt)
                for id_, dt in totp_secret_ids.items()
            }

            totp_check_time = response.get('totp_check_time')
            if totp_check_time:
                self.track.blackbox_totp_check_time = totp_check_time
            self.track.totp_secret_encrypted = response['secret_value']
