# -*- coding: utf-8 -*-
from datetime import datetime
import json

from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundleDeviceInfoMixin,
    BundlePasswordChangeMixin,
)
from passport.backend.core.historydb.events import EVENT_APP_KEY_INFO
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.totp_secret import TotpSecretType

from .base import BaseOtpMigrateView


class OtpMigrateCommitView(BaseOtpMigrateView,
                           BundlePasswordChangeMixin,
                           BundleDeviceInfoMixin):

    def check_track(self):
        super(OtpMigrateCommitView, self).check_track()
        if not self.track.is_otp_checked:
            raise InvalidTrackStateError('Otp not checked')
        if not self.track.totp_secret_encrypted:
            raise InvalidTrackStateError('Encrypted secret not in track')

    def bind_secret(self, secret, extra_events=None):
        events = {
            'consumer': self.consumer,
            'action': 'migrate_otp',
        }
        if extra_events:
            events.update(extra_events)

        with UPDATE(self.account, self.request.env, events):
            self.account.totp_secret.set(secret)
            self.account.totp_secret.secret_ids = {
                int(id_): datetime.fromtimestamp(timestamp)
                for id_, timestamp in self.track.totp_secret_ids.items()
            }
            if self.track.blackbox_totp_check_time:
                self.account.totp_secret.check_time = self.track.blackbox_totp_check_time

    def process(self):
        secret = TotpSecretType(
            encrypted_pin_and_secret=self.track.totp_secret_encrypted,
        )
        with self.track_transaction.commit_on_error():
            self.response_values.update(retpath=self.track.retpath)
            # Если приложение вызывало ручку /account/short, то в треке сохранены параметры от приложения
            app_params = self.get_device_params_from_track()
            extra_events = {}
            if any(app_params.values()):
                self.statbox.bind(app_params)
                extra_events = {EVENT_APP_KEY_INFO: json.dumps(app_params, sort_keys=True)}

            self.bind_secret(secret, extra_events)
            # TODO: отправить письмо

        self.statbox.log(
            action='migrated',
            pin_length=self.account.totp_secret.pin_length,
        )
