# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins.password import looks_like_otp
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import get_unixtime

from .base import BaseBundleAppView
from .exceptions import (
    Yakey2FAPictureExpiredError,
    Yakey2FAPictureInvalidError,
)
from .forms import OtpPrepareAuthForm


log = logging.getLogger('passport.otp.prepare')


class OtpPrepareAuthView(BaseBundleAppView):

    require_track = True

    track_type = 'authorize'

    required_grants = ['otp.app']

    basic_form = OtpPrepareAuthForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            track_id=self.track.track_id,
            ip=self.client_ip,
        )

    def process_request(self):
        self.process_basic_form()
        self.read_track()
        if self.track.track_type != self.track_type or not self.track.is_allow_otp_magic:
            raise InvalidTrackStateError()

        if self.form_values['selected_2fa_picture']:
            if not self.track.correct_2fa_picture or not self.track.correct_2fa_picture_expires_at:
                raise InvalidTrackStateError()
            if get_unixtime() > int(self.track.correct_2fa_picture_expires_at):
                raise Yakey2FAPictureExpiredError()
            if self.form_values['selected_2fa_picture']  != int(self.track.correct_2fa_picture):
                with self.track_transaction.rollback_on_error():
                    self.track.correct_2fa_picture_expires_at = get_unixtime()
                raise Yakey2FAPictureInvalidError()

        if not looks_like_otp(self.form_values['otp']):
            log.info('Got malformed OTP: %r', self.form_values['otp'])

        with self.track_transaction.commit_on_error():
            self.track.otp = self.form_values['otp']
            self.track.login = self.form_values['login']
            self.track.blackbox_login_status = ''
            self.track.blackbox_password_status = ''
            self.track.x_token_status = None
            self.track.cred_status = None
        self.statbox.log(action='otp_auth_prepared')
        super(OtpPrepareAuthView, self).process_request()
