# -*- coding: utf-8 -*-
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    ActionNotRequiredError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.perimeter_api import get_perimeter_api
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import Account
from passport.backend.core.models.rfc_totp_secret import RfcTotpSecret
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.rfc_totp_secret import RfcTotpSecretType
from passport.backend.core.utils.decorators import cached_property

from .forms import (
    RfcOtpManageForm,
    RfcOtpSetCheckTimeForm,
)
from .helpers import build_rfc_totp_url


class BaseManageRfcOtpView(BaseBundleView, BundleAccountGetterMixin):
    require_track = False
    required_grants = ['rfc_otp.manage']
    basic_form = RfcOtpManageForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            uid=self.account.uid,
            login=self.account.login,  # для я-тима скорее удобнее работать с логинами
            consumer=self.consumer,
            mode='rfc_otp',
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
        )

    def check_preconditions(self):
        raise NotImplementedError()  # pragma: no cover

    def update_account(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self):
        self.process_basic_form()
        self.get_account_by_login(login=self.form_values['login'])
        self.check_preconditions()
        self.update_account()


class RfcOtpEnableView(BaseManageRfcOtpView):
    """Ручка включения rfc 2fa, которую дёргает периметр"""

    def check_preconditions(self):
        # Даже если rfc_2fa уже якобы включён, всё равно перезапишем секрет. Периметру виднее.
        if not self.account.password.is_set:
            raise AccountWithoutPasswordError()

    def update_account(self):
        events = {'action': 'enable_rfc_otp'}
        totp_secret = RfcTotpSecretType.generate()
        with UPDATE(self.account, self.request.env, events):
            self.account.rfc_totp_secret.set(totp_secret)
        self.statbox.log(action='enabled')
        self.response_values.update(secret=totp_secret.human_readable_secret)


class RfcOtpDisableView(BaseManageRfcOtpView):
    """Ручка выключения rfc 2fa, которую дёргает периметр"""

    def check_preconditions(self):
        if not self.account.rfc_totp_secret.is_set:
            raise ActionNotRequiredError()

    def update_account(self):
        events = {'action': 'disable_rfc_otp'}
        with UPDATE(self.account, self.request.env, events):
            self.account.rfc_totp_secret = None
        self.statbox.log(action='disabled')


class RfcOtpSetCheckTimeView(BaseBundleView):
    """
    Сейчас ручка используется oauth для установки времени
    проверки одноразового пароля в grant_type=password
    """
    require_track = False
    required_grants = ['rfc_otp.set_check_time']
    basic_form = RfcOtpSetCheckTimeForm

    def process_request(self):
        self.process_basic_form()

        account = Account(uid=self.form_values['uid'])
        account.password = None  # FIXME: костыль, нужно для fix_passport_login_rule
        account.rfc_totp_secret = RfcTotpSecret(parent=account)
        events = {'action': 'set_rfc_otp_check_time'}
        with UPDATE(account, self.request.env, events):
            account.rfc_totp_secret.check_time = self.form_values['totp_check_time']


class RecreateRfcOtpSecretView(BaseBundleView, BundleAccountGetterMixin):
    """Ручка выдачи секрета для включения rfc 2fa, которую дёргает фронт паспорта"""

    require_track = False
    required_grants = ['rfc_otp.recreate_secret']

    required_headers = (
        HEADER_CLIENT_HOST,
        HEADER_CLIENT_COOKIE,
        HEADER_CLIENT_USER_AGENT,
        HEADER_CONSUMER_CLIENT_IP,
    )

    def process_request(self):
        self.get_account_from_session()

        if not self.account.password.is_set:
            raise AccountWithoutPasswordError()

        perimeter_api = get_perimeter_api()
        secret = perimeter_api.recreate_totp_secret(self.account.login)['secret']

        totp_url = build_rfc_totp_url(self.account.login, settings.RFC_OTP_ISSUER, secret)

        self.response_values.update(
            totp_url=totp_url,
            secret=secret,
        )
