# -*- coding: utf-8 -*-
from bisect import bisect
import re

from passport.backend.api.views.bundle.exceptions import InvalidTrackStateError
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleDeviceInfoMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.counters.userinfo_counter import get_short_userinfo_bucket
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import escape_special_chars_to_unicode_entities

from .base import BaseBundleAppView
from .forms import ShortInfoForm


AVATAR_SIZE_TO_CODE_MAPPING = {
    50: 'islands-50',
    68: 'islands-68',
    75: 'islands-75',
    84: 'islands-retina-middle',
    100: 'islands-retina-50',
    150: 'islands-150',
    200: 'islands-200',
}

AVATAR_SIZES_SORTED = sorted(AVATAR_SIZE_TO_CODE_MAPPING.keys())

DEFAULT_AVATAR_CODE = 'islands-75'

UNICODE_ENTITY_POST_PROCESS_RE = re.compile(r'\\\\u([0-9a-fA-F]{4})', re.U)


def unicode_entities_slash_fix(re_match):
    return r'\u%s' % re_match.group(1)


class ShortInfoView(BaseBundleAppView,
                    BundleAccountGetterMixin,
                    BundleDeviceInfoMixin):

    require_track = False

    required_grants = ['otp.app']

    basic_form = ShortInfoForm

    @cached_property
    def statbox(self):
        return StatboxLogger(
            ip=self.client_ip,
            uid=self.account.uid,
        )

    def get_avatar_code(self):
        # Производим маппинг переданных размеров аватара
        # в кодировку размеров соответствующих YaPic:
        # https://wiki.yandex-team.ru/Yapic#razmerysushhestvujushhixizobrazhenijj
        # по принципу - выдать размер из имеющихся не превосходящий запрошенный
        avatar_size = self.form_values['avatar_size']
        if not avatar_size:
            return DEFAULT_AVATAR_CODE
        try:
            avatar_size = int(avatar_size)
            sizes = AVATAR_SIZES_SORTED
            if avatar_size < sizes[0]:
                return AVATAR_SIZE_TO_CODE_MAPPING[sizes[0]]
            return AVATAR_SIZE_TO_CODE_MAPPING[sizes[bisect(sizes, avatar_size) - 1]]
        except ValueError:
            return DEFAULT_AVATAR_CODE

    def get_avatar_url(self):
        if self.account.person.default_avatar:
            return settings.GET_AVATAR_URL % (
                self.account.person.default_avatar,
                self.get_avatar_code(),
            )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()

        # Не надо проверять и увеличивать счетчики если пришел track_id
        if self.track_id:
            self.read_track()
            if not self.track.is_it_otp_enable:  # Нужен трек для включения или миграции 2ФА
                raise InvalidTrackStateError('Track is not for otp enable process')

            self.get_account_by_login(self.form_values['login'])

            if int(self.track.uid) != self.account.uid:  # Нужен трек для указанного пользователя
                raise InvalidTrackStateError('Track is for different account')

        else:
            counter = get_short_userinfo_bucket()
            # PASSP-10339 При слишком частых запросах - деградируем и отдаем только серверное время
            if counter.hit_limit_by_ip(self.client_ip):
                return super(ShortInfoView, self).process_request()

            self.get_account_by_login(self.form_values['login'])

            counter.incr(self.client_ip)

        encoded_display_name = escape_special_chars_to_unicode_entities(self.account.person.display_name.name)
        self.response_values.update(
            display_name=encoded_display_name,
            avatar_url=self.get_avatar_url(),
            uid=self.account.uid,
            **{'2fa': bool(self.account.totp_secret.is_set)}
        )

        if self.track:
            # Запишем входные параметры в соответствующие поля трека
            self.save_device_params_to_track()

        super(ShortInfoView, self).process_request()

    def respond_success(self):
        """
        PASSP-10339 Этот хак позволяет отправлять в json-ответе неизмененными выражения вида "\u003c"
        Эти последовательности будут восприняты браузером как unicode-entity при разборе строки
        Обычно json.dump() экранирует одиночный символ "\" в "\\", поскольку это правильно с т/з JSON-спецификации
        Здесь мы проходимся по всему тексту json-ответа и заменяем последовательности вида
        r'\\u003c' на последовательности вида r'\u003c'

        :ЗАЧЕМ: По требованию Безопастников, пользовательский ввод, передаваемый в json-объектах должен быть
        экранирован через unicode-entities - тогда он не интерпретируется браузером клиента

        :return: Объект ответа АПИ со статус-кодом и тестом ответа
        """
        response = super(ShortInfoView, self).respond_success()
        response.data = UNICODE_ENTITY_POST_PROCESS_RE.sub(unicode_entities_slash_fix, response.data)
        return response
