# -*- coding: utf-8 -*-

import logging
from time import time

from passport.backend.api.common.account_manager import (
    AmPlatform,
    get_am_platform,
)
from passport.backend.api.common.phone import CONFIRM_METHOD_BY_SMS
from passport.backend.api.common.yasms import generate_fake_global_sms_id
from passport.backend.api.views.bundle.constants import (
    BIND_PHONE_OAUTH_SCOPE,
    SESSIONID_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.mixins.common import BundleTvmUserTicketMixin
from passport.backend.core.conf import settings
from passport.backend.core.grants import check_grant
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    StatboxLogger,
)
from passport.backend.core.types.account.account import ACCOUNT_TYPE_NORMAL
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.yasms.phonenumber_alias import PhoneAliasManager
from passport.backend.utils.time import datetime_to_unixtime

from . import (
    exceptions,
    helpers,
)
from .. import exceptions as basic_exceptions
from ..base import BaseBundleView
from ..headers import HEADER_CONSUMER_CLIENT_IP
from ..mixins import (
    BundleAccountGetterMixin,
    BundleAssertCaptchaMixin,
    BundlePhoneMixin,
    BundleVerifyPasswordMixin,
    SmsRetrieverMixin,
)
from .helpers import normalize_code


BASIC_GRANT = 'phone_bundle.base'
SECURE_GRANT = 'phone_bundle.bind_secure'
CONFIRM_BY_CALL_GRANT = 'phone_bundle.confirm_by_call'

ROUTE_GRANT_PREFIX = 'sms_routes'
ROUTE_GRANT_FORMAT = '{prefix}.{route}'

log = logging.getLogger('passport.api.view.bundle.phone')


class BasePhoneBundleView(BaseBundleView, BundlePhoneMixin, BundleTvmUserTicketMixin):
    """
    Базовый класс для методов телефонного бандла: submit и commit.

    Выполняет общую логику обработки запроса:
      * Валидация запроса с помощью формы (класс формы указывается в потомках)
      * Проверка хедера и гранта.
      * Создание нового или чтение существующего трека.
      * Запуск конечной логики обработки запроса.
    """

    required_headers = [HEADER_CONSUMER_CLIENT_IP]
    required_grants = [BASIC_GRANT]
    by_uid_grant = None
    required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE, SESSIONID_SCOPE]

    # FIXME: переименовать, потому что теперь используется и в статбоксе
    track_state = None

    statbox_step = None

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.track_state,
            track_id=self.track.track_id,
            uid=self.track.uid or '-',
            step=self.statbox_step,
            ip=self.client_ip,
            consumer=self.consumer,
            user_agent=self.user_agent,
        )

    @cached_property
    def antifraud_logger(self):
        # Аналогично PASSP-36049
        scenario = None
        if self.track.scenario:
            scenario = self.track.scenario
        elif self.track.track_type:
            scenario = self.track.track_type

        return AntifraudLogger(
            channel='pharma',
            sub_channel=self.consumer,
            status='OK',
            uid=self.track.uid,
            external_id='track-{}'.format(self.track_id),
            phone_confirmation_method=self.track.phone_confirmation_method,
            request_path=self.request.env.request_path,
            scenario=scenario,
            user_phone=self.number.e164,
        )

    # Объект PhoneNumber, построенный на основе номера и страны, указанных в запросе.
    @property
    def specified_number(self):
        return self.form_values['number']

    def process_request(self):
        if self.basic_form:
            self.process_basic_form()
        # TODO: подумать, какой тип лучше подходит в этих ручках. В каких процессах используются эти ручки?
        self.read_or_create_track('register')
        self.process()
        self.statbox.log(ok=1)

    def get_account_for_submit(self,
                               get_session_from_headers=False,
                               expected_uid=None):
        self.get_account_from_available_media(
            by_uid_grant=self.by_uid_grant,
            get_session_from_headers=get_session_from_headers,
            check_disabled_on_deletion=True,
            expected_uid=expected_uid,
            multisession_uid=expected_uid,
            emails=True,
            need_phones=True,
            required_scope=self.required_scopes,
            whitelisted_client_ids=settings.CLIENT_IDS_ALLOWED_TO_BIND_PHONE,
        )
        self.set_uid_to_track(self.account.uid)
        self.statbox.bind_context(uid=self.account.uid)

    def get_account_for_submit_v2(self):
        multisession_uid = int(self.track.uid) if self.track.uid else None
        try:
            self.get_account_for_submit(
                get_session_from_headers=True,
                expected_uid=multisession_uid,
            )
        except basic_exceptions.UidNotInSessionError:
            raise basic_exceptions.InvalidTrackStateError('Uid %s in track does not match uid in session' % multisession_uid)

    def get_account_for_commit(self):
        """
        Загружает данные аккаунта с помощью uid из трека.
        """
        self.get_account_from_track(emails=True, need_phones=True)

    def get_account_for_commit_v2(self):
        if not self.track.uid:
            raise basic_exceptions.InvalidTrackStateError('No uid in track')
        self.get_account_for_submit_v2()

    def process(self):
        raise NotImplementedError()

    def _process(self):
        raise NotImplementedError()


class PhoneValidationMixin(object):

    validation_period = None

    def assert_alias_absence(self):
        if self.account.phonenumber_alias.alias:
            raise exceptions.PhoneAliasExistError()

    def assert_alias_existence(self):
        if not self.account.phonenumber_alias.alias:
            raise exceptions.PhoneAliasNotFoundError()

    def assert_secure_phone_absense(self):
        if self.secure_number is not None:
            raise exceptions.SecurePhoneBoundAndConfirmedError()

    def assert_secure_phone_existence(self):
        if self.secure_number is None:
            raise basic_exceptions.SecurePhoneNotFoundError()

    def assert_is_not_secure(self):
        if self.secure_number and self.secure_number == self.number:
            raise exceptions.SecurePhoneBoundAndConfirmedError()

    def assert_secure_phone_not_changed(self):
        self.assert_secure_phone_existence()
        if self.account_phones.secure.number != self.number:
            raise basic_exceptions.SecurePhoneNotFoundError()

    def assert_secure_number_saved(self):
        """Ожидаем что в треке лежит защищенный телефон(флаг и сам номер)"""
        is_track_ok = (
            self.track.has_secure_phone_number and
            self.track.secure_phone_number
        )
        if not is_track_ok:
            raise basic_exceptions.InvalidTrackStateError()

    def assert_phone_existence(self):
        if not self.account_phones.find_confirmed(self.number):
            raise exceptions.PhoneNotFoundError()

    def set_validation_period(self):
        found_number = self.account_phones.find(self.number)
        if found_number:
            self.validation_period = int(
                time() - datetime_to_unixtime(found_number.validation_datetime),
            )
        else:
            self.validation_period = 0

    @cached_property
    def saved_secure_number(self):
        """Объект PhoneNumber, построенный на основе защищенного номера, сохранённого в треке"""
        number = self.track.secure_phone_number
        return PhoneNumber.parse(number)

    def mount_secure_bound_phone_number(self):
        """Выбирает для дальнейшей работы номер secure-телефона, привязанного к аккаунту."""
        self.assert_secure_phone_existence()

        self.number = self.account_phones.secure.number
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

    def mount_specified_number(self):
        """Выбирает для дальнейшей работы номер телефона, указанный в запросе."""
        self.number = self.specified_number
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

    def mount_alias_number(self):
        """Выбирает для дальнейшей работы номер телефона, привязанный как алиас."""
        self.number = self.account.phonenumber_alias.number
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

    def mount_saved_secure_number(self):
        """
        Выбирает для дальнейшей работы защищенный номер телефона, сохраненный в треке.
        """
        self.assert_secure_number_saved()

        self.number = self.saved_secure_number
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

    def mount_bank_number(self):
        """
        Выбирает для дальнейшей работы банковский номер телефона из аккаунта.
        """
        if not self.account.bank_phonenumber_alias:
            raise basic_exceptions.InvalidTrackStateError()

        self.number = PhoneNumber.parse(self.account.bank_phonenumber_alias.alias)
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)


class BasePhoneBundleSubmitter(BasePhoneBundleView, BundleAccountGetterMixin, BundleAssertCaptchaMixin,
                               PhoneValidationMixin, SmsRetrieverMixin):
    """
    Базовый класс обработки ручки submit.

    Содержит общую логику ручки, набор необходимых свойств и параметров, а также
    вспомогательные методы, для построения конечной логики работы в классах-потомках
    (например, метод проверка наличия телефона у аккаунта, методы выбора номера
    телефона для дальнейшей работы и прочие).

    Непосредственная логика описывается в потомках с помощью переопределения
    метода _process().
    """
    account = None
    number = None
    uid = None
    login_id = None
    statbox_step = 'submit'

    allow_states = []

    @property
    def can_handle_captcha(self):
        return False

    @property
    def mask_antifraud_denial(self):
        return settings.PHONE_CONFIRM_MASK_ANTIFRAUD_DENIAL

    @cached_property
    def confirmator(self):
        language = self.form_values['display_language']
        def sms_id_callback(sms_id):
            if self.consumer != 'mobileproxy':
                self.response_values['global_sms_id'] = sms_id

        return self.build_sending_phone_confirmator(
            previous_number=self.saved_number,
            sms_code_format=self.form_values.get('code_format'),
            sms_id_callback=sms_id_callback,
            sms_template=self._build_sms_template(language),
            can_handle_captcha=self.can_handle_captcha,
            mask_antifraud_denial=self.mask_antifraud_denial,
            login_id=self.login_id,
            phone_confirmation_language=language,
        )

    @cached_property
    def saved_number(self):
        """Объект PhoneNumber, построенный на основе номера, сохранённого в треке"""
        number = self.track.phone_confirmation_phone_number
        if not number:
            return
        return PhoneNumber.parse(number)

    @property
    def return_unmasked_number(self):
        return self.form_values.get('return_unmasked_number', True)

    def process(self):
        """
        Основной метод обработки запроса.

        Открывает commit-транзакцию на треке для сохранения промежуточных изменений
        в нём в случае выброса исключения, запускает конечную логику обработки в
        потомке и собирает данные для ответа (независимо от его успешности).
        """
        with self.track_transaction.commit_on_error():
            self.response_values['track_id'] = self.track.track_id

            # если в трэке нет стэйта, значит в это апи ещё не приходили и стэйт надо выставить
            if not self.track.state or self.track.state in self.allow_states:
                self.track.state = self.track_state

            # Проверяем, что стэйт трэка правильный
            if self.track_state != self.track.state:
                raise basic_exceptions.InvalidTrackStateError()

            if self.can_handle_captcha:
                self.check_track_for_captcha(log_fail_to_statbox=False)

            try:
                self._process()
                self.track.display_language = self.form_values['display_language']
            finally:
                if self.number:
                    self.response_values['number'] = helpers.dump_number(
                        self.number,
                        only_masked=not self.return_unmasked_number,
                    )

    def get_account(self, **kwargs):
        self.get_account_for_submit(**kwargs)

    def is_account_type_allowed(self):
        return self.account.type == ACCOUNT_TYPE_NORMAL

    def save_number(self):
        self.confirmator.save_number(self.number)

    def _get_identity(self, current_method):
        return current_method

    def check_route_grant(self, route):
        if route:
            check_grant(
                ROUTE_GRANT_FORMAT.format(prefix=ROUTE_GRANT_PREFIX, route=route),
                self.consumer_ip,
                self.consumer,
                self.service_ticket,
            )

    def send_code(self, current_method):
        """
        Запускает процесс отправки смс с кодом подтверждения.

        Кроме непосредственной отправки, внутри confirmator выполняются и другие
        операции: проверка лимитов/счётчиков, чтение и сохранение данных в трек.

        Поддерживаем SMS Retriever для Android: PASSP-18005

        Маршрут для СМС берем из параметра route, если он задан: PASSP-18828
        """
        if self.consumer != 'mobileproxy':
            self.response_values['global_sms_id'] = generate_fake_global_sms_id()
        gps_package_name = self.form_values.get('gps_package_name')
        self._bind_sms_retriever_data_to_phone_confirmator(self.confirmator, gps_package_name)

        identity = self._get_identity(current_method)

        route = self.form_values['route']
        self.check_route_grant(route)

        self.confirmator.send_code(self.number, identity, route=route)
        self.response_values.update(
            deny_resend_until=self.confirmator.deny_resend_until,
            code_length=len(normalize_code(self.track.phone_confirmation_code)),
        )

    def make_call(self):
        call_confirmator = self.build_calling_phone_confirmator(
            code_format=self.form_values['code_format'] or (
                'by_3_dash' if get_am_platform(self.track) == AmPlatform.IOS else 'by_3'
            ),
            phone_confirmation_language=self.form_values['display_language'],
            previous_number=self.saved_number,
            can_handle_captcha=self.can_handle_captcha,
            mask_antifraud_denial=self.mask_antifraud_denial,
            login_id=self.login_id,
        )
        call_confirmator.make_call(self.number)
        self.response_values.update(
            code_length=len(normalize_code(self.track.phone_confirmation_code)),
        )

    def make_flash_call(self):
        call_confirmator = self.build_flash_calling_phone_confirmator(
            previous_number=self.saved_number,
            can_handle_captcha=self.can_handle_captcha,
            mask_antifraud_denial=self.mask_antifraud_denial,
            login_id=self.login_id,
        )
        call_confirmator.make_call(self.number)

        template = PhoneNumber.parse(call_confirmator.calling_number, allow_impossible=True).masked_for_flash_call

        self.response_values.update(
            calling_number_template=template,
            code_length=len(normalize_code(self.track.phone_confirmation_code)),
        )


class BasePhoneBundleCommitter(BasePhoneBundleView, BundleAccountGetterMixin,
                               BundleVerifyPasswordMixin, PhoneValidationMixin,
                               BundleAssertCaptchaMixin):
    """
    Базовый класс обработки ручки commit.

    Похож на BasePhoneBundleSubmitter: также описывает вспомогательные методы для построения
    конечной логики работы в классах-потомках: привязка телефона к аккаунту,
    проверка пароля, отправка email-оповещений и так далее.

    В данном классе всегда должна быть реализована следующая схема проверок:
    - проверяем код (это дёшево и не связано с аккаунтом), если не требуется далее проверять пароль
    - загружаем аккаунт: проверяем на существование, заблокированность
    - проверяем у существующего аккаунта телефон

    или

    - загружаем аккаунт: проверяем на существование, заблокированность, алиасность, верифицируем пароль
    - проверяем код
    - проверяем у существующего аккаунта телефон

    Непосредственная логика описывается в потомках с помощью переопределения
    метода _process().
    """
    require_track = True

    statbox_step = 'commit'

    accept_fake_code_for_test_numbers = False

    @cached_property
    def confirmator(self):
        confirmator = helpers.VerifyingPhoneSmsConfirmator(
            track=self.track,
            confirmations_limit=settings.SMS_VALIDATION_MAX_CHECKS_COUNT,
            statbox=self.statbox,
            account=self.account,
            antifraud_logger=self.antifraud_logger,
            accept_fake_code_for_test_numbers=self.accept_fake_code_for_test_numbers,
        )
        return confirmator

    @cached_property
    def call_confirmator(self):
        confirmator = helpers.VerifyingPhoneCallConfirmator(
            track=self.track,
            confirmations_limit=settings.PHONE_VALIDATION_MAX_CALLS_CHECKS_COUNT,
            statbox=self.statbox,
            account=self.account,
            antifraud_logger=self.antifraud_logger,
            client=self.octopus_api,
        )
        return confirmator

    # Объект PhoneNumber, построенный на основе номера и страны, сохранённых в треке.
    @cached_property
    def saved_number(self):
        original_number = self.track.phone_confirmation_phone_number_original
        if not original_number:
            return
        return PhoneNumber.parse(original_number, self.track.country)

    # Короткий хелпер для доступа к объекту телефона, сохранённого в треке,
    # так как во втором шаге повторное указание телефона не требуется.
    @property
    def number(self):
        if not self.saved_number:
            raise basic_exceptions.InvalidTrackStateError()
        self.statbox.bind_context(number=self.saved_number.masked_format_for_statbox)
        return self.saved_number

    @property
    def return_unmasked_number(self):
        return self.form_values.get('return_unmasked_number', True)

    def fill_response_on_success(self):
        pass  # может перегружаться в потомках

    def process(self):
        """
        Основной метод обработки запроса.

        Открывает commit-транзакцию на треке для сохранения промежуточных изменений
        в нём в случае выброса исключения, запускает конечную логику обработки в
        потомке и собирает данные для ответа (независимо от его успешности).
        """
        with self.track_transaction.commit_on_error():
            self.response_values['number'] = helpers.dump_number(
                self.number,
                only_masked=not self.return_unmasked_number,
            )

            # Проверяем, что стэйт правильный
            if self.track_state != self.track.state:
                raise basic_exceptions.InvalidTrackStateError()

            # Если в треке уже есть этот флажок - значит, потребитель ретраит ручку.
            # Тогда осмысленную работу повторять уже не надо, а надо просто отдать такой же ответ,
            # как после первого вызова.
            if not self.track.is_successful_phone_passed:
                self._process()
                self.track.is_successful_phone_passed = True

            self.fill_response_on_success()

    def confirm_code(self, confirm_method=CONFIRM_METHOD_BY_SMS):
        if confirm_method == CONFIRM_METHOD_BY_SMS:
            confirmator = self.confirmator
        else:
            confirmator = self.call_confirmator

        confirmator.confirm_code(self.form_values['code'])

    def verify_account_password(self):
        # На треке уже выставлен флажок необходимости прохождения
        # капчи, но капча не пройдена
        if self.track.is_captcha_required and not self.track.is_captcha_recognized:
            raise basic_exceptions.CaptchaRequiredError()

        response = self.verify_password(
            self.track.uid,
            self.form_values['password'],
            emails=True,
            need_phones=True,
            check_disabled_on_deletion=True,
        )

        # Для каждой попытки должна быть новая капча,
        # в том числе даже при верном пароле.
        # Потому что у нас нет гарантии того,
        # как сделают следующий вызов данной ручки:
        # с верным или неверным паролем и верным или неверным кодом.
        self.invalidate_captcha()

        self.parse_account(response)

    def get_account(self):
        """
        Загружает данные аккаунта с помощью uid из трека.
        """
        self.get_account_for_commit()

    def is_account_type_allowed(self):
        return self.account.type == ACCOUNT_TYPE_NORMAL

    def aliasify_phone(self):
        previous_alias_owner = helpers.find_account_by_number(self.blackbox, self.number)
        reason = self._phone_alias_manager.ALIAS_DELETE_REASON_OWNER_CHANGE
        if previous_alias_owner:
            self._phone_alias_manager.delete_alias_with_logging(previous_alias_owner, reason=reason)

        self.statbox.bind(is_owner_changed=bool(previous_alias_owner))
        self._phone_alias_manager.create_alias_with_logging(
            self.account,
            self.number,
            language=self.track.display_language,
            validation_period=self.validation_period,
        )

    def dealiasify_phone(self):
        self.statbox.bind(validation_period=self.validation_period)
        self._phone_alias_manager.delete_alias_with_logging(
            self.account,
            reason=self._phone_alias_manager.ALIAS_DELETE_REASON_OFF,
            language=self.track.display_language,
        )

    @cached_property
    def _phone_alias_manager(self):
        return PhoneAliasManager(
            self.consumer,
            self.request.env,
            self.statbox,
            need_update_tx=True,
        )
