# -*- coding: utf-8 -*-
from datetime import timedelta
import logging
import time

from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.api.common.phone_karma import (
    get_phone_karma,
    PhoneKarma,
)
from passport.backend.api.common.processes import (
    PROCESS_LOGIN_RESTORE,
    PROCESS_RESTORE,
    PROCESS_WEB_REGISTRATION,
)
from passport.backend.api.common.yasms import generate_fake_global_sms_id
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.api.views.bundle.exceptions import (
    CallsShutDownError,
    CreateCallFailed,
    InvalidTrackStateError,
    PasswordNotMatchedError,
    PhoneCompromisedError,
)
from passport.backend.api.yasms.exceptions import YaSmsError
from passport.backend.core.builders.octopus import (
    BaseOctopusError,
    get_octopus,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.phones.phones import AliasifySecureOperation
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_FEDERAL,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import generate_random_code

from . import (
    forms,
    helpers,
)
from ..headers import (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
)
from ..mixins import (
    BundleAccountGetterMixin,
    KOLMOGOR_COUNTER_CALLS_FAILED,
    OctopusAvailabilityMixin,
    BundlePhoneMixin,
)
from .base import (
    BasePhoneBundleCommitter,
    BasePhoneBundleSubmitter,
    BasePhoneBundleView,
    BASIC_GRANT,
    CONFIRM_BY_CALL_GRANT,
    PhoneValidationMixin,
    SECURE_GRANT,
)
from .exceptions import (
    ConfirmationCallFinished,
    ConfirmationCallHangup,
    ConfirmationCallUnavailable,
    ConfirmationCheckTooLateError,
    ConfirmationFailed,
    ConfirmationNotReady,
    PhoneNotFoundError,
    SecureOperationExistsError,
)


log = logging.getLogger('passport.api.views.bundle.phone.controllers')

CONFIRM_STATE = 'confirm'
CONFIRM_BOUND_STATE = 'confirm_bound'
CONFIRM_AND_BIND_SECURE_STATE = 'confirm_and_bind_secure'
CONFIRM_AND_BIND_STATE = 'confirm_and_bind'
CONFIRM_AND_BIND_SECURE_AND_ALIASIFY_STATE = 'confirm_and_bind_secure_and_aliasify'
CONFIRM_BOUND_SECURE_AND_ALIASIFY_STATE = 'confirm_bound_secure_and_aliasify'
DELETE_ALIAS_STATE = 'delete_alias'
CONFIRM_TRACKED_SECURE_STATE = 'confirm_tracked_secure'
CONFIRM_AND_UPDATE_TRACKED_SECURE_STATE = 'confirm_and_update_tracked_secure'
PROLONG_VALID_STATE = 'prolongvalid'

REQUIRED_HEADERS_V2 = (
    HEADER_CLIENT_USER_AGENT,
    HEADER_CONSUMER_CLIENT_IP,
)


class CheckPhoneKarmaMixin(object):
    def check_phone_karma(self, number):
        if settings.USE_PHONE_KARMA and (
            self.track.process_name == PROCESS_WEB_REGISTRATION or
            self.track.is_force_change_password
        ):
            phone_karma = get_phone_karma(number, safe=True)
            self.statbox.log(
                action='check_phone_karma',
                number=number.masked_format_for_statbox,
                karma=phone_karma,
            )
            if not settings.PHONE_KARMA_DRY_RUN and phone_karma == PhoneKarma.black:
                raise PhoneCompromisedError()


class ConfirmSubmitter(
    BasePhoneBundleSubmitter,
    CheckPhoneKarmaMixin,
    OctopusAvailabilityMixin,
):
    basic_form = forms.ConfirmSubmitForm

    track_state = CONFIRM_STATE

    allow_states = [CONFIRM_TRACKED_SECURE_STATE]

    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    @property
    def can_handle_captcha(self):
        return self.consumer in settings.PHONE_CONFIRM_SHOW_ANTIFRAUD_CAPTCHA_FOR_CONSUMERS

    # Объект PhoneNumber, построенный на основе номера и страны, указанных в запросе или phone_id.
    @property
    def specified_number(self):
        result_number = self.form_values['number']
        if result_number is None:
            phone_id = self.form_values['phone_id']
            if self.account is not None and self.account.phones.has_id(phone_id):
                result_number = self.account.phones.by_id(phone_id).number
            else:
                raise PhoneNotFoundError()

        return result_number

    @property
    def return_unmasked_number(self):
        return self.form_values.get('return_unmasked_number', True) and self.form_values.get('number') is not None

    def is_account_type_allowed(self):
        # Тип аккаунта не проверяем: в этой ручке телефон не привязывается. А если переподтверждаем уже привязанный
        # телефон, то и тут ошибки быть не должно.
        return True

    def _process(self):
        if self.form_values['phone_id'] is not None:
            self.get_account_from_track(need_phones=True)
            self.track.return_masked_number = True

        confirmation_method = self.form_values.get('confirm_method', CONFIRM_METHOD_BY_SMS)
        if confirmation_method == CONFIRM_METHOD_BY_SMS and self.consumer != 'mobileproxy':
            self.response_values['global_sms_id'] = generate_fake_global_sms_id()

        self.mount_specified_number()
        self.check_phone_karma(self.number)

        if confirmation_method == CONFIRM_METHOD_BY_CALL:
            if not (
                self.track.phone_validated_for_call and
                PhoneNumber.parse(self.track.phone_validated_for_call).e164 == self.number.e164 and
                self.track.phone_valid_for_call
            ):
                raise InvalidTrackStateError()
            self.check_grant(CONFIRM_BY_CALL_GRANT)
            if not self.are_calls_available():
                raise CallsShutDownError()
            try:
                self.make_call()
            except BaseOctopusError:
                raise CreateCallFailed()
        elif confirmation_method == CONFIRM_METHOD_BY_FLASH_CALL:
            if not (
                self.track.phone_validated_for_call and
                PhoneNumber.parse(self.track.phone_validated_for_call).e164 == self.number.e164 and
                self.track.phone_valid_for_flash_call
            ):
                raise InvalidTrackStateError()
            self.check_grant(CONFIRM_BY_CALL_GRANT)
            if not self.are_calls_available():
                raise CallsShutDownError()
            try:
                self.make_flash_call()
            except BaseOctopusError:
                raise CreateCallFailed()
        else:
            self.send_code('confirm')
        self.track.phone_confirmation_method = confirmation_method


class ConfirmCommitter(BasePhoneBundleCommitter):
    basic_form = forms.ConfirmCommitForm

    track_state = CONFIRM_STATE

    allowed_processes = [
        PROCESS_WEB_REGISTRATION,
    ]

    accept_fake_code_for_test_numbers = True

    @property
    def return_unmasked_number(self):
        return self.form_values.get('return_unmasked_number', True) and not self.track.return_masked_number

    def _process(self):
        if not self.track.phone_confirmation_method:
            raise InvalidTrackStateError()
        if self.track.phone_call_session_id:
            self.check_grant(CONFIRM_BY_CALL_GRANT)

        self.confirm_code(confirm_method=self.track.phone_confirmation_method)
        self.statbox.bind_context(confirm_method=self.track.phone_confirmation_method)


class ConfirmCheckStatus(BaseBundleView, OctopusAvailabilityMixin):
    require_track = True
    required_grants = [BASIC_GRANT, CONFIRM_BY_CALL_GRANT]

    allowed_processes = [
        PROCESS_LOGIN_RESTORE,
        PROCESS_RESTORE,
        PROCESS_WEB_REGISTRATION,
    ]

    @cached_property
    def octopus_api(self):
        return get_octopus()

    def process_request(self):
        self.read_track()

        if not self.track.phone_call_session_id:
            raise InvalidTrackStateError()
        if not self.track.phone_confirmation_method:
            raise InvalidTrackStateError()
        if self.track.phone_confirmation_method not in (CONFIRM_METHOD_BY_CALL, CONFIRM_METHOD_BY_FLASH_CALL):
            raise InvalidTrackStateError()

        last_called = float(self.track.phone_confirmation_last_called_at or 0)
        if int(time.time()) - last_called > settings.PHONE_CONFIRMATION_CHECK_TIMEOUT:
            raise ConfirmationCheckTooLateError()

        try:
            call_status = helpers.get_call_status(self.octopus_api, self.track)
        except BaseOctopusError:
            self.inc_calls_counter(call_counter_key=KOLMOGOR_COUNTER_CALLS_FAILED)
            raise ConfirmationFailed()

        if call_status == settings.CALL_STATUS_IN_PROGRESS:
            raise ConfirmationNotReady()

        elif call_status in [
            settings.CALL_STATUS_REJECTED,
            settings.CALL_STATUS_NO_ANSWER,
        ]:
            raise ConfirmationCallHangup()

        elif call_status == settings.CALL_STATUS_UNAVAILABLE:
            if self.track.phone_confirmation_method == CONFIRM_METHOD_BY_CALL:
                raise ConfirmationCallHangup()
            elif self.track.phone_confirmation_method == CONFIRM_METHOD_BY_FLASH_CALL:
                raise ConfirmationCallUnavailable()

        elif call_status in [settings.CALL_STATUS_SUCCESS, settings.CALL_STATUS_INTERRUPTED]:
            raise ConfirmationCallFinished()


class ConfirmBoundSubmitter(BasePhoneBundleSubmitter):
    basic_form = forms.ConfirmBoundSubmitForm
    track_state = CONFIRM_BOUND_STATE
    by_uid_grant = 'phone_bundle.confirm_bound_by_uid'

    def is_account_type_allowed(self):
        # Тип аккаунта не проверяем: если аккаунт смог привязать секьюрный
        # телефон, должен и мочь его переподтвердить
        return True

    def _process(self):
        self.mount_specified_number()
        self.get_account()
        self.assert_phone_existence()
        self.send_code('confirm_bound')


class ConfirmBoundCommitter(BasePhoneBundleCommitter):
    basic_form = forms.ConfirmBoundCommitForm

    track_state = CONFIRM_BOUND_STATE

    def is_account_type_allowed(self):
        # Тип аккаунта не проверяем: если аккаунт смог привязать секьюрный
        # телефон, должен и мочь его переподтвердить
        return True

    def _process(self):
        self.get_account()
        self.confirm_code()
        self.assert_phone_existence()

        phone = self.account.phones.by_number(self.number)
        with UPDATE(self.account, self.request.env, {'action': 'confirm_bound', 'consumer': self.consumer}):
            phone.confirm()
        self.statbox.log(operation='update', number=self.number.masked_format_for_statbox)


class ConfirmAndBindSimpleSubmitter(BasePhoneBundleSubmitter):
    basic_form = forms.AccountedNumberedSubmitForm
    track_state = CONFIRM_AND_BIND_STATE
    by_uid_grant = 'phone_bundle.confirm_bound_by_uid'
    required_headers = REQUIRED_HEADERS_V2

    def is_account_type_allowed(self):
        return self.account.type in (
            ACCOUNT_TYPE_FEDERAL,
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NEOPHONISH,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        )

    def _process(self):
        self.mount_specified_number()
        self.get_account()
        self.assert_is_not_secure()
        self.send_code('confirm_and_bind')

    def get_account(self):
        self.get_account_for_submit_v2()


class ConfirmAndBindSimpleCommitter(BasePhoneBundleCommitter):
    basic_form = forms.ConfirmAndBindSimpleCommitForm
    track_state = CONFIRM_AND_BIND_STATE
    by_uid_grant = ConfirmAndBindSimpleSubmitter.by_uid_grant
    required_headers = REQUIRED_HEADERS_V2

    def is_account_type_allowed(self):
        return self.account.type in (
            ACCOUNT_TYPE_FEDERAL,
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NEOPHONISH,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        )

    def get_account(self):
        self.get_account_for_commit_v2()

    def _process(self):
        self.get_account()
        self.confirm_code()
        self.assert_is_not_secure()

        save_simple_phone = self.build_save_simple_phone(phone_number=self.number)
        save_simple_phone.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'confirm_and_bind', 'consumer': self.consumer},
        ):
            save_simple_phone.commit()

        save_simple_phone.after_commit()

    def fill_response_on_success(self):
        if not self.account:
            self.get_account()
        self.response_values['phone_id'] = self.account.phones.by_number(self.number).id


class ConfirmAndBindSecureSubmitter(BasePhoneBundleSubmitter):
    basic_form = forms.ConfirmAndBindSecureSubmitForm
    track_state = CONFIRM_AND_BIND_SECURE_STATE
    required_grants = [BASIC_GRANT, SECURE_GRANT]
    by_uid_grant = 'phone_bundle.confirm_and_bind_secure_by_uid'

    def is_account_type_allowed(self):
        # неофонишей не пускаем: секьюрный номер у них уже есть
        return self.account.type in (
            ACCOUNT_TYPE_FEDERAL,
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        )

    def _process(self):
        self.mount_specified_number()
        self.get_account()

        self.assert_secure_phone_absense()
        self.send_code('confirm_and_bind_secure')
        self.response_values['is_password_required'] = False


class ConfirmAndBindSecureCommitter(BasePhoneBundleCommitter):
    basic_form = forms.ConfirmAndBindSecureCommitForm

    track_state = CONFIRM_AND_BIND_SECURE_STATE

    required_grants = [BASIC_GRANT, SECURE_GRANT]

    def is_account_type_allowed(self):
        # неофонишей не пускаем: секьюрный номер у них уже есть
        return self.account.type in (
            ACCOUNT_TYPE_FEDERAL,
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        )

    def _process(self):
        self.get_account()
        self.confirm_code()
        self.assert_secure_phone_absense()

        save_secure_phone = self.build_save_secure_phone(
            phone_number=self.number,
            language=self.track.display_language,
            notification_language=self.track.display_language,
            should_notify_by_email=True,
        )
        save_secure_phone.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'confirm_and_bind_secure', 'consumer': self.consumer},
        ):
            save_secure_phone.commit()

        save_secure_phone.after_commit()


class ConfirmAndBindSecureAndAliasifySubmitter(BasePhoneBundleSubmitter):
    basic_form = forms.ConfirmAndBindSecureAndAliasifySubmitForm
    track_state = CONFIRM_AND_BIND_SECURE_AND_ALIASIFY_STATE
    required_grants = [BASIC_GRANT, SECURE_GRANT]
    by_uid_grant = 'phone_bundle.confirm_and_bind_secure_and_aliasify_by_uid'

    def is_account_type_allowed(self):
        # неофонишей не пускаем: секьюрный номер у них уже есть
        # ПДД и федералов не пускаем: им нельзя телефонный алиас
        allowed_account_types = {
            ACCOUNT_TYPE_NORMAL,
        }
        if not self.form_values['enable_alias_as_email']:
            allowed_account_types.update(
                {
                    ACCOUNT_TYPE_LITE,
                    ACCOUNT_TYPE_SOCIAL,
                },
            )
        return self.account.type in allowed_account_types

    def _process(self):
        self.mount_specified_number()
        self.get_account()

        self.assert_alias_absence()
        self.assert_secure_phone_absense()
        self.send_code('confirm_and_bind_secure_and_aliasify')

        self.track.enable_phonenumber_alias_as_email = self.form_values['enable_alias_as_email']

        self.response_values['is_password_required'] = False


class ConfirmAndBindSecureAndAliasifyCommitter(BasePhoneBundleCommitter):
    track_state = CONFIRM_AND_BIND_SECURE_AND_ALIASIFY_STATE

    required_grants = [BASIC_GRANT, SECURE_GRANT]

    basic_form = forms.ConfirmAndBindSecureAndAliasifyCommitForm

    def is_account_type_allowed(self):
        # неофонишей не пускаем: секьюрный номер у них уже есть
        # ПДД и федералов не пускаем: им нельзя телефонный алиас
        allowed_account_types = {
            ACCOUNT_TYPE_NORMAL,
        }
        if not self.track.enable_phonenumber_alias_as_email:
            allowed_account_types.update(
                {
                    ACCOUNT_TYPE_LITE,
                    ACCOUNT_TYPE_SOCIAL,
                },
            )
        return self.account.type in allowed_account_types

    def _process(self):
        self.get_account()
        self.confirm_code()
        self.assert_alias_absence()
        self.assert_secure_phone_absense()

        save_secure_phone = self.build_save_secure_phone(
            phone_number=self.number,
            language=self.track.display_language,
            aliasify=True,
            notification_language=self.track.display_language,
            should_notify_by_email=True,
            enable_search_alias=self.track.enable_phonenumber_alias_as_email,
        )
        self.set_validation_period()
        save_secure_phone.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'confirm_and_bind_secure_and_aliasify', 'consumer': self.consumer},
        ):
            save_secure_phone.commit(validation_period=self.validation_period)

        save_secure_phone.after_commit()


class ConfirmSecureBoundAndAliasifySubmitter(BasePhoneBundleSubmitter):
    basic_form = forms.ConfirmSecureBoundAndAliasifySubmitForm
    track_state = CONFIRM_BOUND_SECURE_AND_ALIASIFY_STATE
    by_uid_grant = 'phone_bundle.confirm_secure_bound_and_aliasify_by_uid'

    def is_account_type_allowed(self):
        # ПДД и федералов не пускаем: им нельзя телефонный алиас
        allowed_account_types = {
            ACCOUNT_TYPE_NORMAL,
        }
        if not self.form_values['enable_alias_as_email']:
            allowed_account_types.update(
                {
                    ACCOUNT_TYPE_LITE,
                    ACCOUNT_TYPE_NEOPHONISH,
                    ACCOUNT_TYPE_SOCIAL,
                },
            )
        return self.account.type in allowed_account_types

    def _process(self):
        self.get_account()
        self.mount_secure_bound_phone_number()
        self.assert_alias_absence()
        self.send_code('confirm_secure_bound_and_aliasify')
        self.track.enable_phonenumber_alias_as_email = self.form_values['enable_alias_as_email']


class ConfirmSecureBoundAndAliasifyCommitter(BasePhoneBundleCommitter):
    basic_form = forms.ConfirmSecureBoundAndAliasifyCommitForm

    track_state = CONFIRM_BOUND_SECURE_AND_ALIASIFY_STATE

    def is_account_type_allowed(self):
        # ПДД и федералов не пускаем: им нельзя телефонный алиас
        allowed_account_types = {
            ACCOUNT_TYPE_NORMAL,
        }
        if not self.track.enable_phonenumber_alias_as_email:
            allowed_account_types.update(
                {
                    ACCOUNT_TYPE_LITE,
                    ACCOUNT_TYPE_NEOPHONISH,
                    ACCOUNT_TYPE_SOCIAL,
                },
            )
        return self.account.type in allowed_account_types

    def _process(self):
        self.confirm_code()
        self.get_account()
        self.assert_alias_absence()
        self.assert_secure_phone_not_changed()

        aliasification = self.build_phonenumber_aliasification(
            phone_number=self.number,
            language=self.track.display_language,
            enable_search=self.track.enable_phonenumber_alias_as_email,
        )

        phone = self.account.phones.by_number(self.number)
        if phone.operation:
            raise SecureOperationExistsError()

        code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'acquire_phone', 'consumer': self.consumer},
        ):
            short_ttl = timedelta(seconds=settings.YASMS_MARK_OPERATION_TTL)
            logical_op = AliasifySecureOperation.create(
                phone_manager=self.account.phones,
                phone_id=phone.id,
                code=code,
                statbox=self.statbox,
                ttl=short_ttl,
            )

        self.statbox.dump_stashes(operation_id=logical_op.id)
        alias_owner = aliasification.get_owner()

        if alias_owner:
            with UPDATE(
                alias_owner,
                self.request.env,
                {'action': 'phone_alias_delete', 'consumer': self.consumer},
            ):
                aliasification.take_away()

        self.set_validation_period()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'confirm_secure_bound_and_aliasify', 'consumer': self.consumer},
        ):
            logical_op.confirm_phone(phone.id, code)
            logical_op.apply()
            aliasification.give_out(self.validation_period)
        self.statbox.dump_stashes()
        aliasification.notify()
        self.statbox.log(
            operation='update',
            number=self.number.masked_format_for_statbox,
        )


class DeleteAliasSubmitter(BasePhoneBundleSubmitter):
    # FIXME сделать одну ручку и без display_language
    basic_form = forms.DeleteAliasSubmitForm
    track_state = DELETE_ALIAS_STATE
    by_uid_grant = 'phone_bundle.delete_alias_by_uid'
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE]

    def is_account_type_allowed(self):
        return True

    def _process(self):
        self.get_account()

        self.assert_alias_existence()
        self.mount_alias_number()
        self.save_number()

        self.response_values['is_password_required'] = self.account.have_password


class DeleteAliasCommitter(BasePhoneBundleCommitter):
    track_state = DELETE_ALIAS_STATE
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE]
    basic_form = forms.DeleteAliasCommitForm

    def is_account_type_allowed(self):
        return True

    def get_account(self):
        if not self.form_values['password']:
            self.get_account_for_commit()
            if self.account.have_password:
                raise PasswordNotMatchedError()
        else:
            self.verify_account_password()

    def _process(self):
        self.get_account()
        self.assert_alias_existence()
        self.dealiasify_phone()


class ConfirmTrackedSecureSubmitter(BasePhoneBundleSubmitter, CheckPhoneKarmaMixin):
    """
    При авторизации по смс или принудительной смене пароля, когда у пользователя уже есть защищенный
    телефон(сохранен в треке), и потребуется выслать на него sms с кодом
    подтверждения
    """
    basic_form = forms.ConfirmTrackedSecureSubmitForm
    track_state = CONFIRM_TRACKED_SECURE_STATE
    require_track = True
    allow_states = [CONFIRM_STATE]

    def is_account_type_allowed(self):
        # Тип аккаунта не проверяем: если аккаунт смог привязать секьюрный телефон, должен и мочь его переподтвердить
        return True

    def _process(self):
        self.get_account_from_track()

        try:
            self.mount_saved_secure_number()
        except InvalidTrackStateError:
            self.mount_bank_number()

        self.check_phone_karma(self.number)

        self.send_code('confirm_tracked_secure')
        self.track.phone_confirmation_method = CONFIRM_METHOD_BY_SMS


class ConfirmTrackedSecureCommitter(BasePhoneBundleCommitter):
    """
    Доверяем содержимому трека - валидируем тот телефон(защищенный/банковский), что там лежит
    """
    basic_form = forms.ConfirmTrackedSecureCommitForm
    track_state = CONFIRM_TRACKED_SECURE_STATE
    require_track = True
    accept_fake_code_for_test_numbers = True

    def is_account_type_allowed(self):
        # Тип аккаунта не проверяем: если аккаунт смог привязать секьюрный телефон, должен и мочь его переподтвердить
        return True

    def _process(self):
        # Проверим, что аккаунт валидный
        self.get_account()

        self.confirm_code()


class ConfirmAndUpdateTrackedSecureSubmitter(ConfirmTrackedSecureSubmitter):
    track_state = CONFIRM_AND_UPDATE_TRACKED_SECURE_STATE


class ConfirmAndUpdateTrackedSecureCommitter(ConfirmTrackedSecureCommitter):
    """
    Перед обновлением времени привязки защищенного телефона,
    проверим, что был подтвержден актуальный защищенный телефон пользователя
    """
    track_state = CONFIRM_AND_UPDATE_TRACKED_SECURE_STATE

    def _process(self):
        self.get_account()
        self.confirm_code()
        self.assert_secure_phone_not_changed()

        phone = self.account.phones.by_number(self.number)
        with UPDATE(self.account, self.request.env, {'action': 'confirm_and_update_tracked_secure', 'consumer': self.consumer}):
            phone.confirm()


class TrackedProlongValidSecureView(BasePhoneBundleView,
                                    PhoneValidationMixin,
                                    BundleAccountGetterMixin):
    """
    Ручка берет из трека защищенный телефон и вызывает yasms.prolong_valid
    для него.
    """
    required_grants = [BASIC_GRANT]
    require_track = True
    track_state = PROLONG_VALID_STATE
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE]

    def process(self):
        self.read_track()
        self.get_account_from_track(check_disabled_on_deletion=True)
        self.mount_saved_secure_number()
        self.assert_secure_phone_not_changed()
        try:
            with UPDATE(
                self.account,
                self.request.env,
                {'action': 'prolong_valid', 'consumer': self.consumer},
            ):
                self.yasms_api.prolong_valid(self.account, self.number.e164)
        except YaSmsError:
            pass
        with self.track_transaction.commit_on_error():
            self.track.is_successful_phone_passed = True
        self.response_values['track_id'] = self.track_id
        self.response_values['number'] = helpers.dump_number(self.number)


class PhoneConfirmedView(
    BundleAccountGetterMixin,
    BundlePhoneMixin,
    BaseBundleView,
):
    """
    Ручка проверки свежего подтверждения телефона
    standalone
    """
    # Требуется трек
    require_track = True
    # Необходимые заголовки, нам нужна кука пользователя
    required_headers = (
        HEADER_CLIENT_COOKIE,
        HEADER_CONSUMER_CLIENT_IP,
        HEADER_CLIENT_HOST,
    )
    required_grants = ['phone_bundle.base']

    def process_request(self):
        """
        Устанавливаем требуются ли конфирмить secure phone у аккаунта
        Если выбрасываем PhoneNotConfirmedError то требуется
        """
        self.get_account_from_session(
            need_phones=True,
        )
        self.read_track()
        self.assert_secure_phone_confirmed_with_track()
