# -*- coding: utf-8 -*-

import logging

from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_FEDERAL,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.core.utils.decorators import cached_property

from ..exceptions import (
    AccountDisabledError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    PasswordRequiredError,
)
from ..headers import HEADER_CONSUMER_CLIENT_IP
from ..mixins import BundleAccountPropertiesMixin
from .base import (
    BasePhoneBundleCommitter,
    BasePhoneBundleSubmitter,
    BasePhoneBundleView,
    BASIC_GRANT,
    SECURE_GRANT,
)
from .controllers import (
    CONFIRM_AND_BIND_SECURE_STATE,
    ConfirmAndBindSecureAndAliasifyCommitter,
    ConfirmAndBindSecureAndAliasifySubmitter,
    ConfirmAndBindSecureCommitter,
    ConfirmAndBindSecureSubmitter,
    ConfirmSecureBoundAndAliasifyCommitter,
    ConfirmSecureBoundAndAliasifySubmitter,
    DeleteAliasCommitter,
    DeleteAliasSubmitter,
    REQUIRED_HEADERS_V2,
)
from .exceptions import PhoneOperationExistsError
from .forms import AccountedNumberedSubmitForm
from .forms_v2 import (
    BindOrConfirmBoundCommitForm,
    BindOrConfirmBoundSubmitForm,
    BindPhoneFromPhonishToPortalForm,
    PasswordedCommitForm,
)


"""
В этом модуле собираем вторую версию телефонных бандлов.
В отличие от первой, данные ручки проверяют авторизацию пользователя по куке,
кука берется из заголовков.
"""

log = logging.getLogger(__name__)

BIND_OR_CONFIRM_BOUND_STATE = 'bind_or_confirm_bound_state'


class ConfirmAndBindSecureSubmitterV2(ConfirmAndBindSecureSubmitter):
    basic_form = AccountedNumberedSubmitForm

    track_state = CONFIRM_AND_BIND_SECURE_STATE

    required_grants = [BASIC_GRANT, SECURE_GRANT]
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_submit_v2()

    def _process(self):
        super(ConfirmAndBindSecureSubmitterV2, self)._process()
        self.response_values['is_password_required'] = False


class ConfirmAndBindSecureCommitterV2(ConfirmAndBindSecureCommitter):
    basic_form = PasswordedCommitForm

    track_state = CONFIRM_AND_BIND_SECURE_STATE

    required_grants = [BASIC_GRANT, SECURE_GRANT]
    by_uid_grant = ConfirmAndBindSecureSubmitterV2.by_uid_grant
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_commit_v2()


class ConfirmAndBindSecureAndAliasifySubmitterV2(ConfirmAndBindSecureAndAliasifySubmitter):
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_submit_v2()

    def _process(self):
        super(ConfirmAndBindSecureAndAliasifySubmitterV2, self)._process()
        self.response_values['is_password_required'] = False


class ConfirmAndBindSecureAndAliasifyCommitterV2(ConfirmAndBindSecureAndAliasifyCommitter):
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_commit_v2()


class ConfirmSecureBoundAndAliasifySubmitterV2(ConfirmSecureBoundAndAliasifySubmitter):
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_submit_v2()


class ConfirmSecureBoundAndAliasifyCommitterV2(ConfirmSecureBoundAndAliasifyCommitter):
    required_headers = REQUIRED_HEADERS_V2


class BindOrConfirmBound(object):
    def _phone_is_bound(self, account, number):
        phone = account.phones.by_number(number)
        return bool(phone and phone.bound)

    def is_account_type_allowed(self, account, number):
        if account.type in (
            ACCOUNT_TYPE_FEDERAL,
            ACCOUNT_TYPE_LITE,
            ACCOUNT_TYPE_NEOPHONISH,
            ACCOUNT_TYPE_NORMAL,
            ACCOUNT_TYPE_PDD,
            ACCOUNT_TYPE_SOCIAL,
        ):
            return True

        if account.type == ACCOUNT_TYPE_PHONISH and self._phone_is_bound(account, number):
            return True

        return False

    def set_view_properties(self, view):
        view.track_state = BIND_OR_CONFIRM_BOUND_STATE
        view.required_headers = REQUIRED_HEADERS_V2
        view.by_uid_grant = 'phone_bundle.bind_simple_or_confirm_bound_by_uid'


class BindOrConfirmBoundSubmitter(BasePhoneBundleSubmitter):
    basic_form = BindOrConfirmBoundSubmitForm

    def __init__(self):
        super(BindOrConfirmBoundSubmitter, self).__init__()
        self._utils = BindOrConfirmBound()
        self._utils.set_view_properties(self)

    def is_account_type_allowed(self):
        return self._utils.is_account_type_allowed(self.account, self.number)

    def get_account(self):
        self.get_account_for_submit_v2()

    def _process(self):
        self.mount_specified_number()
        self.get_account()
        self.send_code('bind_or_confirm_bound')


class BindOrConfirmBoundCommitter(BasePhoneBundleCommitter):
    basic_form = BindOrConfirmBoundCommitForm

    def __init__(self):
        super(BindOrConfirmBoundCommitter, self).__init__()
        self._utils = BindOrConfirmBound()
        self._utils.set_view_properties(self)

    def is_account_type_allowed(self):
        return self._utils.is_account_type_allowed(self.account, self.number)

    def get_account(self):
        self.get_account_for_commit_v2()

    def bind_simple_phone(self):
        save_simple_phone = self.build_save_simple_phone(phone_number=self.number)
        save_simple_phone.submit()

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'confirm_and_bind', 'consumer': self.consumer},
        ):
            save_simple_phone.commit()

        save_simple_phone.after_commit()

    def reconfirm_phone(self):
        phone = self.account.phones.by_number(self.number)
        with UPDATE(self.account, self.request.env, {'action': 'confirm_bound', 'consumer': self.consumer}):
            phone.confirm()
        self.statbox.log(operation='update', number=self.number.masked_format_for_statbox)

    def _process(self):
        self.get_account()
        self.confirm_code()

        phone = self.account.phones.by_number(self.number)
        if not phone or not phone.bound:
            self.bind_simple_phone()
        else:
            self.reconfirm_phone()


class DeleteAliasSubmitterV2(DeleteAliasSubmitter,
                             BundleAccountPropertiesMixin):
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_submit_v2()

    def _process(self):
        super(DeleteAliasSubmitterV2, self)._process()
        self.response_values['is_password_required'] = self.account.have_password and self.is_password_verification_required()


class DeleteAliasCommitterV2(DeleteAliasCommitter,
                             BundleAccountPropertiesMixin):
    required_headers = REQUIRED_HEADERS_V2

    def get_account(self):
        self.get_account_for_commit_v2()

        if self.account.have_password and self.is_password_verification_required():
            if not self.form_values['password']:
                raise PasswordRequiredError('Password required')

            self.verify_account_password()


class BindPhoneFromPhonishToPortal(BasePhoneBundleView):
    required_grants = ['phone_bundle.bind_phone_from_phonish_to_portal']
    required_headers = [HEADER_CONSUMER_CLIENT_IP]
    basic_form = BindPhoneFromPhonishToPortalForm
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE]

    def process_request(self):
        self.process_basic_form()
        portal_account, phonish_account = self._get_all_accounts_by_uids(
            [
                self.form_values['portal_uid'],
                self.form_values['phonish_uid'],
            ],
        )
        self._check_account_enabled(portal_account)
        self._check_account_enabled(phonish_account)
        self._check_account_is_like_portal(portal_account)
        self._check_account_is_phonish(phonish_account)

        self.statbox.bind_context(uid=portal_account.uid)

        phonish_phone = phonish_account.phones.default
        portal_phone = portal_account.phones.by_number(phonish_phone.number)
        portal_phone_operation = None if portal_phone is None else portal_phone.get_logical_operation(self.statbox)
        if (
            portal_phone is None or
            not portal_phone.bound and portal_phone_operation is None or
            not portal_phone.bound and portal_phone_operation.is_binding and not portal_phone_operation.is_secure
        ):
            self._copy_phone_to_account(portal_account, phonish_phone)
            self.statbox.log(action='new_phone_copied_from_phonish', source_uid=phonish_account.uid)
        elif portal_phone.bound and phonish_phone.confirmed > portal_phone.confirmed:
            self._update_phone_confirmation_time(
                portal_account,
                portal_phone.id,
                phonish_phone.confirmed,
            )
            self.statbox.log(action='old_phone_updated_from_phonish', source_uid=phonish_account.uid)
        elif portal_phone.bound and phonish_phone.confirmed <= portal_phone.confirmed:
            # Подтверждение портального телефона новее фонишного, поэтому
            # не нужно менять время.
            pass
        else:
            raise PhoneOperationExistsError()

    def _get_all_accounts_by_uids(self, uids):
        userinfo_args = add_phone_arguments(need_aliases=True)
        accounts, unknown_uids = get_many_accounts_by_uids(uids, self.blackbox, userinfo_args)
        if unknown_uids:
            raise AccountNotFoundError()
        accounts.sort(key=lambda x: uids.index(x.uid))
        return accounts

    def _check_account_enabled(self, account):
        if not account.is_user_enabled:
            raise AccountDisabledError()

    def _check_account_is_like_portal(self, account):
        if not (
            account.is_neophonish or
            account.is_normal or
            account.is_lite or
            account.is_social or
            account.is_pdd
        ):
            raise AccountInvalidTypeError()

    def _check_account_is_phonish(self, account):
        if not account.is_phonish:
            raise AccountInvalidTypeError()

    def _copy_phone_to_account(self, account, phone):
        save_simple_phone = self.build_save_simple_phone(
            account=account,
            phone_number=phone.number,
            phone_confirmation_datetime=phone.confirmed,
            washing_enabled=False,
            should_book_confirmation_to_statbox=False,
        )
        save_simple_phone.submit()
        with UPDATE(
            account,
            self.request.env,
            {
                'action': 'bind_phone_from_phonish_to_portal',
                'consumer': self.consumer,
            },
        ):
            save_simple_phone.commit()
        save_simple_phone.after_commit()

    def _update_phone_confirmation_time(self, account, phone_id, confirmed_at):
        phone = account.phones.by_id(phone_id)
        with UPDATE(
            account,
            self.request.env,
            {
                'action': 'bind_phone_from_phonish_to_portal',
                'consumer': self.consumer,
            },
        ):
            phone.confirm(confirmed_at)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='bind_phone_from_phonish_to_portal',
            ip=self.client_ip,
            consumer=self.consumer,
            user_agent=self.user_agent,
        )
