# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.exceptions import BaseBundleError


class BasePhoneBundleError(BaseBundleError):
    """Базовая ошибка для всего телефонного бандла"""


class ConfirmationsLimitExceededError(BasePhoneBundleError):
    """Превышено количество попыток ввода кода подтверждения"""
    error = 'confirmations_limit.exceeded'


class PhoneAliasExistError(BasePhoneBundleError):
    """Телефонный алиас уже есть у пользователя"""
    error = 'phone_alias.exist'


class PhoneAliasNotFoundError(BasePhoneBundleError):
    """Телефонного алиаса ещё нет у пользователя"""
    error = 'phone_alias.not_found'


class PhoneBoundAndConfirmedError(BasePhoneBundleError):
    """Номер телефона уже привязан к аккаунту и подтверждён"""
    error = 'phone.bound_and_confirmed'


class PhoneBoundError(BasePhoneBundleError):
    """Номер телефона уже привязан к аккаунту."""
    error = 'phone.bound'


class SecurePhoneBoundAndConfirmedError(BasePhoneBundleError):
    """Номер телефона является secure телефоном и он уже привязан к аккаунту и подтверждён"""
    error = 'phone_secure.bound_and_confirmed'


class SecurePhoneAlreadyExistsError(BasePhoneBundleError):
    """У аккаунта уже есть привязанный защищенный телефон, текущая ручка в этой ситуации не работает."""
    error = 'phone_secure.already_exists'


class PhoneConfirmedError(BasePhoneBundleError):
    """Номер телефона уже подтверждён"""
    error = 'phone.confirmed'


class PhoneNotConfirmedError(BasePhoneBundleError):
    """Номер телефона не подтвержден"""
    error = 'phone.not_confirmed'


class PasswordNotConfirmedError(BasePhoneBundleError):
    """Пароль не проверен"""
    error = 'password.not_confirmed'


class ConfirmationRequiredError(BasePhoneBundleError):
    """Требуется ввод пароля или подтверждение кодом смс"""
    error = 'confirmation.required'


class PhoneNotFoundError(BasePhoneBundleError):
    """У аккаунта нет привязанного номера телефона"""
    error = 'phone.not_found'


class OperationNotFoundError(BasePhoneBundleError):
    """У аккаунта нет телефонной операции с указанным id"""
    error = 'operation.not_found'


class OperationExpiredError(BasePhoneBundleError):
    """Время жизни телефонной операции подошлу к концу."""
    error = 'operation.expired'


class InvalidOperationStateError(BasePhoneBundleError):
    """Телефонная операция в текущем виде не может быть использована в методе."""
    error = 'operation.invalid_state'


class PhoneOperationExistsError(BaseBundleError):
    """Пытаемся создать операцию на телефоне, на котором уже есть операция."""
    error = 'operation.exists'


class SendingLimitExceededError(BasePhoneBundleError):
    """Превышено разрешённое количество отправленных СМС"""
    error = 'sms_limit.exceeded'


class CallingLimitExceededError(BasePhoneBundleError):
    """Превышено разрешённое количество звонков"""
    error = 'calls_limit.exceeded'


class CallNotMadeError(BasePhoneBundleError):
    """Мы еще не звонили на этот номер"""
    error = 'call.not_made'


class ConfirmationNotReady(BasePhoneBundleError):
    """Еще не получили лог сессии от октопуса с подтверждением кода"""
    error = 'call_confirm.not_ready'


class ConfirmationFailed(BasePhoneBundleError):
    """Не смогли дозвониться/пользователь не взял трубку"""
    error = 'call_confirm.failed'


class ConfirmationCallInProgress(BasePhoneBundleError):
    """Звонок в процессе"""
    error = 'call_confirm.in_progress'


class ConfirmationCallFinished(BasePhoneBundleError):
    """Звонок завершен"""
    error = 'call_confirm.finished'


class ConfirmationCallHangup(BasePhoneBundleError):
    """Звонок сброшен пользователем"""
    error = 'call_confirm.user_hangup'


class ConfirmationCallUnavailable(BasePhoneBundleError):
    """Пользователь временно недоступен"""
    error = 'call_confirm.user_unavailable'


class ConfirmationCallStarted(BasePhoneBundleError):
    """Звонок пошел"""
    error = 'call_confirm.started'


class ConfirmationCheckTooEarlyError(BasePhoneBundleError):
    """Проверка статуса звонка слишком рано"""
    error = 'call_confirm.too_early'


class ConfirmationCheckTooLateError(BasePhoneBundleError):
    """Проверка статуса звонка слишком поздно"""
    error = 'call_confirm.too_late'


class PhoneBlockedError(BasePhoneBundleError):
    """Телефон (временно) заблокирован"""
    error = 'phone.blocked'


class SmsNotSentError(BasePhoneBundleError):
    """СМС ещё не отправлялась на указанный номер"""
    error = 'sms.not_sent'


class PhoneInvalidError(BasePhoneBundleError):
    """Невалидный номер телефона"""
    error = 'number.invalid'


class PhoneOperationInapplicable(BasePhoneBundleError):
    def __init__(self, errors, *args, **kwargs):
        super(PhoneOperationInapplicable, self).__init__(*args, **kwargs)
        self._errors = errors

    @classmethod
    def from_exception(cls, exception):
        errors = []
        if exception.need_password_verification:
            errors.append(PasswordNotConfirmedError.error)

        if exception.need_confirmed_phones:
            errors.append(PhoneNotConfirmedError.error)
        for phone in exception.need_confirmed_phones:
            errors.append('%s(%d)' % (PhoneNotConfirmedError.error, phone.id))
        return cls(errors)

    @property
    def errors(self):
        return self._errors


class InvalidPhoneType(BasePhoneBundleError):
    """
    Работа с данным типом телефона (простой или защищённый) невозможна.
    """
    error = 'phone.invalid_type'


class SecureOperationExistsError(BasePhoneBundleError):
    """
    Уже начата операция в которой участвует защищённый телефон или в завершении
    которой телефон станет защищённым.
    """
    error = 'operation.secure_operation_exists'


class SMSBackendAccessDeniedError(BasePhoneBundleError):
    """Отсутствует грант в Ясмс"""
    error = 'backend.yasms_access_denied'
