# -*- coding: utf-8 -*-

from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.core import validators


# Формы для ручек submit
class BaseSubmitForm(validators.Schema):
    """Базовая форма для ручки submit с обязательным языком текста в смс (должен
    быть из известного паспорту списка языков).

    route - опциональный маршрут отправки СМС в YaSMS. Требует гранта
    sms_routes.<маршрут>, аналогично Route.<маршрут> в YaSMS. PASSP-18828
    """
    display_language = validators.DisplayLanguage()
    route = validators.String(if_missing=None, strip=True, not_empty=True)
    return_unmasked_number = validators.StringBool(if_missing=True)


class NumberedSubmitForm(BaseSubmitForm):
    """
    Форма для ручек submit, где требуется указание телефонного номера, с
    которым будет выполняться дальнейшая работа (как минимум, подтверждение).

    Номер может быть указан в международном, полном, формате - с кодом страны.
    В этом случае дополнительное указание страны через дополнительное поле не
    требуется. Если же номер указан в локальном формате, без кода страны,
    необходимо дополнительно указать (уточнить) страну, в которой этот номер
    телефона является валидным. Например, номер 8 (916) 123-45-67 для страны RU
    будет правильно распарсен в международный формат +79161234567.
    """
    number = validators.String(not_empty=True, strip=True)
    country = validators.CountryCode(if_missing=None, not_empty=True)
    code_format = validators.OneOf(['by_3', 'by_3_dash'], if_missing=None, not_empty=True)

    gps_package_name = validators.GpsPackageName()

    chained_validators = [
        validators.PhoneNumber(phone_number_key='number'),
    ]


class AccountedSubmitForm(BaseSubmitForm):
    """
    Форма для ручек submit, где требуется указание аккаунта, с которым будет
    выполнятся дальнейшая работа (фактически, к какому аккаунту будет выполнена
    привязка подтверждённого номера).

    Указывается тремя способами:
      * Параметром uid - точное указание числового идентификатора паспортного
        аккаунта. Требует наличие дополнительного гранта.
      * Параметром sessionid - значение авторизационной куки Session_id (может
        быть получена потребителем из браузерного запроса). При этом обязательно
        наличие дополнительного хедера Ya-Client-Host (требуется для проверки
        валидности куки в ЧЯ). Дополнительный грант в этом случае не нужен.
      * HTTP-заголовком Ya-Client-Authorization.
    """
    uid = validators.Uid(if_missing=None, not_empty=True)
    sessionid = validators.String(if_missing=None, strip=True, not_empty=True)


class AccountedNumberedSubmitForm(NumberedSubmitForm):
    """Объединённая форма для ручек submit, где требуется указание и телефонного
    номера, и аккаунта."""
    uid = validators.Uid(if_missing=None, not_empty=True)
    sessionid = validators.String(if_missing=None, strip=True, not_empty=True)


# Формы для ручек commit
class BaseCommitForm(validators.Schema):
    """Базовая форма для ручек commit с обязательным кодом подтверждения."""
    code = validators.String(not_empty=True, strip=True)
    return_unmasked_number = validators.StringBool(if_missing=True)


class OptionalPasswordedCommitForm(BaseCommitForm):
    """Форма для ручек commit, где можно указать пароль от аккаунта."""
    password = validators.String(not_empty=True, if_missing=None, strip=True)


# Конечные формы
class ConfirmSubmitForm(NumberedSubmitForm):
    """Форма для confirm/submit: трек, язык, номер, страна и способ подтверждения"""
    confirm_method = validators.OneOf(
        [CONFIRM_METHOD_BY_SMS, CONFIRM_METHOD_BY_CALL, CONFIRM_METHOD_BY_FLASH_CALL],
        if_missing=CONFIRM_METHOD_BY_SMS,
        not_empty=True,
    )
    number = validators.String(strip=True, if_missing=None)
    phone_id = validators.PhoneId(not_empty=False, if_missing=None)
    chained_validators = [
        validators.PhoneNumber(phone_number_key='number'),
        validators.RequireSome(['number', 'phone_id'], min_=1, max_=1),
    ]


class ConfirmCommitForm(BaseCommitForm):
    """Форма для confirm/commit: код"""


class ConfirmBoundSubmitForm(AccountedNumberedSubmitForm):
    """Форма для confirm_bound/submit: язык, номер, страна и аккаунт"""


class ConfirmBoundCommitForm(BaseCommitForm):
    """Форма для confirm_bound/commit: код"""


class ConfirmAndBindSimpleCommitForm(BaseCommitForm):
    """Форма для confirm_and_bind/commit: код"""


class ConfirmAndBindSecureSubmitForm(AccountedNumberedSubmitForm):
    """Форма для confirm_and_bind_secure/submit: язык, номер, страна и аккаунт"""


class ConfirmAndBindSecureCommitForm(OptionalPasswordedCommitForm):
    """
    Форма для confirm_and_bind_secure/commit: код и пароль (если на аккаунте
    есть пароль).
    """


class ConfirmAndBindSecureAndAliasifySubmitForm(AccountedNumberedSubmitForm):
    """
    Форма для confirm_and_bind_secure_and_aliasify/submit:
        язык,
        номер,
        страна,
        аккаунт,
        нужно включить ПЦА
    """
    enable_alias_as_email = validators.StringBool(if_missing=True)


class ConfirmAndBindSecureAndAliasifyCommitForm(OptionalPasswordedCommitForm):
    """Форма для confirm_and_bind_secure_and_aliasify/commit: код и пароль"""


class ConfirmSecureBoundAndAliasifySubmitForm(AccountedSubmitForm):
    """
    Форма для confirm_secure_bound_and_aliasify/submit:
        язык,
        аккаунт,
        нужно включить ПЦА
    """
    enable_alias_as_email = validators.StringBool(if_missing=True)


class ConfirmSecureBoundAndAliasifyCommitForm(BaseCommitForm):
    """Форма для confirm_secure_bound_and_aliasify/commit: код"""


class DeleteAliasSubmitForm(AccountedSubmitForm):
    """Форма для delete_alias/submit: требует язык и аккаунт"""
    # NOTE: возможно для удаления алиаса язык не нужен, будет ясно в процессе


class DeleteAliasCommitForm(validators.Schema):
    """Форма для delete_alias/commit: может требовать пароль"""
    password = validators.String(not_empty=False, if_missing=None, strip=True)


class ConfirmTrackedSecureSubmitForm(BaseSubmitForm):
    """Форма для confirm_tracked_secure/submit: язык и трек"""
    gps_package_name = validators.GpsPackageName()
    code_format = validators.OneOf(['by_3_dash'], if_missing=None, not_empty=True)


class ConfirmTrackedSecureCommitForm(BaseCommitForm):
    """Форма для confirm_tracked_secure/commit: код"""
