# -*- coding: utf-8 -*-
import logging
import random
import time

from passport.backend.api.common.account_manager import (
    AmPlatform,
    get_am_platform,
)
from passport.backend.api.common.common import should_ignore_per_ip_counters
from passport.backend.api.common.octopus import (
    inc_calls_counter,
    inc_sessions_created,
    KOLMOGOR_COUNTER_CALLS_FAILED,
)
from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
    PhoneAntifraudFeatures,
)
from passport.backend.api.common.suggest import get_octopus_language_suggest
from passport.backend.api.common.yasms import (
    build_route_advice,
    generate_fake_global_sms_id,
    log_sms_not_delivered,
)
from passport.backend.api.views.bundle.exceptions import (
    CaptchaRequiredError,
    InvalidTrackStateError,
)
from passport.backend.core.builders.antifraud import get_antifraud_api
from passport.backend.core.builders.blackbox.constants import BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON
from passport.backend.core.builders.kolmogor import get_kolmogor
from passport.backend.core.builders.octopus import BaseOctopusError
from passport.backend.core.builders.yasms import exceptions as yasms_exceptions
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    calls_per_ip,
    calls_per_phone,
    sms_per_ip,
    sms_per_ip_for_app,
    sms_per_ip_for_consumer,
)
from passport.backend.core.counters.change_password_counter import get_per_phone_number_buckets
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.types.mobile_device_info import get_app_id_from_track
from passport.backend.core.types.phone import build_phones
from passport.backend.core.types.phone_number.phone_number import (
    get_alt_phone_numbers_of_phone_number,
    PhoneNumber,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import (
    format_code_by_3,
    generate_random_code,
    normalize_code,
)
from passport.backend.utils.time import string_to_integer_unixtime

from . import exceptions
from .. import exceptions as bundle_exceptions


FAKE_OCTOPUS_SESSION_ID_FOR_TEST_PHONES = 'ELEGANT_CANE_FAKE_SESSION_ID'
FAKE_OCTOPUS_SESSION_ID_FOR_ANTIFRAUD_DENIAL = 'ELEGANT_CANE_FAKE_SESSION_ID_AF'


log = logging.getLogger(__name__)


class AntifraudDeniedError(Exception):
    """
    Антифрод запретил звонить/отправлять смс
    Эта ошибка намеренно небандловая: она должна ловиться внутри конфирматора и не протекать наружу.
    """


def dump_number(number, only_masked=False):
    return number.as_dict(only_masked=only_masked)


def check_change_password_per_phone_number_counter(track, phone_number, statbox):
    # Выходим если трэк не от смены пароля с:
    # - привязкой нового защищённого телефона
    # - заменой утраченного защищенного номера через карантин
    secure_number = None
    if track.has_secure_phone_number and track.secure_phone_number:
        secure_number = PhoneNumber.parse(track.secure_phone_number)

    if not (track.is_change_password_sms_validation_required and
            (not track.has_secure_phone_number or phone_number != secure_number)):
        return

    counter = get_per_phone_number_buckets()
    is_exceeded = counter.hit_limit(phone_number.e164)

    if not is_exceeded:
        return

    statbox.log(
        error='phone.compromised',
        track_id=track.track_id,
        counter_current_value=counter.get(phone_number.e164),
        counter_limit_value=counter.limit,
    )
    raise bundle_exceptions.PhoneCompromisedError()


def check_disable_change_password_phone_experiment(track):
    # Поскольку фронтендовая ручка отправки смс торчит наружу и её можно дёргать
    # указав телефон и трэк, то при выключении эксперимента на смене пароля
    # нельзя чтобы через эту ручку можно было отправить СМС.
    if settings.DISABLE_CHANGE_PASSWORD_PHONE_EXPERIMENT and track.is_password_change:
        raise exceptions.SendingLimitExceededError()


def _code_possible_splits_helper(i, code, res, delimiter='|'):
    if i == len(code):
        res.append(delimiter)
        return res
    if not res or res[-1] == ' ':
        res.append(code[i])
        return _code_possible_splits_helper(i + 1, code, res)
    else:
        return _code_possible_splits_helper(i, code, res + [' ']) + _code_possible_splits_helper(i + 1, code, res + [code[i]])


def code_possible_splits(code):
    return [s for s in ''.join(_code_possible_splits_helper(0, code, [])).split('|') if s]


def split_code_by_3(code):
    """
    Делим код на части по три ровно.
    Если код не делится на три, выбрасываем ошибку
    :rtype: list
    """
    code = normalize_code(code)
    if not code or len(code) % 3:
        raise ValueError('Code is not divisible by 3')

    return format_code_by_3(code).split()


class PhoneConfirmator(object):
    """
    Базовый класс, который подтверждает владение номером телефона. Подтверждение
    выполняется в два этапа, каждый из этих этапов обрабатывается в соответствующих
    классах-потомках.
    """
    def __init__(self, track, statbox):
        self.track = track
        self.statbox = statbox.get_child(operation='confirm')
        self.number = None

    # Количество попыток подтверждения кода.
    @property
    def confirmations_count(self):
        return self.track.phone_confirmation_confirms_count.get()

    # Был ли уже успешно проведен процесс подтверждения.
    @property
    def is_confirmed(self):
        return self.track.phone_confirmation_is_confirmed

    @property
    def last_sms_sent_time(self):
        if self.track.phone_confirmation_last_send_at:
            return string_to_integer_unixtime(self.track.phone_confirmation_last_send_at)

    @property
    def last_called_time(self):
        if self.track.phone_confirmation_last_called_at:
            return string_to_integer_unixtime(self.track.phone_confirmation_last_called_at)

    def _check_state(self):
        raise NotImplementedError()  # pragma: no cover


class PhoneConfirmatorInitialStep(PhoneConfirmator):
    confirmation_method = None  # переопределяется в потомках

    def __init__(
        self,
        client,
        track,
        env,
        previous_number,
        code_format,
        code_length,
        statbox,
        caller=None,
        account=None,
        can_handle_captcha=False,
        mask_antifraud_denial=False,
        login_id=None,
        phone_confirmation_language=None,
    ):
        super(PhoneConfirmatorInitialStep, self).__init__(track, statbox)
        self.client = client
        self.client_ip = env.user_ip
        self.env = env
        self.account = account
        self.login_id = login_id
        self.phone_confirmation_language = phone_confirmation_language

        if previous_number and not isinstance(previous_number, PhoneNumber):
            self.previous_number = PhoneNumber.parse(previous_number, allow_impossible=True)
        else:
            self.previous_number = previous_number

        if get_am_platform(self.track) == AmPlatform.IOS and not code_format:
            self.code_format = 'by_3_dash'
        else:
            self.code_format = code_format
        self.code_length = code_length
        self.caller = caller
        self.gps_package_name = None
        self.sms_retriever = None
        self.can_handle_captcha = can_handle_captcha
        self.mask_antifraud_denial = mask_antifraud_denial

    def _generate_random_code(self):
        return generate_random_code(self.code_length)

    def _is_code_valid(self, code):
        return len(normalize_code(code)) == self.code_length

    # Отдаёт code. Если указанный номер совпадает с предыдущим, а способ подтверждения не изменился - значит
    # выполняется повторная отсылка смс и нужно взять старый code из трека, без
    # генерация нового.
    # Если старый code из трека не валиден, перегенерирует его.
    @cached_property
    def code(self):
        if (
            not self._did_phone_or_confirm_method_change() and
            self.track.phone_confirmation_code and
            self._is_code_valid(self.track.phone_confirmation_code)
        ):
            return self.track.phone_confirmation_code
        else:
            code = self._generate_random_code()
            if self.code_format == 'by_3':
                return format_code_by_3(code)
            elif self.code_format == 'by_3_dash':
                return format_code_by_3(code, delimiter='-')
            else:
                return code

    def _did_phone_or_confirm_method_change(self):
        phone_confirmation_method = self.track.phone_confirmation_method or CONFIRM_METHOD_BY_SMS
        return self.previous_number is not None and (
            self.number != self.previous_number or
            phone_confirmation_method != self.confirmation_method
        )

    def _assert_phone_not_confirmed(self):
        if self.is_confirmed and not self._did_phone_or_confirm_method_change():
            raise exceptions.PhoneConfirmedError()

    def _check_state(self):
        """
        Кидает исключение, если телефон из запроса уже был подтверждён текущим методом.
        Кроме того проверяет что в треке не установлены флаги превышения лимитов на отправку СМС/звонков.
        """
        self._assert_phone_not_confirmed()
        if (
            self.track.phone_confirmation_send_ip_limit_reached or
            self.track.phone_confirmation_send_count_limit_reached
        ):
            log_sms_not_delivered(
                reason='sms_limit.exceeded',
                number=self.number,
                global_sms_id=generate_fake_global_sms_id(),
                caller=self.caller,
                client_ip=str(self.client_ip),
                user_agent=self.user_agent,
            )
            raise exceptions.SendingLimitExceededError()

        # При смене пароля или восстановлении аккаунта, если у пользователя уже есть защищённый номер,
        # то не даём отправлять смс на какой-либо другой номер.
        # Вообще, проверка должна быть непосредственно в привязывающих телефон ручках,
        # что секьюрный == подтвержденному -- и нет ничего страшного в том, что
        # злоумышленник отправит СМС-ку на левый номер.
        if (
            (self.track.is_password_change or self.track.track_type == 'restore' or self.track.is_it_otp_enable) and
            self.track.has_secure_phone_number and self.track.secure_phone_number
        ):
            secure_number = PhoneNumber.parse(self.track.secure_phone_number)
            if self.number != secure_number:
                # Продолжение костыля, на принудительной смене можно перепривязать номер
                if self.track.is_force_change_password and (self.track.is_fuzzy_hint_answer_checked or self.track.is_short_form_factors_checked):
                    self.score_phone()
                    return
                raise exceptions.PhoneInvalidError()

        self.score_phone()

    def _check_state_and_find_if_simulation_is_required(self, af_error_class=exceptions.SendingLimitExceededError):
        """
        Проверяет счётчики, скорит номер в АФ.

        :return: надо ли симулировать звонок/смс (не звонить и не отправлять смс, но ответить, что всё ок)
        """
        try:
            self._check_state()
            return False  # всё ок, симуляция не нужна
        except AntifraudDeniedError:
            if self.mask_antifraud_denial:
                # ошибку не показываем, но и смс/звонка не будет
                return True
            else:
                # отвечаем общей ошибкой, как будто перегрелись счётчики
                raise af_error_class()

    def save_number(self, number):
        self.number = number
        if self.number.original_country:
            self.track.country = self.number.original_country
        self.track.phone_confirmation_phone_number = self.number.e164
        self.track.phone_confirmation_phone_number_original = self.number.original
        self.statbox.bind_context(number=self.number.masked_format_for_statbox)

    def score_phone(self):
        if not settings.PHONE_CONFIRM_CHECK_ANTIFRAUD_SCORE:
            return

        phone_features = PhoneAntifraudFeatures.default(
            sub_channel=self.caller,
            user_phone_number=self.number,
        )
        phone_features.phone_confirmation_method = self.confirmation_method
        phone_features.add_environment_features(self.env)
        phone_features.add_track_features(self.track)
        if self.account and self.account.phones.secure:
            phone_features.add_secure_phone_flag()
        if self.login_id:
            phone_features.login_id = self.login_id
        if self.phone_confirmation_language:
            phone_features.phone_confirmation_language = self.phone_confirmation_language

        phone_features.score(
            antifraud=get_antifraud_api(),
            consumer=self.caller,
            error_class=AntifraudDeniedError,
            captcha_class=CaptchaRequiredError if self.can_handle_captcha else None,
            statbox=self.statbox,
            mask_denial=self.mask_antifraud_denial,
        )


class SendingPhoneConfirmator(PhoneConfirmatorInitialStep):
    """
    Класс, обрабатывающий первый шаг подтверждения телефона посредством смс.

    Проверяет различные лимиты и счётчики, генерирует код, составляет текст СМС,
    отправляет её с помощью yasms и сохраняет состояние в трек. Основной внешний
    метод - send_code().
    """
    confirmation_method = CONFIRM_METHOD_BY_SMS

    def __init__(
        self,
        client,
        track,
        env,
        previous_number,
        sent_sms_limit,
        sms_code_format,
        sms_code_length,
        sms_resend_timeout,
        sms_template,
        statbox,
        caller=None,
        user_agent=None,
        sms_id_callback=None,
        account=None,
        can_handle_captcha=False,
        mask_antifraud_denial=False,
        login_id=None,
        phone_confirmation_language=None,
    ):
        super(SendingPhoneConfirmator, self).__init__(
            client,
            track,
            env,
            previous_number,
            sms_code_format,
            sms_code_length,
            statbox,
            caller=caller,
            account=account,
            can_handle_captcha=can_handle_captcha,
            mask_antifraud_denial=mask_antifraud_denial,
            login_id=login_id,
            phone_confirmation_language=phone_confirmation_language,
        )
        self.sms_template = sms_template
        self.sent_sms_limit = sent_sms_limit
        self.sms_resend_timeout = sms_resend_timeout
        self.user_agent = user_agent
        self.sms_id_callback = sms_id_callback

    # Отдаёт код подтверждения. Если указанный номер совпадает с предыдущим, значит
    # выполняется повторная отсылка смс и нужно взять старый код из трека, без
    # генерация нового.
    @cached_property
    def sms_code(self):
        if self.number == self.previous_number and self.track.phone_confirmation_code:
            return self.track.phone_confirmation_code
        else:
            return self.code

    # TODO Удалить после выкатки PASSP-32458 (Через 3 часа)
    @cached_property
    def sms_text(self):
        if self.number == self.previous_number and self.track.phone_confirmation_sms:
            return self.track.phone_confirmation_sms
        else:
            return self.sms_template.replace('{{code}}', self.code)

    @property
    def sent_sms_count(self):
        return self.track.phone_confirmation_sms_count.get(default=0)

    # Временная отметка, до которой посылка повторной смс-ки запрещена.
    @property
    def deny_resend_until(self):
        # TODO довольно сбивающая с толку штука, т.к. зависит только от
        # нашего внутреннего счётчика, но ведь есть ещё ограничители у yasms,
        # которые тоже нужно как-то учитывать. дождаться, когда yasms сможет
        # отдавать свою временную отметку, до которой sendsms будет отдавать
        # TEMPORARYBLOCK. и брать max от двух отметок. плюс, есть counter,
        # хорошо бы и из него вытащить отметку, когда он пропустит.
        if self.last_sms_sent_time:
            return int(self.last_sms_sent_time + self.sms_resend_timeout)

    def _check_sent_sms_count_limit(self):
        result = self.sent_sms_count >= self.sent_sms_limit
        self.track.phone_confirmation_send_count_limit_reached = result
        if result:
            self.statbox.log(error='sms_limit.exceeded')
            raise exceptions.SendingLimitExceededError('too many. count=%d' %
                                                       self.sent_sms_count)

    def _check_sms_sending_rate(self):
        if self.sent_sms_count > 0:
            now = int(time.time())
            if self.deny_resend_until and now < self.deny_resend_until:
                self.statbox.log(error='sms_limit.exceeded')
                raise exceptions.SendingLimitExceededError('too early. now=%s deny_resend_until=%s' %
                                                           (now, self.deny_resend_until))

    def _check_global_sms_per_ip_counter(self):
        if sms_per_ip_for_consumer.exists(self.caller):
            limit_reached = sms_per_ip_for_consumer.get_counter(self.caller).hit_limit_by_ip(self.client_ip)
            symptoms = dict(consumer=self.caller)
        else:
            app_id = get_app_id_from_track(self.track)
            if should_ignore_per_ip_counters(app_id, self.track.phone_confirmation_phone_number):
                log.debug('sms_per_ip_limit_counter ignored due to YANGO trusted phone code')
                return
            if sms_per_ip_for_app.exists(app_id):
                limit_reached = sms_per_ip_for_app.get_counter(app_id).hit_limit_by_ip(self.client_ip)
                symptoms = dict(app_id=app_id)
            else:
                limit_reached = sms_per_ip.get_counter(self.client_ip).hit_limit_by_ip(self.client_ip)
                symptoms = dict()

        self.track.phone_confirmation_send_ip_limit_reached = limit_reached

        if limit_reached:
            self.statbox.log(error='sms_limit.exceeded')
            symptoms.update(ip=self.client_ip)
            symptoms = ', '.join('%s=%s' % s for s in symptoms.items())
            raise exceptions.SendingLimitExceededError('Sms sent per ip: ' + symptoms)

    def _check_specific_sms_per_ip_counter(self):
        self._check_registration_sms_sent_per_ip_counter()

    def _check_registration_sms_sent_per_ip_counter(self):
        # Выходим если трек не от регистрации с телефоном
        if self.track.track_type != 'register':
            return

        app_id = get_app_id_from_track(self.track)
        if (
            sms_per_ip_for_consumer.exists(self.caller) or
            sms_per_ip_for_app.exists(app_id)
        ):
            # Для особых потребителей полагаемся только на общие для всех процессов
            # счётчики коротких сообщений.
            return

        if should_ignore_per_ip_counters(app_id, self.track.phone_confirmation_phone_number):
            log.debug('registration_sms_sent_counter ignored due to YANGO trusted phone code')
            return

        counter = sms_per_ip.get_registration_sms_sent_counter(user_ip=self.client_ip)
        if not counter.hit_limit_by_ip(self.client_ip):
            return

        # Устанавливаем флаг превышения кол-ва смс в текущий трек
        self.track.phone_confirmation_send_ip_limit_reached = True

        self.statbox.log(
            action='submit_phone_confirmation',
            error='registration_sms_sent_per_ip_limit_has_exceeded',
            track_id=self.track.track_id,
            user_ip=self.client_ip,
            counter_current_value=counter.get(self.client_ip),
            counter_limit_value=counter.limit,
        )
        raise exceptions.SendingLimitExceededError('Registration sms sent per ip: %s' % self.client_ip)

    def _check_specific_per_phone_number_counter(self):
        check_change_password_per_phone_number_counter(self.track, self.number, self.statbox)

    def _check_disable_change_password_phone_experiment(self):
        check_disable_change_password_phone_experiment(self.track)

    def _save_sms(self):
        self.track.phone_confirmation_code = self.code
        # TODO Удалить после выкатки PASSP-32458
        self.track.phone_confirmation_sms = self.sms_text

    def _save_gate_ids(self, sms_response):
        if sms_response.get('used_gate_ids'):
            self.track.phone_confirmation_used_gate_ids = sms_response['used_gate_ids']

    def _reset_state_and_counts(self):
        # Если в треке была магия или бомж, надо все сбросить все про них

        # Если в трэке тот же номер, что и в запросе, то это значит,
        # что до пользователя не дошла смс и мы ещё раз смс переотправляем.
        # В этом случае не надо сбрасывать состояние трэка.
        # Если телефон из запроса отличается от сохранённого в трэке, или же метод подтверждения изменился,
        # то трэк надо поресетить.
        if self._did_phone_or_confirm_method_change():
            self.track.is_successful_phone_passed = False
            self.track.phone_confirmation_is_confirmed = False
            self.track.phone_confirmation_code = None
            # TODO Удалить после выкатки PASSP-32458
            self.track.phone_confirmation_sms = None
            if self.sent_sms_count > 0:
                self.track.phone_confirmation_sms_count.reset()
            if self.confirmations_count > 0:
                self.track.phone_confirmation_confirms_count.reset()
            # Сбрасываем предыдущие гейты, новый номер - новые правила роутинга
            self.track.phone_confirmation_used_gate_ids = None

    def send_code(self, number, identity, route=None):
        self.save_number(number)
        self._reset_state_and_counts()

        simulate_sms = self._check_state_and_find_if_simulation_is_required()

        try:
            self._check_sent_sms_count_limit()
            self._check_sms_sending_rate()
            self._check_global_sms_per_ip_counter()
            self._check_specific_sms_per_ip_counter()
            self._check_specific_per_phone_number_counter()
            self._check_disable_change_password_phone_experiment()
        except exceptions.SendingLimitExceededError:
            self.log_sms_not_delivered(
                reason='sms_limit.exceeded',
                global_sms_id=generate_fake_global_sms_id(),
            )
            raise

        self._save_sms()

        if simulate_sms:
            global_sms_id = generate_fake_global_sms_id()
        else:
            sms_resp = self._send_sms(identity, route=route)
            self._save_gate_ids(sms_resp)
            global_sms_id = sms_resp['id']

        # Времена и каунтеры важно обновлять в случае именно успешной отправки СМС
        # (или если мы притворились, что отправили).
        self._save_timings()
        self._count_sending()
        if not simulate_sms:
            self._write_code_sent_to_statbox(sms_id=global_sms_id)
        if self.sms_id_callback:
            self.sms_id_callback(str(global_sms_id))

    def _write_code_sent_to_statbox(self, sms_id):
        if self.gps_package_name is not None:
            self.statbox.bind(
                gps_package_name=self.gps_package_name,
                sms_retriever=self.sms_retriever,
            )
        self.statbox.log(
            action='send_code',
            ip=self.client_ip,
            sms_count=self.sent_sms_count,
            sms_id=sms_id,
        )

    def _send_sms(self, identity, route=None):
        global_sms_id = generate_fake_global_sms_id()
        if self.sms_id_callback:
            self.sms_id_callback(global_sms_id)
        try:
            uid = self.track.uid
            from_uid = None
            if uid:
                from_uid = uid
            return self.client.send_sms(
                self.number.e164,
                self.sms_template,
                text_template_params=dict(code=self.sms_code),
                from_uid=from_uid,
                caller=self.caller,
                identity=identity,
                route=build_route_advice(self.number, self.caller, self.track, route),
                used_gate_ids=self.track.phone_confirmation_used_gate_ids,
                client_ip=self.client_ip,
                user_agent=self.user_agent,
            )
        except (yasms_exceptions.YaSmsLimitExceeded,
                yasms_exceptions.YaSmsUidLimitExceeded):
            self.statbox.log(error='sms_limit.exceeded')
            self.log_sms_not_delivered(
                reason='yasms.uid_limit_exceeded',
                global_sms_id=global_sms_id,
            )
            raise exceptions.SendingLimitExceededError()
        except (yasms_exceptions.YaSmsPermanentBlock,
                yasms_exceptions.YaSmsTemporaryBlock):
            self.statbox.log(error='phone.blocked')
            self.log_sms_not_delivered(
                reason='yasms.phone_blocked',
                global_sms_id=global_sms_id,
            )
            raise exceptions.PhoneBlockedError()
        except yasms_exceptions.YaSmsPhoneNumberValueError:
            self.statbox.log(error='sms.isnt_sent')
            self.log_sms_not_delivered(
                reason='yasms.phone_number_value_error',
                global_sms_id=global_sms_id,
            )
            raise exceptions.PhoneInvalidError()
        except yasms_exceptions.YaSmsDeliveryError:
            self.statbox.log(error='sms.isnt_sent')
            self.log_sms_not_delivered(
                reason='yasms.delivery_error',
                global_sms_id=global_sms_id,
            )
            raise bundle_exceptions.YaSmsUnavailableError()
        except yasms_exceptions.YaSmsAccessDenied:
            self.statbox.log(error='sms.isnt_sent')
            self.log_sms_not_delivered(
                reason='yasms.access_denied',
                global_sms_id=global_sms_id,
            )
            raise exceptions.SMSBackendAccessDeniedError()
        except yasms_exceptions.YaSmsError:
            self.log_sms_not_delivered(
                reason='yasms.generic_error',
                global_sms_id=global_sms_id,
            )
            self.statbox.log(error='sms.isnt_sent')
            raise

    def _count_sending(self):
        self.track.phone_confirmation_sms_count.incr()
        if sms_per_ip_for_consumer.exists(self.caller):
            sms_per_ip_for_consumer.get_counter(self.caller).incr(self.client_ip)
        else:
            app_id = get_app_id_from_track(self.track)
            if sms_per_ip_for_app.exists(app_id):
                sms_per_ip_for_app.get_counter(app_id).incr(self.client_ip)
            else:
                sms_per_ip.get_counter(self.client_ip).incr(self.client_ip)
                if self.track.track_type == 'register':
                    sms_per_ip.get_registration_sms_sent_counter(self.client_ip).incr(self.client_ip)

    def _save_timings(self):
        now = time.time()
        if not self.track.phone_confirmation_first_send_at:
            self.track.phone_confirmation_first_send_at = now
        self.track.phone_confirmation_last_send_at = now

    def log_sms_not_delivered(self, reason, global_sms_id):
        log_sms_not_delivered(
            reason=reason,
            number=self.number,
            global_sms_id=global_sms_id,
            caller=self.caller,
            client_ip=str(self.client_ip),
            user_agent=self.user_agent,
        )


class CallingPhoneConfirmator(PhoneConfirmatorInitialStep):
    """
    Класс, обрабатывающий первый шаг подтверждения телефона звонком на номер.

    Проверяет различные лимиты и счётчики, генерирует код, составляет скрипт звонка,
    получает сессию из octopus и сохраняет состояние в трек.
    Основной внешний метод - make_call().
    """
    confirmation_method = CONFIRM_METHOD_BY_CALL

    def __init__(
        self,
        client,
        track,
        env,
        previous_number,
        calls_limit,
        code_format,
        code_length,
        statbox,
        phone_confirmation_language,
        caller,
        account,
        can_handle_captcha=False,
        mask_antifraud_denial=False,
        login_id=None
    ):
        super(CallingPhoneConfirmator, self).__init__(
            client,
            track,
            env,
            previous_number,
            code_format,
            code_length,
            statbox,
            caller,
            account,
            can_handle_captcha=can_handle_captcha,
            mask_antifraud_denial=mask_antifraud_denial,
            login_id=login_id,
        )
        self.calls_limit = calls_limit
        self.phone_confirmation_language = get_octopus_language_suggest(display_language=phone_confirmation_language)

    @cached_property
    def kolmogor(self):
        return get_kolmogor()

    @property
    def calls_count(self):
        return self.track.phone_confirmation_calls_count.get(default=0)

    def _check_state(self):
        """
        Кидает исключение, если телефон из запроса уже был подтверждён данным методом.
        Кроме того проверяет что в треке не установлены флаги превышения лимитов на звонки.
        """
        self._assert_phone_not_confirmed()
        if self.track.phone_confirmation_calls_ip_limit_reached:
            raise exceptions.CallingLimitExceededError()
        self.score_phone()

    def make_call(self, number):
        self.save_number(number)
        # проверяем превышение лимитов и подтверждение до сброса трека
        simulate_call = self._check_state_and_find_if_simulation_is_required(
            af_error_class=exceptions.CallingLimitExceededError,
        )

        self._reset_state_and_counts()

        # Проверяем достижение этого лимита отдельно, чтобы был возможен сценарий
        # смены телефона в пределах одного трека
        if self.track.phone_confirmation_calls_count_limit_reached:
            raise exceptions.CallingLimitExceededError()

        self._check_calls_count_limit()
        self._check_calls_global_counters()

        self._save_code()

        if simulate_call:
            call_session_id = FAKE_OCTOPUS_SESSION_ID_FOR_ANTIFRAUD_DENIAL
        else:
            first_code, second_code = split_code_by_3(self.code)
            call_session_id = self._make_call(first_code, second_code, number.e164)

        self.track.phone_call_session_id = call_session_id

        self._save_timings()
        self._count_calling()
        if not simulate_call:
            self.statbox.log(
                action='call_with_code',
                ip=self.client_ip,
                calls_count=self.calls_count,
                call_session_id=call_session_id,
            )

    def _make_call(self, first_code, second_code, phone_number):
        try:
            if PhoneNumber.in_fakelist(PhoneNumber.parse(phone_number), settings.PHONE_NUMBERS_FAKELIST):
                result = FAKE_OCTOPUS_SESSION_ID_FOR_TEST_PHONES
            else:
                result = self.client.create_session(
                    first_code,
                    second_code,
                    settings.PASSPORT_CALLING_NUMBER,
                    phone_number,
                    self.phone_confirmation_language,
                )
        except BaseOctopusError:
            self.statbox.log(error='call.not_made')
            inc_calls_counter(self.kolmogor, call_counter_key=KOLMOGOR_COUNTER_CALLS_FAILED)
            raise
        else:
            inc_sessions_created(self.kolmogor)
        return result

    def _check_calls_count_limit(self):
        """
        Считает звонки только на трек
        """
        result = self.calls_count >= self.calls_limit
        self.track.phone_confirmation_calls_count_limit_reached = result
        if result:
            self.statbox.log(error='calls_limit.exceeded')
            raise exceptions.CallingLimitExceededError('too many. count=%d' % self.calls_count)

    def _check_calls_global_counters(self):
        """
        ~Так себе~ глобальные счетчики, у которых лимит немного резиновый из-за
        возможности попадания запросов в разные ДЦ
        """
        phone_limit_reached = calls_per_phone.get_counter().hit_limit(self.number.digital)
        self.track.phone_confirmation_calls_count_limit_reached = phone_limit_reached

        if phone_limit_reached:
            self.statbox.log(error='calls_limit.exceeded')
            raise exceptions.CallingLimitExceededError('phone=%s' % self.number.digital)

        ip_limit_reached = calls_per_ip.get_counter(self.client_ip).hit_limit_by_ip(self.client_ip)

        self.track.phone_confirmation_calls_ip_limit_reached = ip_limit_reached

        if ip_limit_reached:
            self.statbox.log(error='calls_limit.exceeded')
            raise exceptions.CallingLimitExceededError('ip=%s' % self.client_ip)

    def _save_code(self):
        self.track.phone_confirmation_code = self.code

    def _reset_state_and_counts(self):
        # Если в трэке тот же номер, что и в запросе, то это значит,
        # что до пользователя не дозвонились, и мы перезваниваем.
        # В этом случае не надо сбрасывать состояние трэка.
        # Если телефон из запроса отличается от сохранённого в трэке, или же метод подтверждения изменился,
        # то трэк надо поресетить.
        if self._did_phone_or_confirm_method_change():
            self.track.phone_confirmation_calls_count_limit_reached = False
            self.track.is_successful_phone_passed = False
            self.track.phone_confirmation_is_confirmed = False
            self.track.phone_confirmation_code = None
            if self.calls_count > 0:
                self.track.phone_confirmation_calls_count.reset()
            if self.confirmations_count > 0:
                self.track.phone_confirmation_confirms_count.reset()

    def _save_timings(self):
        now = time.time()
        if not self.track.phone_confirmation_first_called_at:
            self.track.phone_confirmation_first_called_at = now
        self.track.phone_confirmation_last_called_at = now

    def _count_calling(self):
        self.track.phone_confirmation_calls_count.incr()
        calls_per_ip.get_counter(self.client_ip).incr(self.client_ip)
        calls_per_phone.get_counter().incr(self.number.digital)


class FlashCallingPhoneConfirmator(CallingPhoneConfirmator):
    """
    Класс, обрабатывающий первый шаг подтверждения телефона звонком на номер со сбросом.

    Проверяет различные лимиты и счётчики,
    получает сессию из octopus и сохраняет состояние в трек.
    Основной внешний метод - make_call().
    """
    confirmation_method = CONFIRM_METHOD_BY_FLASH_CALL

    def __init__(
        self,
        client,
        track,
        env,
        previous_number,
        calls_limit,
        statbox,
        caller,
        account,
        can_handle_captcha=False,
        mask_antifraud_denial=False,
        login_id=None,
    ):
        super(FlashCallingPhoneConfirmator, self).__init__(
            account=account,
            caller=caller,
            calls_limit=calls_limit,
            client=client,
            code_format=None,
            code_length=None,
            env=env,
            phone_confirmation_language=None,
            previous_number=previous_number,
            statbox=statbox,
            track=track,
            can_handle_captcha=can_handle_captcha,
            mask_antifraud_denial=mask_antifraud_denial,
            login_id=login_id,
        )
        self.calling_number = None
        self.code_to_number = {phone_number[-4:]: phone_number  for phone_number in settings.FLASH_CALL_NUMBERS}

    def _is_code_valid(self, code):
        return code in self.code_to_number

    def _generate_random_code(self):
        self.calling_number = random.SystemRandom().choice(settings.FLASH_CALL_NUMBERS)
        code = self.calling_number[-4:]
        return code

    def make_call(self, number):
        self.save_number(number)
        # проверяем превышение лимитов и подтверждение до сброса трека
        simulate_flash_call = self._check_state_and_find_if_simulation_is_required(
            af_error_class=exceptions.CallingLimitExceededError,
        )

        self._reset_state_and_counts()

        # Проверяем достижение этого лимита отдельно, чтобы был возможен сценарий
        # смены телефона в пределах одного трека
        if self.track.phone_confirmation_calls_count_limit_reached:
            raise exceptions.CallingLimitExceededError()

        self._check_calls_count_limit()
        self._check_calls_global_counters()

        self._save_code()

        # Это для случая, если код прочитали из трека, а не сгенерировали
        # Тогда по коду вычисляем возможный номер телефона, связанный с этим кодом
        if self.calling_number is None:
            self.calling_number = self.code_to_number[self.code]

        if simulate_flash_call:
            call_session_id = FAKE_OCTOPUS_SESSION_ID_FOR_ANTIFRAUD_DENIAL
        else:
            call_session_id = self._make_call(number.e164)

        self.track.phone_call_session_id = call_session_id

        self._save_timings()
        self._count_calling()

        if not simulate_flash_call:
            self.statbox.log(
                action='flash_call',
                ip=self.client_ip,
                calls_count=self.calls_count,
                call_session_id=call_session_id,
            )

    def _make_call(self, phone_number):
        try:
            if PhoneNumber.in_fakelist(PhoneNumber.parse(phone_number), settings.PHONE_NUMBERS_FAKELIST):
                result = FAKE_OCTOPUS_SESSION_ID_FOR_TEST_PHONES
            else:
                result = self.client.create_flash_call_session(
                    self.calling_number,
                    phone_number,
                )
        except BaseOctopusError:
            self.statbox.log(error='call.not_made')
            inc_calls_counter(self.kolmogor, call_counter_key=KOLMOGOR_COUNTER_CALLS_FAILED)
            raise
        else:
            inc_sessions_created(self.kolmogor)
        return result


class VerifyingPhoneConfirmator(PhoneConfirmator):
    """
    Класс, обрабатывающий второй шаг подтверждения телефона.

    Проверяет лимиты, сверяет отправленный код с переданным из-вне и сохраняет
    состояние в трек. Основной внешний метод - confirm_code().
    """
    def __init__(self, track, statbox, antifraud_logger, confirmations_limit, client=None, account=None,
                 accept_fake_code_for_test_numbers=False):
        super(VerifyingPhoneConfirmator, self).__init__(track, statbox)
        self.confirmations_limit = confirmations_limit
        self.number = PhoneNumber.parse(self.track.phone_confirmation_phone_number)
        self.client = client
        self.antifraud_logger = antifraud_logger
        self.accept_fake_code_for_test_numbers = accept_fake_code_for_test_numbers
        if account:
            secure_phone = account.phones.secure.number.e164 if account.phones.secure else None
            self.antifraud_logger.bind_context(is_secure_phone=self.number.e164 == secure_phone)
        self.antifraud_logger.bind_context(user_phone=self.number.e164)

    def confirm_code(self, code):
        self._check_state()
        self._check_confirmations_count_limit()

        # тайминги и каунтеры надо увеличивать в любом случае, если проводили верификацию
        # нам не важно хороший был код или плохой
        self._save_timings()
        self._count_confirmation()
        self._verify_code(code)

    def _check_confirmations_count_limit(self):
        result = self.confirmations_count is not None and self.confirmations_count >= self.confirmations_limit
        self.track.phone_confirmation_confirms_count_limit_reached = result
        if result:
            self.statbox.log(error='confirmations_limit.exceeded')
            raise exceptions.ConfirmationsLimitExceededError()

    def _verify_code(self, code):
        normalized_code = normalize_code(code)
        is_code_ok = False
        if normalized_code == normalize_code(self.track.phone_confirmation_code):
            is_code_ok = True
        elif (
            self.accept_fake_code_for_test_numbers and
            self.track.phone_confirmation_phone_number in settings.TEST_PHONE_NUMBERS_ACCEPTING_FAKE_CODE and
            normalized_code == settings.FAKE_CODE
        ):
            is_code_ok = True

        if is_code_ok:
            self._enter_code_statbox(is_good=True)
            self.antifraud_logger.log()
            self.track.phone_confirmation_is_confirmed = True
        else:
            self._enter_code_statbox(is_good=False)
            self.track.phone_confirmation_is_confirmed = False
            raise bundle_exceptions.CodeInvalidError()

    def _save_timings(self):
        now = time.time()
        if not self.track.phone_confirmation_first_checked:
            self.track.phone_confirmation_first_checked = now
        self.track.phone_confirmation_last_checked = now

    def _count_confirmation(self):
        self.track.phone_confirmation_confirms_count.incr()

    def _enter_code_statbox(self, is_good):
        raise NotImplementedError()     # pragma: no cover


class VerifyingPhoneSmsConfirmator(VerifyingPhoneConfirmator):

    def _check_state(self):
        if not self.track.phone_confirmation_sms_count.get(default=0):
            raise exceptions.SmsNotSentError()

    def _enter_code_statbox(self, is_good):
        values = {
            'action': 'enter_code',
            # last_sms_sent_time может быть None
            'time_passed': self.last_sms_sent_time and int(time.time() - self.last_sms_sent_time),
            'good': is_good,
        }
        if not is_good:
            values.update({'error': 'code.invalid'})

        self.statbox.log(**values)


class VerifyingPhoneCallConfirmator(VerifyingPhoneConfirmator):
    def _check_state(self):
        if not self.track.phone_confirmation_calls_count.get(default=0):
            raise exceptions.CallNotMadeError()
        if not (
            self.track.phone_call_session_id and
            self.track.phone_confirmation_code
        ):
            raise InvalidTrackStateError()

    def _enter_code_statbox(self, is_good):
        action = 'enter_code'

        values = {
            'action': action,
            # last_called_time может быть None
            'time_passed': self.last_called_time and int(time.time() - self.last_called_time),
            'good': is_good,
        }
        if not is_good:
            values.update({'error': 'code.invalid'})

        self.statbox.log(**values)


def load_phones(client, account):
    """Загружает только подтверждённые телефоны пользователя"""
    items = client.userphones(account)
    valid_phones = [phone for phone in items if phone['valid'] == 'valid']
    # allow_impossible, потому что мы верим телефонам, которые записаны в истории. Они все проходили валидацию
    phones = build_phones(valid_phones, allow_impossible=True)
    return phones


def find_account_by_number(client, number):
    all_alt_numbers = [number] + get_alt_phone_numbers_of_phone_number(number)
    for alt_number in all_alt_numbers:
        response = client.userinfo(
            login=alt_number.digital,
            emails=True,
            find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
        )
        try:
            return Account().parse(response)
        except UnknownUid:
            pass


def get_call_status(octopus_api, track):
    # ветка кода для авто-тестов, чтобы не ходить в октопус и проверять наши сценарии
    if track.phone_call_session_id == FAKE_OCTOPUS_SESSION_ID_FOR_TEST_PHONES:
        if track.phone_confirmation_phone_number is None:
            raise InvalidTrackStateError()
        if not PhoneNumber.in_fakelist(track.phone_confirmation_phone_number, settings.PHONE_NUMBERS_FAKELIST):
            raise InvalidTrackStateError()
        return settings.CALL_STATUS_SUCCESS

    response = octopus_api.get_session_log(track.phone_call_session_id)
    return response['status']
