# -*- coding: utf-8 -*-

from itertools import groupby
import json
import logging
from operator import attrgetter

from passport.backend.api.views.bundle import exceptions as bundle_exceptions
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
    BundlePhoneMixin,
)
from passport.backend.api.views.bundle.mixins.phone import YASMS_EXCEPTIONS_MAPPING
from passport.backend.api.views.bundle.phone.helpers import normalize_code
from passport.backend.api.yasms.utils import build_send_confirmation_code
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.phones.phones import (
    ConfirmationLimitExceeded,
    PhoneConfirmedAlready,
    ReplaceSecurePhoneWithBoundPhoneOperation,
    ReplaceSecurePhoneWithNonboundPhoneOperation,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.yasms.phonenumber_alias import PhoneAliasManager
from passport.backend.utils.time import (
    datetime_to_unixtime,
    unixtime_to_datetime,
)

from .. import exceptions


BASIC_GRANT = 'phone_bundle.base'
BY_UID_GRANT = 'phone_bundle.by_uid'

REPLACE_SECURE_PHONE_MODE = 'phone_secure_replace'

log = logging.getLogger('passport.api.view.bundle.phone.manage')


class BasePhoneManageBundleView(
    BundlePhoneMixin,
    BaseBundleView,
    BundleAccountGetterMixin,
    BundleAccountResponseRendererMixin,
):
    mode = None
    step = None
    required_grants = [BASIC_GRANT]
    track_type = 'authorize'
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE]

    event_action = None

    def __init__(self):
        self.phone = None
        self.login_id = None
        super(BasePhoneManageBundleView, self).__init__()

    def get_account(self, **kwargs):
        kwargs.setdefault('check_disabled_on_deletion', True)
        kwargs.setdefault('need_phones', True)
        # Нужны почтовые адреса, чтобы высылать уведомления
        kwargs.setdefault('emails', True)

        uid = self.form_values.get('uid')
        if uid is None:
            kwargs.setdefault('required_scope', self.token_required_scopes)
            blackbox_response = self.get_account_from_session_or_oauth_token(**kwargs)
        else:
            self.check_grant(BY_UID_GRANT)
            blackbox_response = self.get_account_by_uid(uid, **kwargs)

        return blackbox_response

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode=self.mode,
            step=self.step,
            consumer=self.consumer,
            ip=self.client_ip,
            user_agent=self.user_agent,
        )

    def get_track(self, create=True):
        """
        Создаем новый трек, либо читаем существующий.
        """
        if self.track_id:
            self.read_track()
        else:
            if create:
                self.create_track(self.track_type)

        if self.track_id:
            self.statbox.bind_context(track_id=self.track_id)

    def is_account_type_allowed(self):
        return (
            self.account.is_lite or
            self.account.is_neophonish or
            self.account.is_normal or
            self.account.is_pdd or
            self.account.is_social or
            self.account.is_federal
        )

    @staticmethod
    def detect_statbox_mode(logical_op):
        """
        Для некоторых ручек statbox_mode вычисляется на лету на основе операции.
        """
        if type(logical_op) in {ReplaceSecurePhoneWithNonboundPhoneOperation, ReplaceSecurePhoneWithBoundPhoneOperation}:
            return REPLACE_SECURE_PHONE_MODE
        else:
            return logical_op.name

    def _send_code(self, phone, display_language):
        logical_op = phone.get_logical_operation(self.statbox)
        confirmation_info = logical_op.get_confirmation_info(phone.id)
        send_code = build_send_confirmation_code(
            phone.number,
            confirmation_info,
            self.account,
            self.request.env,
            self.yasms,
            self.consumer,
            self.statbox,
            display_language,
            restricter=self.build_phone_confirmation_restricter(
                confirmation_info=confirmation_info,
                phone_number=phone.number,
                login_id=self.login_id,
            ),
        )
        try:
            with self.statbox.make_context(action=self.mode + '.' + self.step):
                send_code()
        except Exception as e:
            self.statbox.dump_stashes()
            if e.__class__ not in YASMS_EXCEPTIONS_MAPPING:
                raise
            exc_cls = YASMS_EXCEPTIONS_MAPPING[e.__class__]
            raise exc_cls(e)
        logical_op.set_confirmation_info(phone.id, confirmation_info)
        self.response_values.update(
            code_length=len(normalize_code(confirmation_info.code_value)),
        )

    def _check_code(self, phone, code):
        logical_op = phone.get_logical_operation(self.statbox)
        if logical_op.is_expired:
            raise exceptions.OperationExpiredError()

        confirmation_info = logical_op.get_confirmation_info(phone.id)
        if confirmation_info.code_send_count == 0:
            raise exceptions.SmsNotSentError()
        try:
            is_phone_confirmed, _ = logical_op.check_code(phone.id, code)
        except PhoneConfirmedAlready:
            raise bundle_exceptions.ActionNotRequiredError()
        except ConfirmationLimitExceeded:
            self.statbox.dump_stashes()
            raise exceptions.ConfirmationsLimitExceededError()
        return is_phone_confirmed

    def fill_response_with_account(self, with_phones=True, **kwargs):
        """
        Пишем в response_values['account'] данные о текущих телефонах и операциях на аккаунте.
        """
        super(BasePhoneManageBundleView, self).fill_response_with_account(
            with_phones=with_phones,
            **kwargs
        )

    def _check_account_state(self):
        """
        Проверим, что ручка может работать с данным аккаунтом.
        Проверки не относятся к текущей операции и телефону, только к состоянию аккаунта.
        """
        pass

    def _assert_phone_not_bound(self):
        """
        Проверим, что номер телефона не привязан.
        """
        if self.phone.bound:
            raise exceptions.PhoneBoundError()

    def _assert_phone_bound(self):
        """
        Проверим, что номер телефона есть на аккаунте и привязан.
        """
        if not self.phone.bound:
            raise exceptions.PhoneNotFoundError()

    def _assert_account_has_no_secure_phone(self):
        """
        Убедимся, что на аккаунте нет защищенного телефона.
        """
        if self.account.phones.secure:
            raise exceptions.SecurePhoneAlreadyExistsError()

    def _assert_phone_has_no_operation(self):
        """
        Убедимся, что на текущем телефоне нет активной операции.
        """
        if self.phone.operation:
            raise exceptions.PhoneOperationExistsError()

    def _load_phone_by_number(self, number, create_if_not_exists):
        """
        По переданному номеру получим объект телефона.
        Если на аккаунте нет такого привязанного номера, то создадим новый или
        выбросим исключение в зависимости от `create_if_not_exists`.
        """
        self.phone = self.account.phones.by_number(number)

        if not self.phone:
            if create_if_not_exists:
                self.phone = self.account.phones.create(number=number)
            else:
                raise exceptions.PhoneNotFoundError()

    def _load_phone_by_id(self, phone_id):
        """
        По переданному phone_id получим объект телефона.
        Если на аккаунте нет такого привязанного номера - выбросим исключение.
        """
        self.phone = self.account.phones.by_id(phone_id, assert_exists=False)

        if not self.phone:
            raise exceptions.PhoneNotFoundError()

    def _load_phone_by_operation_id(self, operation_id):
        """
        По переданному operation_id получим объект телефона.
        Если на аккаунте нет такого номера - выбросим исключение.
        """
        self.phone = self.account.phones.by_operation_id(operation_id)

        if self.phone is None:
            raise exceptions.OperationNotFoundError()

    def _load_alias_phone(self):
        if not self.account.phonenumber_alias:
            raise exceptions.PhoneAliasNotFoundError()
        self.phone = self.account.phones.by_number(self.account.phonenumber_alias.number)
        if not self.phone:
            # Состояние, когда в центральной БД был ЛЦА, но в шарде не было
            # основного телефона, возникала в тестовом окружении, вероятно
            # из-за отставания реплик.
            raise exceptions.PhoneAliasNotFoundError()

    @cached_property
    def _phone_alias_manager(self):
        return PhoneAliasManager(
            self.consumer,
            environment=self.request.env,
            statbox=self.statbox,
            need_update_tx=False,
        )

    @property
    def _display_language(self):
        form_language = self.form_values.get('display_language')
        if self.track and self.track.display_language:
            track_language = self.track.display_language
        else:
            track_language = None
        return form_language or track_language

    @property
    def event_action(self):
        return self.mode + '_' + self.step

    def _is_phone_alias_allowed(self):
        return (
            self._phone_alias_manager.is_alias_allowed(self.account, enable_search=True) or
            self._phone_alias_manager.is_alias_allowed(self.account, enable_search=False)
        )


class Confirmation(object):
    def __init__(self, logical_operation_id, phone_id, phone_confirmed=None,
                 password_verified=None):
        self._logical_operation_id = logical_operation_id
        self._phone_id = phone_id
        self.phone_confirmed = phone_confirmed
        self.password_verified = password_verified

    @property
    def logical_operation_id(self):
        return self._logical_operation_id

    @property
    def phone_id(self):
        return self._phone_id

    @classmethod
    def from_json(cls, document):
        confirmation = json.loads(document)
        if confirmation['phone_confirmed'] is not None:
            confirmation['phone_confirmed'] = unixtime_to_datetime(confirmation['phone_confirmed'])
        if confirmation['password_verified'] is not None:
            confirmation['password_verified'] = unixtime_to_datetime(confirmation['password_verified'])
        return Confirmation(
            logical_operation_id=confirmation['logical_op_id'],
            phone_id=confirmation['phone_id'],
            phone_confirmed=confirmation['phone_confirmed'],
            password_verified=confirmation['password_verified'],
        )

    def to_json(self):
        _dict = self.to_dict()
        for key in ['phone_confirmed', 'password_verified']:
            if _dict[key]:
                _dict[key] = datetime_to_unixtime(_dict[key])
        return json.dumps(_dict)

    def to_dict(self):
        return {
            'logical_op_id': self.logical_operation_id,
            'phone_id': self.phone_id,
            'phone_confirmed': self.phone_confirmed,
            'password_verified': self.password_verified,
        }

    def __eq__(self, other):
        if not isinstance(other, Confirmation):
            return False
        return self.to_dict() == other.to_dict()

    def __ne__(self, other):
        return not (self == other)

    def __hash__(self):
        return hash((self.logical_operation_id, self.phone_id))


def get_last_confirmations_from_track(logical_op, track):
    if track is None:
        return {}, None

    documents = track.phone_operation_confirmations.get()

    all_confirmations = []
    for document in documents:
        confirmation = Confirmation.from_json(document)
        if logical_op.id == confirmation.logical_operation_id:
            all_confirmations.append(confirmation)

    phone_id_to_confirmations = groupby(
        sorted(all_confirmations, key=attrgetter('phone_id')),
        key=attrgetter('phone_id'),
    )
    last_confirmations = {}
    for phone_id, confirmations in phone_id_to_confirmations:
        phone_confirmations = [c.phone_confirmed for c in confirmations
                               if c.phone_confirmed]
        if phone_confirmations:
            last = max(phone_confirmations)
        else:
            last = None
        last_confirmations[phone_id] = last

    password_verifications = [c.password_verified for c in all_confirmations
                              if c.password_verified]
    if password_verifications:
        password_verified = max(password_verifications)
    else:
        password_verified = False

    return last_confirmations, password_verified
