# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.views.bundle import exceptions as bundle_exceptions
from passport.backend.api.views.bundle.constants import (
    BIND_PHONE_OAUTH_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.mixins import BundleAccountPropertiesMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.phone import exceptions
from passport.backend.api.yasms.utils import does_binding_allow_washing
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import DummyLogger
from passport.backend.core.models.phones.phones import (
    AliasifySecureOperation,
    DealiasifySecureOperation,
    OperationInapplicable,
    RemoveSecureOperation,
    ReplaceSecurePhoneWithBoundPhoneOperation,
    ReplaceSecurePhoneWithNonboundPhoneOperation,
    SecureBindOperation,
    SecurifyOperation,
    SimpleBindOperation,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.yasms.notifications import notify_about_phone_changes
from passport.backend.core.yasms.phonenumber_alias import Aliasification
from passport.backend.core.yasms.unbinding import unbind_old_phone

from .. import (
    forms,
    helpers,
)
from ..base import (
    BasePhoneManageBundleView,
    get_last_confirmations_from_track,
    REPLACE_SECURE_PHONE_MODE,
)


logger = logging.getLogger('passport.backend.api.bundle.phone.manage')


class BaseCommitView(BasePhoneManageBundleView, BundlePushMixin, KolmogorMixin):
    """
    Базовый класс всех ручек commit.
    * Валидирует форму.
    * Получает аккаунт.
    * Проверяет условия, при которых ручка может/должна работать.
    """

    _expected_operation_types = []
    step = 'commit'
    require_track = True

    def __init__(self):
        self.phone = None
        super(BaseCommitView, self).__init__()

    def process_request(self):
        self.process_basic_form()

        self.get_account()

        operation_id = self.form_values['operation_id']
        # NOTE: На последнем этапе уже нет смысла создавать трек.
        self.get_track(create=False)

        self.statbox.bind_context(operation_id=operation_id)
        self.statbox.log(action='submitted')

        self.statbox.bind_context(uid=self.account.uid)
        self._check_account_state()

        self._load_phone_by_operation_id(operation_id)
        self.statbox.bind_context(number=self.phone.number.masked_format_for_statbox)

        logical_op = self.phone.get_logical_operation(self.statbox)
        self._check_operation_state(logical_op)

        datetime_ = datetime.now()

        with notify_about_phone_changes(
            account=self.account,
            yasms_builder=self.yasms,
            statbox=self.statbox,
            consumer=self.consumer,
            language=self._display_language,
            client_ip=self.client_ip,
            user_agent=self.user_agent,
            view=self,
        ), UPDATE(
            self.account,
            self.request.env,
            {'action': self.event_action, 'consumer': self.consumer},
            datetime_=datetime_,
            initiator_uid=self.account.uid,
        ):
            self._apply_confirmations(logical_op)

            try:
                next_logical_op, changes = logical_op.apply(need_authenticated_user=self.account.have_password)
            except OperationInapplicable as e:
                if (
                    logical_op.is_ready_for_quarantine(need_authenticated_user=self.account.have_password) and
                    not logical_op.in_quarantine
                ):
                    logical_op.start_quarantine()
                if logical_op.in_quarantine:
                    self._fill_response()
                    self.statbox.log(action='completed')
                    return
                raise exceptions.PhoneOperationInapplicable.from_exception(e)

            if next_logical_op is not None:
                # Next logical op не равен None, только когда исполняется
                # операция замены, но этот класс не испольузется для замены.
                raise NotImplementedError()

            if changes.unbound_numbers:
                self._dealiasify_phone(changes.unbound_numbers)

            if changes.bound_numbers:
                self._wash_account(self.account, self.phone, logical_op.flags, changes.bound_numbers)

            should_aliasify = logical_op.flags.aliasify
            if should_aliasify:
                # Перед тем как создать алиас, нужно убедиться, что
                # логическая операция исполнилась полностью, а не перешла в
                # следующий шаг (новую логическую операцию).
                self._aliasify_phone(datetime_)

        if should_aliasify:
            self._aliasification.notify()

        self.statbox.dump_stashes()

        self._unbind_old_phones(changes.bound_numbers, datetime_)

        self._fill_response()
        self.statbox.log(action='completed')

    def _wash_account(self, account, phone, binding_flags, bound_numbers):
        """
        По возможности обеляет учётную запись.
        """
        bindings_history = self.yasms_api.phone_bindings_history(bound_numbers)

        number_allows_washing = []
        for number in bound_numbers:
            number_allows_washing.append(
                does_binding_allow_washing(
                    account,
                    phone.number.e164,
                    phone.bound,
                    binding_flags.should_ignore_binding_limit,
                    bindings_history,
                ),
            )

        if any(number_allows_washing):
            # Обеляем учётную запись
            account.karma.prefix = settings.KARMA_PREFIX_WASHED

    def _check_operation_state(self, logical_op):
        """
        Проверим, что операция находится в правильном состоянии:
        тип операции подходит, проверен пароль, код смс и прочее.
        Если что-то не так - бросаем исключение.
        """
        if type(logical_op) not in self._expected_operation_types:
            raise exceptions.InvalidOperationStateError()
        if logical_op.is_expired:
            raise exceptions.OperationExpiredError()

        if logical_op.flags.aliasify:
            if not self._is_phone_alias_allowed():
                raise bundle_exceptions.AccountInvalidTypeError()
            if self.account.phonenumber_alias.alias:
                raise exceptions.PhoneAliasExistError()

    def _dealiasify_phone(self, unbound_numbers):
        if not (self.account.phonenumber_alias.number and
                self.account.phonenumber_alias.number.e164 in unbound_numbers):
            return

        self._phone_alias_manager.delete_alias(
            self.account,
            self._phone_alias_manager.ALIAS_DELETE_REASON_OFF,
            self._display_language,
        )

    @cached_property
    def _aliasification(self):
        enable_search = self._phone_alias_manager.is_alias_allowed(self.account, enable_search=True)
        return Aliasification(
            account=self.account,
            phone_number=self.phone.number,
            consumer=self.consumer,
            blackbox=self.blackbox,
            statbox=DummyLogger(),
            language=self._display_language,
            enable_search=enable_search,
        )

    def _aliasify_phone(self, datetime_):
        prev_owner = self._aliasification.get_owner()

        if prev_owner is not None:
            with UPDATE(
                prev_owner,
                self.request.env,
                {'action': 'phone_alias_delete', 'consumer': self.consumer},
                datetime_=datetime_,
                initiator_uid=self.account.uid,
            ):
                self._aliasification.take_away()

        self._aliasification.give_out()

    def _fill_response(self):
        self.response_values.update({
            'phone': helpers.get_phone_info(self.phone),
            'track_id': self.track_id,
        })
        self.fill_response_with_account()

    def _unbind_old_phones(self, numbers, event_timestamp):
        for number in numbers:
            phone = self.account.phones.by_number(number)
            unbind_old_phone(
                subject_phone=phone,
                blackbox_builder=self.blackbox,
                statbox=self.statbox,
                consumer=self.consumer,
                event_timestamp=event_timestamp,
                environment=self.request.env,
            )

    def _get_phone_confirmation_time(self, phone):
        logical_op = phone.get_logical_operation(self.statbox)
        confirmations, _ = get_last_confirmations_from_track(logical_op, self.track)
        return confirmations.get(phone.id)

    def _get_password_verification_time(self, logical_op):
        _, password_verified = get_last_confirmations_from_track(logical_op, self.track)
        return password_verified

    def _apply_confirmations(self, logical_op):
        phone_confirmations, password_verified = get_last_confirmations_from_track(logical_op, self.track)
        for phone_id, phone_confirmed in phone_confirmations.items():
            confirmation_info = logical_op.get_confirmation_info(phone_id)
            if phone_confirmed and not confirmation_info.code_confirmed:
                logical_op.confirm_phone(
                    phone_id,
                    code=None,
                    timestamp=phone_confirmed,
                    should_check_code=False,
                )
        if password_verified and not logical_op.password_verified:
            logical_op.password_verified = password_verified


class BindPhoneCommit(BaseCommitView):
    basic_form = forms.BindPhoneCommitForm
    _expected_operation_types = [SimpleBindOperation]
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    mode = SimpleBindOperation.name

    def _check_operation_state(self, logical_op):
        super(BindPhoneCommit, self)._check_operation_state(logical_op)
        self._assert_phone_not_bound()


class BindSecurePhoneCommit(BaseCommitView):
    basic_form = forms.OperationIdForm
    _expected_operation_types = [SecureBindOperation]

    mode = SecureBindOperation.name

    def _check_account_state(self):
        super(BindSecurePhoneCommit, self)._check_account_state()
        self._assert_account_has_no_secure_phone()

    def _check_operation_state(self, logical_op):
        super(BindSecurePhoneCommit, self)._check_operation_state(logical_op)
        self._assert_phone_not_bound()


class SecurifyCommitPhone(BaseCommitView):
    basic_form = forms.OperationIdForm
    _expected_operation_types = [SecurifyOperation]

    mode = SecurifyOperation.name

    def _check_account_state(self):
        super(SecurifyCommitPhone, self)._check_account_state()
        self._assert_account_has_no_secure_phone()

    def _check_operation_state(self, logical_op):
        super(SecurifyCommitPhone, self)._check_operation_state(logical_op)
        self._assert_phone_bound()


class RemoveSecurePhoneCommit(BaseCommitView, BundleAccountPropertiesMixin):
    basic_form = forms.OperationIdForm
    _expected_operation_types = [RemoveSecureOperation]

    mode = RemoveSecureOperation.name

    def _check_account_state(self):
        super(RemoveSecurePhoneCommit, self)._check_account_state()
        if self.account.totp_secret.is_set:
            raise bundle_exceptions.Account2FAEnabledError()
        self.check_sms_2fa_disabled()

    def _fill_response(self):
        if self.account.phones.has_id(self.phone.id):
            self.response_values['phone'] = helpers.get_phone_info(self.phone)
        else:
            self.response_values['phone'] = None
        self.response_values['track_id'] = self.track_id
        self.fill_response_with_account()


class ReplaceSecurePhoneCommit(BaseCommitView):
    basic_form = forms.OperationIdForm
    mode = REPLACE_SECURE_PHONE_MODE
    _expected_operation_types = [
        ReplaceSecurePhoneWithBoundPhoneOperation,
        ReplaceSecurePhoneWithNonboundPhoneOperation,
    ]

    def process_request(self):
        self.process_basic_form()

        self.get_account()

        operation_id = self.form_values['operation_id']
        self.get_track(create=False)

        self.statbox.bind_context(operation_id=operation_id)
        self.statbox.log(action='submitted')

        self.statbox.bind_context(uid=self.account.uid)

        phone = self.account.phones.by_operation_id(operation_id)
        if phone is None:
            raise exceptions.OperationNotFoundError()
        operation = phone.get_logical_operation(self.statbox)

        self._check_operation_state(operation)

        secure_phone = self.account.phones.secure
        simple_phone = self.account.phones.by_id(secure_phone.operation.phone_id2)

        executor = self.build_replace_secure_phone(
            account=self.account,
            phone_number=simple_phone.number,
            yasms_builder=self.yasms,
            yasms=self.yasms_api,
            language=self._display_language,
            notification_language=self._display_language,
            is_secure_phone_confirmed=self._get_phone_confirmation_time(secure_phone),
            is_simple_phone_confirmed=self._get_phone_confirmation_time(simple_phone),
            is_password_verified=self._get_password_verification_time(operation),
        )

        try:
            executor.submit()
        except OperationInapplicable as e:
            raise exceptions.PhoneOperationInapplicable.from_exception(e)

        with UPDATE(
            self.account,
            self.request.env,
            {'action': self.event_action, 'consumer': self.consumer},
        ):
            executor.commit()

        try:
            executor.after_commit()
        except OperationInapplicable as e:
            # Падаем, если выполнили привязку простого, но не смогли выполнить всю
            # замену или начать карантин.
            raise exceptions.PhoneOperationInapplicable.from_exception(e)

        self._fill_response()
        self.statbox.log(action='completed')

    def _fill_response(self):
        self.response_values.update({
            'secure_phone': helpers.get_phone_info(self.account.phones.secure),
            'track_id': self.track_id,
        })
        self.fill_response_with_account()


class AliasifySecurePhoneCommit(BaseCommitView):
    basic_form = forms.OperationIdForm
    _expected_operation_types = [AliasifySecureOperation]

    mode = AliasifySecureOperation.name

    def _check_account_state(self):
        super(AliasifySecurePhoneCommit, self)._check_account_state()
        if self.account.phones.secure is None:
            raise bundle_exceptions.SecurePhoneNotFoundError()


class DealiasifySecurePhoneCommit(BaseCommitView):
    basic_form = forms.OperationIdForm
    _expected_operation_types = [DealiasifySecureOperation]

    mode = DealiasifySecureOperation.name

    def process_request(self):
        self.process_basic_form()

        operation_id = self.form_values['operation_id']
        self.statbox.bind_context(operation_id=operation_id)

        self.get_track(create=False)

        self.statbox.log(action='submitted')

        self.get_account()
        self.statbox.bind_context(uid=self.account.uid)
        self._check_account_state()

        self._load_phone_by_operation_id(operation_id)
        self.statbox.bind_context(number=self.phone.number.masked_format_for_statbox)

        logical_op = self.phone.get_logical_operation(self.statbox)
        self._check_operation_state(logical_op)

        with UPDATE(
            self.account,
            self.request.env,
            {'action': self.event_action, 'consumer': self.consumer},
        ):
            self._apply_confirmations(logical_op)
            try:
                logical_op.apply(need_authenticated_user=self.account.have_password)
            except OperationInapplicable as e:
                raise exceptions.PhoneOperationInapplicable.from_exception(e)

            self._phone_alias_manager.delete_alias(
                self.account,
                self._phone_alias_manager.ALIAS_DELETE_REASON_OFF,
                self._display_language,
            )

        self.statbox.dump_stashes(operation_id=logical_op.id)
        self._fill_response()
        self.statbox.log(action='completed')

    def _fill_response(self):
        self.response_values['phone'] = helpers.get_phone_info(self.phone)
        self.response_values['track_id'] = self.track_id
        self.fill_response_with_account()

    def _check_account_state(self):
        super(DealiasifySecurePhoneCommit, self)._check_account_state()
        if not self.account.phonenumber_alias.alias:
            raise exceptions.PhoneAliasNotFoundError()
