# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.views.bundle import exceptions as bundle_exceptions
from passport.backend.api.views.bundle.constants import (
    BIND_PHONE_OAUTH_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.exceptions import OAuthTokenValidationError
from passport.backend.api.views.bundle.mixins import (
    BundleAssertCaptchaMixin,
    BundleVerifyPasswordMixin,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.phone import exceptions
from passport.backend.api.views.bundle.phone.manage import forms
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import DummyLogger
from passport.backend.core.models.phones.phones import (
    MarkOperation,
    SimpleBindOperation,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.yasms.notifications import notify_about_phone_changes
from passport.backend.core.yasms.phonenumber_alias import is_phonenumber_alias_as_email_allowed
from passport.backend.utils.common import generate_random_code

from ..base import (
    BasePhoneManageBundleView,
    Confirmation,
)


logger = logging.getLogger('passport.backend.api.bundle.phone.manage')


class BaseOtherControllerView(BasePhoneManageBundleView, BundlePushMixin, KolmogorMixin):
    with_track = True
    with_operation = True
    with_phone = True

    def process_request(self):
        self.process_basic_form()
        self.prepare()
        self.statbox.log(action='submitted')

        blackbox_response = self.get_account()
        self.statbox.bind_context(uid=self.account.uid)
        self._check_account_state()

        if self.with_operation:
            self._load_phone_by_operation_id(self.form_values['operation_id'])

            self.mode = self.detect_statbox_mode(self.phone.get_logical_operation(self.statbox))
            self.statbox.bind_context(mode=self.mode)
        elif self.with_phone:
            self._load_phone_by_id(self.form_values['phone_id'])
        else:
            self._load_custom_phone()

        self.statbox.bind_context(number=self.phone.number.masked_format_for_statbox)

        self._check_token_scope_matches_phone_operation(blackbox_response)

        with notify_about_phone_changes(
            account=self.account,
            yasms_builder=self.yasms,
            statbox=self.statbox,
            consumer=self.consumer,
            language=self._display_language,
            client_ip=self.client_ip,
            user_agent=self.user_agent,
            view=self,
        ):
            self._process_request()

        self.fill_response()
        self.statbox.log(action='completed')

    def _fill_track(self):
        pass

    def prepare(self):
        if self.with_operation:
            self.statbox.bind_context(operation_id=self.form_values['operation_id'])

        if self.with_track:
            self.get_track()
            self.check_track()

    def fill_response(self):
        if self.with_track:
            self._fill_track()
            self.response_values['track_id'] = self.track_id
        self._fill_response()

    def _process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def check_track(self):
        """Проверим состояние трека."""

    def _fill_response(self):
        pass

    def _check_token_scope_matches_phone_operation(self, blackbox_response):
        blackbox_oauth_response = blackbox_response.get('oauth', {})
        token_scope = blackbox_oauth_response.get('scope', [])
        if blackbox_oauth_response and X_TOKEN_OAUTH_SCOPE not in token_scope:
            assert self.phone
            logical_op = self.phone.get_logical_operation(self.statbox)
            if not (BIND_PHONE_OAUTH_SCOPE in token_scope and type(logical_op) is SimpleBindOperation):
                raise OAuthTokenValidationError()


class SendSmsCode(BaseOtherControllerView):
    basic_form = forms.SendCodeForm
    step = 'resend_code'
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    def _process_request(self):
        logical_op = self.phone.get_logical_operation(self.statbox)
        if logical_op.is_expired:
            raise exceptions.OperationExpiredError()

        if not self.phone.operation.does_user_admit_phone:
            raise bundle_exceptions.ActionNotRequiredError()

        events = {'action': 'confirmation_code_send', 'consumer': self.consumer}
        with UPDATE(self.account, self.request.env, events, disable_history_db=True):

            confirmation_info = logical_op.get_confirmation_info(self.phone.id)
            if not confirmation_info.code_value:
                confirmation_info.code_value = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)
                logical_op.set_confirmation_info(self.phone.id, confirmation_info)

            self._send_code(self.phone, self.form_values['display_language'])

        self.statbox.dump_stashes()

    def _fill_track(self):
        with self.track_transaction.rollback_on_error():
            self.track.display_language = self.form_values['display_language']


class CheckSmsCode(BaseOtherControllerView):
    basic_form = forms.CheckCodeForm
    step = 'check_code'
    require_track = True
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    def _process_request(self):
        events = {'action': 'confirmation_code_check', 'consumer': self.consumer}
        with UPDATE(self.account, self.request.env, events):
            code_entered_correctly = self._check_code(self.phone, self.form_values['code'])

            with self.track_transaction.rollback_on_error():
                if code_entered_correctly:
                    logical_op = self.phone.get_logical_operation(self.statbox)
                    confirmation = Confirmation(
                        logical_op.id,
                        self.phone.id,
                        phone_confirmed=datetime.now(),
                    )
                    self.track.phone_operation_confirmations.append(confirmation.to_json())

        self.statbox.dump_stashes()

        if not code_entered_correctly:
            raise bundle_exceptions.CodeInvalidError()


class CheckPassword(BaseOtherControllerView, BundleVerifyPasswordMixin, BundleAssertCaptchaMixin):
    basic_form = forms.CheckPasswordForm
    step = 'check_password'
    require_track = True

    def _process_request(self):
        events = {'action': 'password_check', 'consumer': self.consumer}

        logical_op = self.phone.get_logical_operation(self.statbox)
        if logical_op.is_expired:
            raise exceptions.OperationExpiredError()

        if logical_op.password_verified:
            raise bundle_exceptions.ActionNotRequiredError()

        with UPDATE(self.account, self.request.env, events, disable_history_db=True):
            with self.track_transaction.commit_on_error():
                self.verify_password(
                    self.account.uid,
                    self.form_values['current_password'],
                    ignore_statbox_log=True,
                    useragent=self.user_agent,
                    referer=self.referer,
                    yandexuid=self.cookies.get('yandexuid'),
                    dbfields=[],  # Не выбирать из ЧЯ доп информацию о пользователе
                    attributes=[],  # и атрибутах
                )

                logical_op = self.phone.get_logical_operation(self.statbox)
                confirmation = Confirmation(
                    logical_op.id,
                    self.phone.id,
                    password_verified=datetime.now(),
                )
                self.track.phone_operation_confirmations.append(confirmation.to_json())

                self.statbox.log(action='password_checked')

        self.statbox.dump_stashes()

    def check_track(self):
        self.check_track_for_captcha(log_fail_to_statbox=False)


class CancelOperation(BaseOtherControllerView):
    basic_form = forms.OperationIdForm
    step = 'cancel_operation'
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    def _process_request(self):
        logical_op = self.phone.get_logical_operation(self.statbox)
        if logical_op.is_expired:
            raise exceptions.OperationExpiredError()

        events = {'action': 'cancel_operation', 'consumer': self.consumer}
        with UPDATE(self.account, self.request.env, events):
            logical_op.cancel()
        self.statbox.dump_stashes()


class GetState(BaseOtherControllerView):
    """
    Ручка получения состояния аккаунта: все телефоны и операции на аккаунте
    в удобном для фронтенда виде.

    NOTE: не читаем и не создаем трек.
    """
    step = 'get_state'
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    def process_request(self):
        self.get_account()
        self._check_account_state()
        self.fill_response_with_account()


class RemoveSimplePhone(BaseOtherControllerView):
    basic_form = forms.RemoveSimplePhoneForm
    step = 'remove_simple_phone'

    with_track = False
    with_operation = False

    def _process_request(self):
        if self.phone == self.account.phones.secure:
            raise exceptions.InvalidPhoneType()
        self._assert_phone_has_no_operation()

        events = {'action': 'simple_phone_remove', 'consumer': self.consumer}

        with UPDATE(self.account, self.request.env, events):
            # Чтобы из-за гонки не удалить защищённый номер, создадим операцию.
            mark_op = MarkOperation.create(
                self.account.phones,
                self.phone.id,
                generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
                DummyLogger(),
            )

        with UPDATE(self.account, self.request.env, events):
            mark_op.apply()
            self.account.phones.remove(self.phone)

    def _fill_response(self):
        self.fill_response_with_account()


class ProlongValidPhone(BaseOtherControllerView):
    """
    Ручка подтверждения любого телефона без отправки смс.
    Требование подтверждения номера сейчас появляется
    с периодичностью в полгода.
    """
    basic_form = forms.ProlongValidPhoneForm
    step = 'prolong_valid'

    with_track = False
    with_operation = False

    def _process_request(self):
        events = {'action': 'prolong_valid', 'consumer': self.consumer}

        self._assert_phone_bound()
        with UPDATE(self.account, self.request.env, events):
            self.phone.admitted = datetime.now()

    def _fill_response(self):
        self.fill_response_with_account()


class SetDefaultPhone(BaseOtherControllerView):
    """
    Ручка для явного указания привязанного телефона
    номером по умолчанию для приема нотификаций.
    """
    basic_form = forms.SetDefaultPhoneForm
    step = 'set_default'

    with_track = False
    with_operation = False

    def _process_request(self):
        events = {'action': 'set_default', 'consumer': self.consumer}

        self._assert_phone_bound()
        with UPDATE(self.account, self.request.env, events):
            self.account.phones.default = self.phone

    def _fill_response(self):
        self.fill_response_with_account()


class DisablePhonenumberAliasAsEmail(BaseOtherControllerView):
    basic_form = forms.DisablePhonenumberAliasAsEmailForm
    step = 'disable_phonenumber_alias_as_email'

    with_track = False
    with_operation = False
    with_phone = False

    def _process_request(self):
        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'disable_phonenumber_alias_as_email', 'consumer': self.consumer},
        ):
            self._phone_alias_manager.disable_as_email(
                self.account,
                self.form_values['display_language'],
            )

    def _load_custom_phone(self):
        self._load_alias_phone()

    def _fill_response(self):
        self.fill_response_with_account()


class EnablePhonenumberAliasAsEmail(BaseOtherControllerView):
    basic_form = forms.EnablePhonenumberAliasAsEmailForm
    step = 'enable_phonenumber_alias_as_email'

    with_track = False
    with_operation = False
    with_phone = False

    def _process_request(self):
        if not is_phonenumber_alias_as_email_allowed(self.account):
            return

        with UPDATE(
            self.account,
            self.request.env,
            {'action': 'enable_phonenumber_alias_as_email', 'consumer': self.consumer},
        ):
            self._phone_alias_manager.enable_as_email(
                self.account,
                self.form_values['display_language'],
            )

    def _load_custom_phone(self):
        self._load_alias_phone()

    def _fill_response(self):
        self.fill_response_with_account()


'''
ПРОЦЕССЫ

* Привязка телефона
  - BindPhoneSubmit
  - SendSms
  - CheckSmsCode
  - BindPhoneCommit

* Привязка защищенного телефона
  - BindSecurePhoneSubmit
  - SendSms
  - CheckSmsCode
  - CheckPassword
  - BindSecurePhoneCommit

* Сделать обычный номер защищенным
  - SecurifySubmitPhone
  - SendSms
  - CheckSmsCode
  - CheckPassword
  - SecurifyCommitPhone

* Удалить обычный телефон.
  - RemovePhone

* Удалить защищенный телефон.
  - RemoveSecurePhoneSubmit
  <Одно из двух>
  - CheckPassword
  <или>
  - SendSms
  - CheckSmsCode
  </Одно из двух>
  - RemoveSecurePhoneCommit

* Заменить защищенный телефон (сделать обычный защищенным).
  - ChangeSecurePhoneSubmit
  - SendSms: для обычного телефона
  - CheckSmsCode
  <Одно из двух>
  - SendSms: для защищенного телефона
  - CheckSmsCode
  <или>
  - CheckPassword
  </Одно из двух>
  - ChangeSecurePhoneCommit

* Заменить защищенный телефон (добавить новый телефон и сделать защищенным).
  - ПРОЦЕСС "Привязка телефона"
  - ПРОЦЕСС "Заменить защищенный телефон (сделать обычный защищенным)"
  При этом не нужно отправлять смс на обычный телефон, так как он был привязан недавно.
'''
