# -*- coding: utf-8 -*-
from passport.backend.api.common.phone_karma import (
    get_phone_karma,
    PhoneKarma,
)
from passport.backend.api.views.bundle import exceptions as bundle_exceptions
from passport.backend.api.views.bundle.constants import (
    BIND_PHONE_OAUTH_SCOPE,
    X_TOKEN_OAUTH_SCOPE,
)
from passport.backend.api.views.bundle.mixins import BundleAccountPropertiesMixin
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.phone import YASMS_EXCEPTIONS_MAPPING
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.api.views.bundle.phone import exceptions
from passport.backend.api.yasms import exceptions as yasms_exceptions
from passport.backend.core.conf import settings
from passport.backend.core.models.phones.phones import (
    AliasifySecureOperation,
    DealiasifySecureOperation,
    RemoveSecureOperation,
    SecureBindOperation,
    SecurifyOperation,
    SimpleBindOperation,
    SingleSecureOperationError,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.yasms.notifications import notify_about_phone_changes
from passport.backend.utils.common import generate_random_code

from .. import (
    forms,
    helpers,
)
from ..base import (
    BasePhoneManageBundleView,
    REPLACE_SECURE_PHONE_MODE,
)


class BaseSubmitView(BasePhoneManageBundleView, BundlePushMixin, KolmogorMixin):
    """
    Базовый класс всех ручек submit.
    * Валидирует форму.
    * Получает аккаунт.
    * Проверяет условия, при которых ручка может/должна работать.
    * Создает телефонную операцию.
    """

    step = 'submit'

    def __init__(self):
        self.phone = None
        super(BaseSubmitView, self).__init__()

    def _check_phone_karma(self, number):
        if not settings.USE_PHONE_KARMA:
            return
        phone_karma = get_phone_karma(number, safe=True)
        self.statbox.log(
            action='check_phone_karma',
            number=number.masked_format_for_statbox,
            karma=phone_karma,
        )
        if not settings.PHONE_KARMA_DRY_RUN and phone_karma == PhoneKarma.black:
            raise bundle_exceptions.PhoneCompromisedError()

    def process_request(self):
        self.process_basic_form()

        self.get_track()

        if 'number' in self.form_values:
            self.statbox.bind_context(number=self.form_values['number'].masked_format_for_statbox)

        if 'phone_id' in self.form_values:
            self.statbox.bind_context(phone_id=self.form_values['phone_id'])

        self.statbox.log(action='submitted')

        self.get_account()
        self.statbox.bind_context(uid=self.account.uid)
        self._check_account_state()

        # Создаем телефон и операцию.
        events = {'action': self.event_action, 'consumer': self.consumer}
        with notify_about_phone_changes(
            account=self.account,
            yasms_builder=self.yasms,
            statbox=self.statbox,
            consumer=self.consumer,
            language=self.form_values['display_language'],
            client_ip=self.client_ip,
            user_agent=self.user_agent,
            view=self,
        ), UPDATE(
            self.account,
            self.request.env,
            events,
        ):
            self._load_and_check_target_phone()
            self.statbox.bind_context(number=self.phone.number.masked_format_for_statbox)

            operation = self._create_operation()

            if self._should_send_code():
                self._send_code(self.phone, self.form_values['display_language'])

        # До завершения транзакции у нас не было operation_id, поэтому запись
        # в statbox об отправке СМС происходит только сейчас.
        self.statbox.dump_stashes(operation_id=operation.id)

        self._fill_track()

        self.response_values['track_id'] = self.track_id
        self.response_values['phone'] = helpers.get_phone_info(self.phone, self.form_values.get('number'))
        self.fill_response_with_account()

    def _fill_track(self):
        with self.track_transaction.rollback_on_error():
            self.track.display_language = self.form_values['display_language']

    def _load_and_check_target_phone(self):
        """
        Получаем телефон из аккаунта, либо создаем новый телефон, если нужно.
        Проверим, что телефон можно использовать в текущем процессе.
        """
        raise NotImplementedError()  # pragma: no cover

    def _create_operation(self):
        """
        Создаем операцию на текущем телефоне.
        """
        raise NotImplementedError()  # pragma: no cover

    def _should_send_code(self):
        return True


class BindPhoneSubmit(BaseSubmitView):
    basic_form = forms.PhoneNumberSubmitForm
    mode = SimpleBindOperation.name
    token_required_scopes = [X_TOKEN_OAUTH_SCOPE, BIND_PHONE_OAUTH_SCOPE]

    def _load_and_check_target_phone(self):
        self._load_phone_by_number(self.form_values['number'], create_if_not_exists=True)
        self._assert_phone_not_bound()
        self._assert_phone_has_no_operation()

    def _create_operation(self):
        return SimpleBindOperation.create(
            phone_manager=self.account.phones,
            phone_id=self.phone.id,
            code=generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
            should_ignore_binding_limit=False,
            statbox=self.statbox,
        )


class BindSecurePhoneSubmit(BaseSubmitView):
    basic_form = forms.BindSecurePhoneSubmitForm
    mode = SecureBindOperation.name

    def _check_account_state(self):
        super(BindSecurePhoneSubmit, self)._check_account_state()
        self._assert_account_has_no_secure_phone()

        if self.form_values['is_alias']:
            if self.account.phonenumber_alias.alias:
                raise exceptions.PhoneAliasExistError()

    def _load_and_check_target_phone(self):
        self._load_phone_by_number(self.form_values['number'], create_if_not_exists=True)
        self._assert_phone_not_bound()
        self._assert_phone_has_no_operation()
        self._check_phone_karma(self.phone.number)

    def _create_operation(self):
        try:
            return SecureBindOperation.create(
                phone_manager=self.account.phones,
                phone_id=self.phone.id,
                code=generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
                should_ignore_binding_limit=False,
                statbox=self.statbox,
                aliasify=self.form_values['is_alias'],
            )
        except SingleSecureOperationError:
            raise exceptions.SecureOperationExistsError()

    def is_account_type_allowed(self):
        is_allowed = super(BindSecurePhoneSubmit, self).is_account_type_allowed()
        if self.form_values['is_alias']:
            is_allowed = is_allowed and self._is_phone_alias_allowed()
        return is_allowed


class SecurifySubmitPhone(BaseSubmitView):
    basic_form = forms.SecurifyPhoneSubmitForm
    mode = SecurifyOperation.name

    def _check_account_state(self):
        super(SecurifySubmitPhone, self)._check_account_state()
        self._assert_account_has_no_secure_phone()

    def _load_and_check_target_phone(self):
        self._load_phone_by_id(self.form_values['phone_id'])
        self._assert_phone_has_no_operation()
        self._assert_phone_bound()
        self._check_phone_karma(self.phone.number)

    def _create_operation(self):
        try:
            return SecurifyOperation.create(
                phone_manager=self.account.phones,
                phone_id=self.phone.id,
                code=generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
                statbox=self.statbox,
            )
        except SingleSecureOperationError:
            raise exceptions.SecureOperationExistsError()


class ReplaceSecurePhoneSubmit(BaseSubmitView):
    basic_form = forms.ReplaceSecurePhoneSubmitForm
    mode = REPLACE_SECURE_PHONE_MODE

    def process_request(self):
        self.process_basic_form()
        self.get_track()

        self.statbox.log(action='submitted')

        self.get_account()
        self.statbox.bind_context(uid=self.account.uid)
        self._check_account_state()

        if self.form_values['number']:
            phone_number = self.form_values['number']
        else:
            self._load_phone_by_id(self.form_values['phone_id'])
            phone_number = self.phone.number

        secure_phone = self.account.phones.secure
        if not secure_phone:
            raise bundle_exceptions.SecurePhoneNotFoundError()

        if secure_phone.operation:
            raise exceptions.PhoneOperationExistsError()

        self._check_phone_karma(phone_number)

        executor = self.build_replace_secure_phone(
            account=self.account,
            does_user_admit_secure_number=self.form_values['does_user_admit_secure_number'],
            is_long_lived=True,
            language=self.form_values['display_language'],
            notification_language=self.form_values['display_language'],
            phone_number=phone_number,
            yasms=self.yasms_api,
            yasms_builder=self.yasms,
        )

        try:
            executor.submit()
        except yasms_exceptions.YaSmsConflictedOperationExists:
            raise exceptions.PhoneOperationExistsError()
        except yasms_exceptions.YaSmsActionNotRequiredError:
            raise bundle_exceptions.ActionNotRequiredError()
        except yasms_exceptions.YaSmsAccountInvalidTypeError:
            raise bundle_exceptions.AccountInvalidTypeError()
        except Exception as e:
            if e.__class__ not in YASMS_EXCEPTIONS_MAPPING:
                raise
            exc_cls = YASMS_EXCEPTIONS_MAPPING[e.__class__]
            raise exc_cls(e)

        with self.track_transaction.rollback_on_error():
            self.track.display_language = self.form_values['display_language']
        self.response_values['track_id'] = self.track_id

        secure_phone = self.account.phones.secure
        self.response_values['secure_phone'] = helpers.get_phone_info(secure_phone)

        simple_phone = self.account.phones.by_number(phone_number)
        self.response_values['simple_phone'] = helpers.get_phone_info(simple_phone, self.form_values.get('number'))
        self.fill_response_with_account()


class RemoveSecurePhoneSubmit(BaseSubmitView, BundleAccountPropertiesMixin):
    basic_form = forms.RemoveSecurePhoneSubmitForm
    mode = RemoveSecureOperation.name

    def _check_account_state(self):
        super(RemoveSecurePhoneSubmit, self)._check_account_state()
        if self.account.phones.secure is None:
            raise bundle_exceptions.SecurePhoneNotFoundError()
        if self.account.totp_secret.is_set:
            raise bundle_exceptions.Account2FAEnabledError()
        self.check_sms_2fa_disabled()

    def _load_and_check_target_phone(self):
        self._load_phone_by_id(self.account.phones.secure.id)
        self._assert_phone_has_no_operation()

    def _create_operation(self):
        if self.form_values['does_user_admit_secure_number']:
            code = generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH)
        else:
            code = None
        return RemoveSecureOperation.create(
            phone_manager=self.account.phones,
            phone_id=self.phone.id,
            code=code,
            statbox=self.statbox,
        )

    def _should_send_code(self):
        return self.form_values['does_user_admit_secure_number']

    def is_account_type_allowed(self):
        if self.account.is_neophonish:
            return False
        return super(RemoveSecurePhoneSubmit, self).is_account_type_allowed()


class AliasifySecurePhoneSubmit(BaseSubmitView):
    basic_form = forms.AliasifySecurePhoneSubmitForm
    mode = AliasifySecureOperation.name

    def _check_account_state(self):
        super(AliasifySecurePhoneSubmit, self)._check_account_state()
        if self.account.phones.secure is None:
            raise bundle_exceptions.SecurePhoneNotFoundError()

        if self.account.phonenumber_alias.alias:
            raise exceptions.PhoneAliasExistError()

    def _load_and_check_target_phone(self):
        self._load_phone_by_id(self.account.phones.secure.id)
        self._assert_phone_has_no_operation()

    def _create_operation(self):
        return AliasifySecureOperation.create(
            phone_manager=self.account.phones,
            phone_id=self.phone.id,
            code=generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
            statbox=self.statbox,
        )

    def is_account_type_allowed(self):
        return (
            super(AliasifySecurePhoneSubmit, self).is_account_type_allowed() and
            self._is_phone_alias_allowed()
        )


class DealiasifySecurePhoneSubmit(BaseSubmitView):
    basic_form = forms.DealiasifySecurePhoneSubmitForm
    mode = DealiasifySecureOperation.name

    def _check_account_state(self):
        super(DealiasifySecurePhoneSubmit, self)._check_account_state()
        if not self.account.phonenumber_alias.alias:
            raise exceptions.PhoneAliasNotFoundError()
        if not self.account.phones.secure:
            raise bundle_exceptions.SecurePhoneNotFoundError()

    def _load_and_check_target_phone(self):
        self._load_phone_by_id(self.account.phones.secure.id)
        self._assert_phone_has_no_operation()

    def _create_operation(self):
        return DealiasifySecureOperation.create(
            phone_manager=self.account.phones,
            phone_id=self.phone.id,
            code=generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
            statbox=self.statbox,
        )

    def _should_send_code(self):
        return False
